/***********************************************************************/
/*                                                                     */
/*                      ADOBE CONFIDENTIAL                             */
/*                   _ _ _ _ _ _ _ _ _ _ _ _ _                         */
/*                                                                     */
/*  Copyright 2016 Adobe Systems Incorporated                          */
/*  All Rights Reserved.                                               */
/*                                                                     */
/* NOTICE:  All information contained herein is, and remains           */
/* the property of Adobe Systems Incorporated and its suppliers,       */
/* if any.  The intellectual and technical concepts contained          */
/* herein are proprietary to Adobe Systems Incorporated and its        */
/* suppliers and are protected by all applicable intellectual property */
/* laws, including trade secret and copyright laws.                    */
/* Dissemination of this information or reproduction of this material  */
/* is strictly forbidden unless prior written permission is obtained   */
/* from Adobe Systems Incorporated.                                    */
/*                                                                     */
/***********************************************************************/

function LocalContentStorage(/*[String]*/ inBasePath)
{
	var basePath = inBasePath;
	var stringTables = {};
	
	// initialize
	(function()
	{
		if (isValidProperty(basePath))
		{
			if (basePath.lastIndexOf('/') != basePath.length-1)
			{
				basePath += '/';
			}
		}
	})();
	
	//////////////////////////////////////////////////////////////////////////////
	//
	// Return absolute file path of cached content file
	//
	this.getURL = function(/*[string]*/ inWorkflowID, /*[string]*/ inContentID)
	{
		var job = new Job();
		var path = null;

		if (getContentExists(inWorkflowID, inContentID, true))
		{
			path = getContentPath(inWorkflowID, inContentID, true);
		}
		else if (getContentExists(inWorkflowID, inContentID, false))
		{
			path = getContentPath(inWorkflowID, inContentID, false);
		}

		if (isValidProperty(path))
		{
			job.setResult(path);
			dbgLogContent("LocalContentStorage.getURL", inWorkflowID, inContentID, path);
		}
		else
		{
			job.setError("LocalContentStorage.getURL '" + inWorkflowID + "' / '" + inContentID + "' NOT exist");
			dbglogError(job.getError());
		}

		return job;
	}
	
	//////////////////////////////////////////////////////////////////////////////
	//
	// Return content as it is (e.g. raw text is returned as raw text, html is returned as html, etc.)
	//
	this.getRaw = function(/*[string]*/ inWorkflowID, /*[string]*/ inContentID)
	{
		var job = new Job();
		var ret = null;
		var contentPath = null;
		
		if (getContentExists(inWorkflowID, inContentID, true))
		{
			contentPath = getContentPath(inWorkflowID, inContentID, true);
		}
		else if (getContentExists(inWorkflowID, inContentID, false))
		{
			contentPath = getContentPath(inWorkflowID, inContentID, false);
		}
		
		if (isValidProperty(contentPath))
		{
			var result = cep.fs.readFile(contentPath);
			
			if (isValidProperty(result) && result.err == cep.fs.NO_ERROR)
			{
				ret = result.data;
			}
		}

		if (isValidProperty(ret))
		{
			job.setResult(ret);
			dbgLogContent("LocalContentStorage.getRaw", inWorkflowID, inContentID, contentPath);
		}
		else
		{
			job.setError("LocalContentStorage.getRaw '" + inWorkflowID + "' / '" + inContentID + "' NOT exist");
			dbglogError(job.getError());
		}

		return job;
	}
	
	//////////////////////////////////////////////////////////////////////////////
	//
	// Return content as localized string
	//
	this.getString = function(/*[string]*/ inWorkflowID, /*[string]*/ inContentID)
	{
		var job = new Job();
		var ret = null;
		var table = getStringTable(inWorkflowID);
		
		if (isValidProperty(table))
		{
			ret = table.getString(inContentID);
		}

		if (isValidProperty(ret))
		{
			job.setResult(ret);
			dbgLogContent("LocalContentStorage.getString", inWorkflowID, inContentID, ret);
		}
		else
		{
			job.setError("LocalContentStorage.getString '" + inWorkflowID + "' / '" + inContentID + "' NOT exist");
			dbglogError(job.getError());
		}

		return job;
	}
	
	//////////////////////////////////////////////////////////////////////////////
	//
	// Set a stringtable
	//
	this.setStringtable = function(/*[String]*/ inWorkflowID, /*[Stringtable]*/ inStringtable)
	{
		stringTables[inWorkflowID] = inStringtable;
	}
	
	// private ///////////////////////////////////////////////////////////////////

	//////////////////////////////////////////////////////////////////////////////
	//
	// Return true if content exists
	// (if param inLocalized is undefined, then check both locations)
	//
	function getContentExists(/*[string]*/ inWorkflowID, /*[string]*/ inContentID, /*[Boolean]*/ inLocalized)
	{
		// string table entry?
		//
		var table = getStringTable(inWorkflowID);
		var ret = (isValidProperty(table) && table.exists(inContentID));
		
		if (!ret)
		{
			// distinct between localized/fallback location
			var paramValid = isValidProperty(inLocalized);
			var dontcare = !paramValid;
			var localized = paramValid && inLocalized;

			if (dontcare || localized)
			{
				// localized file
				//
				var path = getContentPath(inWorkflowID, inContentID, true);
				ret = fileExists(path);
			}		

			if (!ret && (dontcare || !localized))
			{
				// fallback location
				//
				var path = getContentPath(inWorkflowID, inContentID, false);
				ret = fileExists(path);
			}		
		}
		
		return ret;
	}
	
	//////////////////////////////////////////////////////////////////////////////
	//
	// Return string table for workflow ID
	// (create if not created yet)
	//
	function getStringTable(/*[string]*/ inWorkflowID)
	{
		var tableID = inWorkflowID;
		var wfPath = inWorkflowID + '/';
		
		if (!isValidProperty(inWorkflowID))
		{
			tableID = 'root';
			wfPath = '';
		}
		
		if (!isValidProperty(stringTables[tableID]) && isValidProperty(basePath))
		{
			// localized path
			var path = basePath + wfPath + getLocale() + '/' + StringTable.FILENAME;
			
			if (!fileExists(path))
			{
				// fallback path
				path = basePath + wfPath + StringTable.FILENAME;
			}
		
			if (fileExists(path))
			{
				stringTables[tableID] = new StringTable(path);
			}
		}
		
		return stringTables[tableID];
	}

	//////////////////////////////////////////////////////////////////////////////
	//
	// Create path to content file 
	// (if inContentID is a filename)
	//
	function getContentPath(/*[string]*/ inWorkflowID, /*[string]*/ inContentID, /*[Boolean]*/ inLocalized)
	{
		var ret = '';
		
		if (isValidProperty(basePath))
		{
			var localized = isValidProperty(inLocalized) && inLocalized;
			var wfPath = isValidProperty(inWorkflowID) ? (inWorkflowID + '/') : '';
			ret = basePath + wfPath + (localized ? (getLocale() + '/') : '') + inContentID;
		}
		
		return ret;
	}
}
