/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.license;

import java.time.Clock;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.atomic.AtomicReference;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.elasticsearch.ElasticsearchException;
import org.elasticsearch.Version;
import org.elasticsearch.action.ActionListener;
import org.elasticsearch.cluster.AckedClusterStateUpdateTask;
import org.elasticsearch.cluster.ClusterChangedEvent;
import org.elasticsearch.cluster.ClusterState;
import org.elasticsearch.cluster.ClusterStateListener;
import org.elasticsearch.cluster.metadata.Metadata;
import org.elasticsearch.cluster.node.DiscoveryNode;
import org.elasticsearch.cluster.service.ClusterService;
import org.elasticsearch.common.component.AbstractLifecycleComponent;
import org.elasticsearch.common.component.Lifecycle;
import org.elasticsearch.common.logging.LoggerMessageFormat;
import org.elasticsearch.common.settings.Setting;
import org.elasticsearch.common.settings.Settings;
import org.elasticsearch.common.time.DateFormatter;
import org.elasticsearch.common.util.set.Sets;
import org.elasticsearch.core.TimeValue;
import org.elasticsearch.discovery.DiscoveryModule;
import org.elasticsearch.env.Environment;
import org.elasticsearch.gateway.GatewayService;
import org.elasticsearch.license.ExpirationCallback;
import org.elasticsearch.license.License;
import org.elasticsearch.license.LicenseUtils;
import org.elasticsearch.license.LicenseVerifier;
import org.elasticsearch.license.LicensesMetadata;
import org.elasticsearch.license.OperationModeFileWatcher;
import org.elasticsearch.license.PostStartBasicRequest;
import org.elasticsearch.license.PostStartBasicResponse;
import org.elasticsearch.license.PostStartTrialRequest;
import org.elasticsearch.license.PostStartTrialResponse;
import org.elasticsearch.license.PutLicenseRequest;
import org.elasticsearch.license.SelfGeneratedLicense;
import org.elasticsearch.license.StartBasicClusterTask;
import org.elasticsearch.license.StartTrialClusterTask;
import org.elasticsearch.license.StartupSelfGeneratedLicenseTask;
import org.elasticsearch.license.XPackLicenseState;
import org.elasticsearch.protocol.xpack.XPackInfoResponse;
import org.elasticsearch.protocol.xpack.license.DeleteLicenseRequest;
import org.elasticsearch.protocol.xpack.license.LicensesStatus;
import org.elasticsearch.protocol.xpack.license.PutLicenseResponse;
import org.elasticsearch.threadpool.ThreadPool;
import org.elasticsearch.watcher.ResourceWatcherService;
import org.elasticsearch.xpack.core.XPackPlugin;
import org.elasticsearch.xpack.core.XPackSettings;
import org.elasticsearch.xpack.core.scheduler.SchedulerEngine;

public class LicenseService
extends AbstractLifecycleComponent
implements ClusterStateListener,
SchedulerEngine.Listener {
    private static final Logger logger = LogManager.getLogger(LicenseService.class);
    public static final Setting<License.LicenseType> SELF_GENERATED_LICENSE_TYPE = new Setting<License.LicenseType>("xpack.license.self_generated.type", s -> License.LicenseType.BASIC.getTypeName(), s -> {
        License.LicenseType type = License.LicenseType.parse(s);
        return SelfGeneratedLicense.validateSelfGeneratedType(type);
    }, Setting.Property.NodeScope);
    static final List<License.LicenseType> ALLOWABLE_UPLOAD_TYPES = LicenseService.getAllowableUploadTypes();
    public static final Setting<List<License.LicenseType>> ALLOWED_LICENSE_TYPES_SETTING = Setting.listSetting("xpack.license.upload.types", Collections.unmodifiableList(ALLOWABLE_UPLOAD_TYPES.stream().map(License.LicenseType::getTypeName).collect(Collectors.toList())), License.LicenseType::parse, LicenseService::validateUploadTypesSetting, Setting.Property.NodeScope);
    static final TimeValue NON_BASIC_SELF_GENERATED_LICENSE_DURATION = TimeValue.timeValueHours(720L);
    static final Set<License.LicenseType> VALID_TRIAL_TYPES = Collections.unmodifiableSet(Sets.newHashSet(License.LicenseType.GOLD, License.LicenseType.PLATINUM, License.LicenseType.ENTERPRISE, License.LicenseType.TRIAL));
    static final TimeValue LICENSE_EXPIRATION_WARNING_PERIOD = LicenseService.days(7);
    public static final long BASIC_SELF_GENERATED_LICENSE_EXPIRATION_MILLIS = XPackInfoResponse.BASIC_SELF_GENERATED_LICENSE_EXPIRATION_MILLIS;
    private final Settings settings;
    private final ClusterService clusterService;
    private final XPackLicenseState licenseState;
    private final AtomicReference<License> currentLicense = new AtomicReference();
    private SchedulerEngine scheduler;
    private final Clock clock;
    private final OperationModeFileWatcher operationModeFileWatcher;
    private List<ExpirationCallback> expirationCallbacks = new ArrayList<ExpirationCallback>();
    private final List<License.LicenseType> allowedLicenseTypes;
    static final int SELF_GENERATED_LICENSE_MAX_NODES = 1000;
    static final int SELF_GENERATED_LICENSE_MAX_RESOURCE_UNITS = 1000;
    public static final String LICENSE_JOB = "licenseJob";
    public static final DateFormatter DATE_FORMATTER = DateFormatter.forPattern("EEEE, MMMM dd, yyyy");
    private static final String ACKNOWLEDGEMENT_HEADER = "This license update requires acknowledgement. To acknowledge the license, please read the following messages and update the license again, this time with the \"acknowledge=true\" parameter:";

    public LicenseService(Settings settings, ThreadPool threadPool, ClusterService clusterService, Clock clock, Environment env, ResourceWatcherService resourceWatcherService, XPackLicenseState licenseState) {
        this.settings = settings;
        this.clusterService = clusterService;
        this.clock = clock;
        this.scheduler = new SchedulerEngine(settings, clock);
        this.licenseState = licenseState;
        this.allowedLicenseTypes = ALLOWED_LICENSE_TYPES_SETTING.get(settings);
        this.operationModeFileWatcher = new OperationModeFileWatcher(resourceWatcherService, XPackPlugin.resolveConfigFile(env, "license_mode"), logger, () -> this.updateLicenseState(this.getLicensesMetadata()));
        this.scheduler.register(this);
        this.populateExpirationCallbacks();
        threadPool.scheduleWithFixedDelay(licenseState::cleanupUsageTracking, TimeValue.timeValueHours(1L), "generic");
    }

    private void logExpirationWarning(long expirationMillis, boolean expired) {
        logger.warn("{}", (Object)LicenseService.buildExpirationMessage(expirationMillis, expired));
    }

    static CharSequence buildExpirationMessage(long expirationMillis, boolean expired) {
        String expiredMsg = expired ? "expired" : "will expire";
        String general = LoggerMessageFormat.format(null, "License [{}] on [{}].\n# If you have a new license, please update it. Otherwise, please reach out to\n# your support contact.\n# ", expiredMsg, DATE_FORMATTER.formatMillis(expirationMillis));
        if (expired) {
            general = general.toUpperCase(Locale.ROOT);
        }
        StringBuilder builder = new StringBuilder(general);
        builder.append(System.lineSeparator());
        if (expired) {
            builder.append("# COMMERCIAL PLUGINS OPERATING WITH REDUCED FUNCTIONALITY");
        } else {
            builder.append("# Commercial plugins operate with reduced functionality on license expiration:");
        }
        XPackLicenseState.EXPIRATION_MESSAGES.forEach((feature, messages) -> {
            if (((String[])messages).length > 0) {
                builder.append(System.lineSeparator());
                builder.append("# - ");
                builder.append((String)feature);
                for (String message : messages) {
                    builder.append(System.lineSeparator());
                    builder.append("#  - ");
                    builder.append(message);
                }
            }
        });
        return builder;
    }

    private void populateExpirationCallbacks() {
        this.expirationCallbacks.add(new ExpirationCallback.Pre(LicenseService.days(7), LicenseService.days(25), LicenseService.days(1)){

            @Override
            public void on(License license) {
                LicenseService.this.logExpirationWarning(license.expiryDate(), false);
            }
        });
        this.expirationCallbacks.add(new ExpirationCallback.Post(LicenseService.days(0), null, TimeValue.timeValueMinutes(10L)){

            @Override
            public void on(License license) {
                LicenseService.this.logExpirationWarning(license.expiryDate(), true);
            }
        });
    }

    public void registerLicense(PutLicenseRequest request, ActionListener<PutLicenseResponse> listener) {
        License.LicenseType licenseType;
        final License newLicense = request.license();
        long now = this.clock.millis();
        if (!LicenseVerifier.verifyLicense(newLicense) || newLicense.issueDate() > now || newLicense.startDate() > now) {
            listener.onResponse(new PutLicenseResponse(true, LicensesStatus.INVALID));
            return;
        }
        try {
            licenseType = License.LicenseType.resolve(newLicense);
        }
        catch (Exception e) {
            listener.onFailure(e);
            return;
        }
        if (licenseType == License.LicenseType.BASIC) {
            listener.onFailure(new IllegalArgumentException("Registering basic licenses is not allowed."));
        } else if (!this.isAllowedLicenseType(licenseType)) {
            listener.onFailure(new IllegalArgumentException("Registering [" + licenseType.getTypeName() + "] licenses is not allowed on this cluster"));
        } else if (newLicense.expiryDate() < now) {
            listener.onResponse(new PutLicenseResponse(true, LicensesStatus.EXPIRED));
        } else {
            Map<String, String[]> acknowledgeMessages;
            License currentLicense;
            if (!request.acknowledged() && (currentLicense = this.getLicense()) != null && !(acknowledgeMessages = LicenseService.getAckMessages(newLicense, currentLicense)).isEmpty()) {
                listener.onResponse(new PutLicenseResponse(false, LicensesStatus.VALID, ACKNOWLEDGEMENT_HEADER, acknowledgeMessages));
                return;
            }
            if (XPackSettings.SECURITY_ENABLED.get(this.settings).booleanValue()) {
                if (XPackLicenseState.isTransportTlsRequired(newLicense, this.settings) && !XPackSettings.TRANSPORT_SSL_ENABLED.get(this.settings).booleanValue() && LicenseService.isProductionMode(this.settings, this.clusterService.localNode())) {
                    throw new IllegalStateException("Cannot install a [" + (Object)((Object)newLicense.operationMode()) + "] license unless TLS is configured or security is disabled");
                }
                if (XPackSettings.FIPS_MODE_ENABLED.get(this.settings).booleanValue() && !XPackLicenseState.isFipsAllowedForOperationMode(newLicense.operationMode())) {
                    throw new IllegalStateException("Cannot install a [" + (Object)((Object)newLicense.operationMode()) + "] license unless FIPS mode is disabled");
                }
            }
            this.clusterService.submitStateUpdateTask("register license [" + newLicense.uid() + "]", new AckedClusterStateUpdateTask(request, listener){

                @Override
                protected PutLicenseResponse newResponse(boolean acknowledged) {
                    return new PutLicenseResponse(acknowledged, LicensesStatus.VALID);
                }

                @Override
                public ClusterState execute(ClusterState currentState) throws Exception {
                    XPackPlugin.checkReadyForXPackCustomMetadata(currentState);
                    Version oldestNodeVersion = currentState.nodes().getSmallestNonClientNodeVersion();
                    if (!LicenseService.licenseIsCompatible(newLicense, oldestNodeVersion)) {
                        throw new IllegalStateException("The provided license is not compatible with node version [" + oldestNodeVersion + "]");
                    }
                    Metadata currentMetadata = currentState.metadata();
                    LicensesMetadata licensesMetadata = (LicensesMetadata)currentMetadata.custom("licenses");
                    Version trialVersion = null;
                    if (licensesMetadata != null) {
                        trialVersion = licensesMetadata.getMostRecentTrialVersion();
                    }
                    Metadata.Builder mdBuilder = Metadata.builder(currentMetadata);
                    mdBuilder.putCustom("licenses", new LicensesMetadata(newLicense, trialVersion));
                    return ClusterState.builder(currentState).metadata(mdBuilder).build();
                }
            });
        }
    }

    private static boolean licenseIsCompatible(License license, Version version) {
        int maxVersion = LicenseUtils.getMaxLicenseVersion(version);
        return license.version() <= maxVersion;
    }

    private boolean isAllowedLicenseType(License.LicenseType type) {
        logger.debug("Checking license [{}] against allowed license types: {}", (Object)type, (Object)this.allowedLicenseTypes);
        return this.allowedLicenseTypes.contains((Object)type);
    }

    public static Map<String, String[]> getAckMessages(License newLicense, License currentLicense) {
        HashMap<String, String[]> acknowledgeMessages = new HashMap<String, String[]>();
        if (!License.isAutoGeneratedLicense(currentLicense.signature()) && currentLicense.issueDate() > newLicense.issueDate()) {
            acknowledgeMessages.put("license", new String[]{"The new license is older than the currently installed license. Are you sure you want to override the current license?"});
        }
        XPackLicenseState.ACKNOWLEDGMENT_MESSAGES.forEach((feature, ackMessages) -> {
            String[] messages = (String[])ackMessages.apply(currentLicense.operationMode(), newLicense.operationMode());
            if (messages.length > 0) {
                acknowledgeMessages.put((String)feature, messages);
            }
        });
        return acknowledgeMessages;
    }

    private static TimeValue days(int days) {
        return TimeValue.timeValueHours(days * 24);
    }

    @Override
    public void triggered(SchedulerEngine.Event event) {
        LicensesMetadata licensesMetadata = this.getLicensesMetadata();
        if (licensesMetadata != null) {
            License license = licensesMetadata.getLicense();
            if (event.getJobName().equals(LICENSE_JOB)) {
                this.updateLicenseState(license, licensesMetadata.getMostRecentTrialVersion());
            } else if (event.getJobName().startsWith(".license_expiration_job_")) {
                this.expirationCallbacks.stream().filter(expirationCallback -> expirationCallback.getId().equals(event.getJobName())).forEach(expirationCallback -> expirationCallback.on(license));
            }
        }
    }

    public void removeLicense(DeleteLicenseRequest request, ActionListener<PostStartBasicResponse> listener) {
        PostStartBasicRequest startBasicRequest = new PostStartBasicRequest().acknowledge(true);
        this.clusterService.submitStateUpdateTask("delete license", new StartBasicClusterTask(logger, this.clusterService.getClusterName().value(), this.clock, startBasicRequest, listener));
    }

    public License getLicense() {
        License license = LicenseService.getLicense(this.clusterService.state().metadata());
        return license == LicensesMetadata.LICENSE_TOMBSTONE ? null : license;
    }

    private LicensesMetadata getLicensesMetadata() {
        return (LicensesMetadata)this.clusterService.state().metadata().custom("licenses");
    }

    void startTrialLicense(PostStartTrialRequest request, ActionListener<PostStartTrialResponse> listener) {
        License.LicenseType requestedType = License.LicenseType.parse(request.getType());
        if (!VALID_TRIAL_TYPES.contains((Object)requestedType)) {
            throw new IllegalArgumentException("Cannot start trial of type [" + requestedType.getTypeName() + "]. Valid trial types are [" + VALID_TRIAL_TYPES.stream().map(License.LicenseType::getTypeName).sorted().collect(Collectors.joining(",")) + "]");
        }
        StartTrialClusterTask task = new StartTrialClusterTask(logger, this.clusterService.getClusterName().value(), this.clock, request, listener);
        this.clusterService.submitStateUpdateTask("started trial license", task);
    }

    void startBasicLicense(PostStartBasicRequest request, ActionListener<PostStartBasicResponse> listener) {
        StartBasicClusterTask task = new StartBasicClusterTask(logger, this.clusterService.getClusterName().value(), this.clock, request, listener);
        this.clusterService.submitStateUpdateTask("start basic license", task);
    }

    private void registerOrUpdateSelfGeneratedLicense() {
        this.clusterService.submitStateUpdateTask("maybe generate license for cluster", new StartupSelfGeneratedLicenseTask(this.settings, this.clock, this.clusterService));
    }

    @Override
    protected void doStart() throws ElasticsearchException {
        ClusterState clusterState;
        this.clusterService.addListener(this);
        this.scheduler.start(Collections.emptyList());
        logger.debug("initializing license state");
        if (this.clusterService.lifecycleState() == Lifecycle.State.STARTED && !(clusterState = this.clusterService.state()).blocks().hasGlobalBlock(GatewayService.STATE_NOT_RECOVERED_BLOCK) && clusterState.nodes().getMasterNode() != null && XPackPlugin.isReadyForXPackCustomMetadata(clusterState)) {
            boolean noLicense;
            LicensesMetadata currentMetadata = (LicensesMetadata)clusterState.metadata().custom("licenses");
            boolean bl = noLicense = currentMetadata == null || currentMetadata.getLicense() == null;
            if (clusterState.getNodes().isLocalNodeElectedMaster() && (noLicense || LicenseUtils.licenseNeedsExtended(currentMetadata.getLicense()))) {
                this.registerOrUpdateSelfGeneratedLicense();
            }
        }
    }

    @Override
    protected void doStop() throws ElasticsearchException {
        this.clusterService.removeListener(this);
        this.scheduler.stop();
        this.currentLicense.set(null);
    }

    @Override
    protected void doClose() throws ElasticsearchException {
    }

    @Override
    public void clusterChanged(ClusterChangedEvent event) {
        ClusterState previousClusterState = event.previousState();
        ClusterState currentClusterState = event.state();
        if (!currentClusterState.blocks().hasGlobalBlock(GatewayService.STATE_NOT_RECOVERED_BLOCK)) {
            boolean noLicense;
            boolean noLicenseInCurrentMetadata;
            boolean noLicenseInPrevMetadata;
            if (!XPackPlugin.isReadyForXPackCustomMetadata(currentClusterState)) {
                logger.debug("cannot add license to cluster as the following nodes might not understand the license metadata: {}", () -> XPackPlugin.nodesNotReadyForXPackCustomMetadata(currentClusterState));
                return;
            }
            LicensesMetadata prevLicensesMetadata = (LicensesMetadata)previousClusterState.getMetadata().custom("licenses");
            LicensesMetadata currentLicensesMetadata = (LicensesMetadata)currentClusterState.getMetadata().custom("licenses");
            if (logger.isDebugEnabled()) {
                logger.debug("previous [{}]", (Object)prevLicensesMetadata);
                logger.debug("current [{}]", (Object)currentLicensesMetadata);
            }
            if (previousClusterState.blocks().hasGlobalBlock(GatewayService.STATE_NOT_RECOVERED_BLOCK) || prevLicensesMetadata == null) {
                if (currentLicensesMetadata != null) {
                    this.onUpdate(currentLicensesMetadata);
                }
            } else if (!prevLicensesMetadata.equals(currentLicensesMetadata)) {
                this.onUpdate(currentLicensesMetadata);
            }
            License currentLicense = null;
            boolean bl = noLicenseInPrevMetadata = prevLicensesMetadata == null || prevLicensesMetadata.getLicense() == null;
            if (!noLicenseInPrevMetadata) {
                currentLicense = prevLicensesMetadata.getLicense();
            }
            boolean bl2 = noLicenseInCurrentMetadata = currentLicensesMetadata == null || currentLicensesMetadata.getLicense() == null;
            if (!noLicenseInCurrentMetadata) {
                currentLicense = currentLicensesMetadata.getLicense();
            }
            boolean bl3 = noLicense = noLicenseInPrevMetadata && noLicenseInCurrentMetadata;
            if (currentClusterState.getNodes().isLocalNodeElectedMaster() && (noLicense || LicenseUtils.licenseNeedsExtended(currentLicense) || LicenseUtils.signatureNeedsUpdate(currentLicense, currentClusterState.nodes()))) {
                this.registerOrUpdateSelfGeneratedLicense();
            }
        } else if (logger.isDebugEnabled()) {
            logger.debug("skipped license notifications reason: [{}]", (Object)GatewayService.STATE_NOT_RECOVERED_BLOCK);
        }
    }

    private void updateLicenseState(LicensesMetadata licensesMetadata) {
        if (licensesMetadata != null) {
            this.updateLicenseState(LicenseService.getLicense(licensesMetadata), licensesMetadata.getMostRecentTrialVersion());
        }
    }

    protected void updateLicenseState(License license, Version mostRecentTrialVersion) {
        if (license == LicensesMetadata.LICENSE_TOMBSTONE) {
            this.licenseState.update(License.OperationMode.MISSING, false, license.expiryDate(), mostRecentTrialVersion);
            return;
        }
        if (license != null) {
            long time = this.clock.millis();
            boolean active = license.expiryDate() == BASIC_SELF_GENERATED_LICENSE_EXPIRATION_MILLIS ? true : time >= license.issueDate() && time < license.expiryDate();
            this.licenseState.update(license.operationMode(), active, license.expiryDate(), mostRecentTrialVersion);
            if (active) {
                logger.debug("license [{}] - valid", (Object)license.uid());
            } else {
                logger.warn("license [{}] - expired", (Object)license.uid());
            }
        }
    }

    private void onUpdate(LicensesMetadata currentLicensesMetadata) {
        License license = LicenseService.getLicense(currentLicensesMetadata);
        if (license != null) {
            License previousLicense = this.currentLicense.get();
            if (!license.equals(previousLicense)) {
                this.currentLicense.set(license);
                license.setOperationModeFileWatcher(this.operationModeFileWatcher);
                this.scheduler.add(new SchedulerEngine.Job(LICENSE_JOB, LicenseService.nextLicenseCheck(license)));
                for (ExpirationCallback expirationCallback : this.expirationCallbacks) {
                    this.scheduler.add(new SchedulerEngine.Job(expirationCallback.getId(), (startTime, now) -> expirationCallback.nextScheduledTimeForExpiry(license.expiryDate(), startTime, now)));
                }
                if (previousLicense != null) {
                    previousLicense.removeOperationModeFileWatcher();
                }
                logger.info("license [{}] mode [{}] - valid", (Object)license.uid(), (Object)license.operationMode().name().toLowerCase(Locale.ROOT));
            }
            this.updateLicenseState(license, currentLicensesMetadata.getMostRecentTrialVersion());
        }
    }

    static SchedulerEngine.Schedule nextLicenseCheck(License license) {
        return (startTime, time) -> {
            if (time < license.issueDate()) {
                return license.issueDate();
            }
            if (time < license.expiryDate()) {
                return license.expiryDate();
            }
            return -1L;
        };
    }

    public static License getLicense(Metadata metadata) {
        LicensesMetadata licensesMetadata = (LicensesMetadata)metadata.custom("licenses");
        return LicenseService.getLicense(licensesMetadata);
    }

    static License getLicense(LicensesMetadata metadata) {
        if (metadata != null) {
            boolean autoGeneratedLicense;
            License license = metadata.getLicense();
            if (license == LicensesMetadata.LICENSE_TOMBSTONE) {
                return license;
            }
            if (license != null && ((autoGeneratedLicense = License.isAutoGeneratedLicense(license.signature())) && SelfGeneratedLicense.verify(license) || !autoGeneratedLicense && LicenseVerifier.verifyLicense(license))) {
                return license;
            }
        }
        return null;
    }

    private static boolean isProductionMode(Settings settings, DiscoveryNode localNode) {
        boolean singleNodeDisco = "single-node".equals(DiscoveryModule.DISCOVERY_TYPE_SETTING.get(settings));
        return !singleNodeDisco && !LicenseService.isBoundToLoopback(localNode);
    }

    private static boolean isBoundToLoopback(DiscoveryNode localNode) {
        return localNode.getAddress().address().getAddress().isLoopbackAddress();
    }

    private static List<License.LicenseType> getAllowableUploadTypes() {
        return Collections.unmodifiableList(Stream.of(License.LicenseType.values()).filter(t -> t != License.LicenseType.BASIC).collect(Collectors.toList()));
    }

    private static void validateUploadTypesSetting(List<License.LicenseType> value) {
        if (!ALLOWABLE_UPLOAD_TYPES.containsAll(value)) {
            throw new IllegalArgumentException("Invalid value [" + value.stream().map(License.LicenseType::getTypeName).collect(Collectors.joining(",")) + "] for " + ALLOWED_LICENSE_TYPES_SETTING.getKey() + ", allowed values are [" + ALLOWABLE_UPLOAD_TYPES.stream().map(License.LicenseType::getTypeName).collect(Collectors.joining(",")) + "]");
        }
    }
}

