/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.xpack.searchablesnapshots.cache.common;

import java.io.IOException;
import java.nio.channels.FileChannel;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.StandardOpenOption;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Objects;
import java.util.Set;
import java.util.SortedSet;
import java.util.concurrent.Executor;
import java.util.concurrent.Future;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.function.Consumer;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.apache.logging.log4j.message.ParameterizedMessage;
import org.apache.lucene.store.AlreadyClosedException;
import org.elasticsearch.action.ActionListener;
import org.elasticsearch.action.support.PlainActionFuture;
import org.elasticsearch.common.util.concurrent.AbstractRunnable;
import org.elasticsearch.core.AbstractRefCounted;
import org.elasticsearch.core.Nullable;
import org.elasticsearch.core.Releasable;
import org.elasticsearch.core.Releasables;
import org.elasticsearch.core.internal.io.IOUtils;
import org.elasticsearch.xpack.searchablesnapshots.cache.common.ByteRange;
import org.elasticsearch.xpack.searchablesnapshots.cache.common.CacheKey;
import org.elasticsearch.xpack.searchablesnapshots.cache.common.SparseFileTracker;

public class CacheFile {
    private static final Logger logger = LogManager.getLogger(CacheFile.class);
    private static final StandardOpenOption[] CREATE_OPTIONS = new StandardOpenOption[]{StandardOpenOption.READ, StandardOpenOption.WRITE, StandardOpenOption.CREATE_NEW, StandardOpenOption.SPARSE};
    private static final StandardOpenOption[] OPEN_OPTIONS = new StandardOpenOption[]{StandardOpenOption.READ, StandardOpenOption.WRITE};
    private final AbstractRefCounted refCounter = new AbstractRefCounted("CacheFile"){

        protected void closeInternal() {
            assert (CacheFile.this.evicted.get());
            assert (CacheFile.this.assertNoPendingListeners());
            try {
                Files.deleteIfExists(CacheFile.this.file);
            }
            catch (IOException e) {
                logger.warn(() -> new ParameterizedMessage("Failed to delete [{}]", (Object)CacheFile.this.file), (Throwable)e);
            }
            finally {
                CacheFile.this.listener.onCacheFileDelete(CacheFile.this);
            }
        }
    };
    private final SparseFileTracker tracker;
    private final CacheKey cacheKey;
    private final Path file;
    private final Set<EvictionListener> listeners = new HashSet<EvictionListener>();
    private final AtomicBoolean needsFsync = new AtomicBoolean();
    private final ModificationListener listener;
    private final AtomicBoolean evicted = new AtomicBoolean(false);
    @Nullable
    private volatile FileChannelReference channelRef;
    private volatile boolean fileExists;

    public CacheFile(CacheKey cacheKey, long length, Path file, ModificationListener listener) {
        this(cacheKey, new SparseFileTracker(file.toString(), length), file, listener, false);
    }

    public CacheFile(CacheKey cacheKey, long length, Path file, SortedSet<ByteRange> ranges, ModificationListener listener) {
        this(cacheKey, new SparseFileTracker(file.toString(), length, ranges), file, listener, true);
    }

    private CacheFile(CacheKey cacheKey, SparseFileTracker tracker, Path file, ModificationListener listener, boolean fileExists) {
        this.cacheKey = Objects.requireNonNull(cacheKey);
        this.tracker = Objects.requireNonNull(tracker);
        this.file = Objects.requireNonNull(file);
        this.listener = Objects.requireNonNull(listener);
        assert (fileExists == Files.exists(file, new LinkOption[0])) : file + " exists? " + fileExists;
        this.fileExists = fileExists;
        assert (this.invariant());
    }

    public CacheKey getCacheKey() {
        return this.cacheKey;
    }

    public long getLength() {
        return this.tracker.getLength();
    }

    public Path getFile() {
        return this.file;
    }

    @Nullable
    FileChannel getChannel() {
        FileChannelReference reference = this.channelRef;
        return reference == null ? null : reference.fileChannel;
    }

    SortedSet<ByteRange> getCompletedRanges() {
        return this.tracker.getCompletedRanges();
    }

    public long getInitialLength() {
        return this.tracker.getInitialLength();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void acquire(EvictionListener listener) throws IOException {
        assert (listener != null);
        this.ensureOpen();
        boolean success = false;
        if (this.refCounter.tryIncRef()) {
            try {
                Set<EvictionListener> set = this.listeners;
                synchronized (set) {
                    this.ensureOpen();
                    if (this.listeners.isEmpty()) {
                        assert (this.channelRef == null);
                        this.channelRef = new FileChannelReference(this.fileExists ? OPEN_OPTIONS : CREATE_OPTIONS);
                        this.fileExists = true;
                    }
                    boolean added = this.listeners.add(listener);
                    assert (added) : "listener already exists " + listener;
                }
                success = true;
            }
            finally {
                if (!success) {
                    this.decrementRefCount();
                }
            }
        }
        assert (this.evicted.get());
        CacheFile.throwAlreadyEvicted();
        assert (this.invariant());
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void release(EvictionListener listener) {
        assert (listener != null);
        boolean success = false;
        try {
            Set<EvictionListener> set = this.listeners;
            synchronized (set) {
                boolean removed = this.listeners.remove(Objects.requireNonNull(listener));
                assert (removed) : "listener does not exist " + listener;
                if (!removed) {
                    throw new IllegalStateException("Cannot remove an unknown listener");
                }
                if (this.listeners.isEmpty()) {
                    this.channelRef.decRef();
                    this.channelRef = null;
                }
            }
            success = true;
        }
        finally {
            if (success) {
                this.decrementRefCount();
            }
        }
        assert (this.invariant());
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private boolean assertNoPendingListeners() {
        Set<EvictionListener> set = this.listeners;
        synchronized (set) {
            assert (this.listeners.isEmpty());
            assert (this.channelRef == null);
        }
        return true;
    }

    private void decrementRefCount() {
        boolean released = this.refCounter.decRef();
        assert (this.assertRefCounted(released));
    }

    private boolean assertRefCounted(boolean isReleased) {
        boolean isEvicted = this.evicted.get();
        boolean fileExists = Files.exists(this.file, new LinkOption[0]);
        assert (!isReleased || isEvicted && !fileExists) : "fully released cache file should be deleted from disk but got [released=" + isReleased + ", evicted=" + isEvicted + ", file exists=" + fileExists + ']';
        return true;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void startEviction() {
        if (this.evicted.compareAndSet(false, true)) {
            HashSet<EvictionListener> evictionListeners;
            Set<EvictionListener> set = this.listeners;
            synchronized (set) {
                evictionListeners = new HashSet<EvictionListener>(this.listeners);
            }
            this.decrementRefCount();
            evictionListeners.forEach(listener -> listener.onEviction(this));
        }
        assert (this.invariant());
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private boolean invariant() {
        Set<EvictionListener> set = this.listeners;
        synchronized (set) {
            if (this.listeners.isEmpty()) {
                assert (this.channelRef == null);
            } else {
                assert (this.channelRef != null);
                assert (this.refCounter.refCount() > 0);
                assert (this.channelRef.refCount() > 0);
                assert (Files.exists(this.file, new LinkOption[0]));
            }
        }
        return true;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public String toString() {
        Set<EvictionListener> set = this.listeners;
        synchronized (set) {
            return "CacheFile{key='" + this.cacheKey + "', file=" + this.file + ", length=" + this.tracker.getLength() + ", channel=" + (this.channelRef != null ? "yes" : "no") + ", listeners=" + this.listeners.size() + ", evicted=" + this.evicted + ", tracker=" + this.tracker + '}';
        }
    }

    private void ensureOpen() {
        if (this.evicted.get()) {
            CacheFile.throwAlreadyEvicted();
        }
    }

    private static void throwAlreadyEvicted() {
        throw new AlreadyClosedException("Cache file is evicted");
    }

    public Future<Integer> populateAndRead(ByteRange rangeToWrite, ByteRange rangeToRead, RangeAvailableHandler reader, final RangeMissingHandler writer, Executor executor) {
        PlainActionFuture future = PlainActionFuture.newFuture();
        Releasable decrementRef = null;
        try {
            final FileChannelReference reference = this.acquireFileChannelReference();
            decrementRef = Releasables.releaseOnce(() -> ((FileChannelReference)reference).decRef());
            List<SparseFileTracker.Gap> gaps = this.tracker.waitForRange(rangeToWrite, rangeToRead, CacheFile.rangeListener(rangeToRead, reader, (PlainActionFuture<Integer>)future, reference, decrementRef));
            for (final SparseFileTracker.Gap gap : gaps) {
                executor.execute((Runnable)new AbstractRunnable(){

                    protected void doRun() throws Exception {
                        if (!reference.tryIncRef()) {
                            throw new AlreadyClosedException("Cache file channel has been released and closed");
                        }
                        try {
                            CacheFile.this.ensureOpen();
                            writer.fillCacheRange(reference.fileChannel, gap.start(), gap.end(), gap::onProgress);
                            gap.onCompletion();
                            CacheFile.this.markAsNeedsFSync();
                        }
                        finally {
                            reference.decRef();
                        }
                    }

                    public void onFailure(Exception e) {
                        gap.onFailure(e);
                    }
                });
            }
        }
        catch (Exception e) {
            CacheFile.releaseAndFail((PlainActionFuture<Integer>)future, decrementRef, e);
        }
        return future;
    }

    @Nullable
    public Future<Integer> readIfAvailableOrPending(ByteRange rangeToRead, RangeAvailableHandler reader) {
        PlainActionFuture future = PlainActionFuture.newFuture();
        Releasable decrementRef = null;
        try {
            FileChannelReference reference = this.acquireFileChannelReference();
            decrementRef = Releasables.releaseOnce(() -> ((FileChannelReference)reference).decRef());
            if (this.tracker.waitForRangeIfPending(rangeToRead, CacheFile.rangeListener(rangeToRead, reader, (PlainActionFuture<Integer>)future, reference, decrementRef))) {
                return future;
            }
            decrementRef.close();
            return null;
        }
        catch (Exception e) {
            CacheFile.releaseAndFail((PlainActionFuture<Integer>)future, decrementRef, e);
            return future;
        }
    }

    private static void releaseAndFail(PlainActionFuture<Integer> future, Releasable decrementRef, Exception e) {
        try {
            Releasables.close((Releasable)decrementRef);
        }
        catch (Exception ex) {
            e.addSuppressed(ex);
        }
        future.onFailure(e);
    }

    private static ActionListener<Void> rangeListener(ByteRange rangeToRead, RangeAvailableHandler reader, PlainActionFuture<Integer> future, FileChannelReference reference, Releasable releasable) {
        return ActionListener.runAfter((ActionListener)ActionListener.wrap(success -> {
            int read = reader.onRangeAvailable(reference.fileChannel);
            assert ((long)read == rangeToRead.length()) : "partial read [" + read + "] does not match the range to read [" + rangeToRead.end() + '-' + rangeToRead.start() + ']';
            future.onResponse((Object)read);
        }, arg_0 -> future.onFailure(arg_0)), () -> ((Releasable)releasable).close());
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private FileChannelReference acquireFileChannelReference() {
        FileChannelReference reference;
        Set<EvictionListener> set = this.listeners;
        synchronized (set) {
            this.ensureOpen();
            reference = this.channelRef;
            assert (reference != null && reference.refCount() > 0) : "impossible to run into a fully released channel reference under the listeners mutex";
            assert (this.refCounter.refCount() > 0) : "file should not be fully released";
            reference.incRef();
        }
        return reference;
    }

    public ByteRange getAbsentRangeWithin(ByteRange range) {
        this.ensureOpen();
        return this.tracker.getAbsentRangeWithin(range);
    }

    boolean needsFsync() {
        return this.needsFsync.get();
    }

    private void markAsNeedsFSync() {
        assert (this.refCounter.refCount() > 0) : "file should not be fully released";
        if (!this.needsFsync.getAndSet(true)) {
            this.listener.onCacheFileNeedsFsync(this);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public SortedSet<ByteRange> fsync() throws IOException {
        block13: {
            if (this.refCounter.tryIncRef()) {
                try {
                    if (this.needsFsync.compareAndSet(true, false)) {
                        boolean success = false;
                        try {
                            SortedSet<ByteRange> completedRanges = this.tracker.getCompletedRanges();
                            assert (completedRanges != null);
                            assert (!completedRanges.isEmpty());
                            IOUtils.fsync((Path)this.file, (boolean)false, (boolean)false);
                            success = true;
                            SortedSet<ByteRange> sortedSet = completedRanges;
                            if (!success) {
                                this.markAsNeedsFSync();
                            }
                            return sortedSet;
                        }
                        catch (Throwable throwable) {
                            if (!success) {
                                this.markAsNeedsFSync();
                            }
                            throw throwable;
                        }
                    }
                    break block13;
                }
                finally {
                    this.decrementRefCount();
                }
            }
            assert (this.evicted.get());
        }
        return Collections.emptySortedSet();
    }

    public static interface ModificationListener {
        public void onCacheFileNeedsFsync(CacheFile var1);

        public void onCacheFileDelete(CacheFile var1);
    }

    private final class FileChannelReference
    extends AbstractRefCounted {
        private final FileChannel fileChannel;

        FileChannelReference(StandardOpenOption[] options) throws IOException {
            super("FileChannel[" + CacheFile.this.file + "]");
            this.fileChannel = FileChannel.open(CacheFile.this.file, options);
            CacheFile.this.refCounter.incRef();
        }

        protected void closeInternal() {
            try {
                this.fileChannel.close();
            }
            catch (IOException e) {
                logger.warn(() -> new ParameterizedMessage("Failed to close [{}]", (Object)CacheFile.this.file), (Throwable)e);
            }
            finally {
                CacheFile.this.decrementRefCount();
            }
        }
    }

    @FunctionalInterface
    public static interface RangeAvailableHandler {
        public int onRangeAvailable(FileChannel var1) throws IOException;
    }

    @FunctionalInterface
    public static interface RangeMissingHandler {
        public void fillCacheRange(FileChannel var1, long var2, long var4, Consumer<Long> var6) throws IOException;
    }

    @FunctionalInterface
    public static interface EvictionListener {
        public void onEviction(CacheFile var1);
    }
}

