/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.nio;

import java.io.Closeable;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.nio.channels.CancelledKeyException;
import java.nio.channels.ClosedChannelException;
import java.nio.channels.ClosedSelectorException;
import java.nio.channels.SelectionKey;
import java.nio.channels.Selector;
import java.util.Iterator;
import java.util.Objects;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ConcurrentLinkedQueue;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.Future;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicReference;
import java.util.concurrent.locks.ReentrantLock;
import java.util.function.BiConsumer;
import java.util.stream.Collectors;
import org.elasticsearch.nio.ChannelContext;
import org.elasticsearch.nio.EventHandler;
import org.elasticsearch.nio.NioChannel;
import org.elasticsearch.nio.ServerChannelContext;
import org.elasticsearch.nio.SocketChannelContext;
import org.elasticsearch.nio.TaskScheduler;
import org.elasticsearch.nio.WriteOperation;

public class NioSelector
implements Closeable {
    private final ConcurrentLinkedQueue<WriteOperation> queuedWrites = new ConcurrentLinkedQueue();
    private final ConcurrentLinkedQueue<ChannelContext<?>> channelsToClose = new ConcurrentLinkedQueue();
    private final ConcurrentLinkedQueue<ChannelContext<?>> channelsToRegister = new ConcurrentLinkedQueue();
    private final EventHandler eventHandler;
    private final Selector selector;
    private final ByteBuffer ioBuffer;
    private final TaskScheduler taskScheduler = new TaskScheduler();
    private final ReentrantLock runLock = new ReentrantLock();
    private final CountDownLatch exitedLoop = new CountDownLatch(1);
    private final AtomicBoolean isClosed = new AtomicBoolean(false);
    private final CompletableFuture<Void> isRunningFuture = new CompletableFuture();
    private final AtomicReference<Thread> thread = new AtomicReference<Object>(null);
    private final AtomicBoolean wokenUp = new AtomicBoolean(false);

    public NioSelector(EventHandler eventHandler) throws IOException {
        this(eventHandler, Selector.open());
    }

    public NioSelector(EventHandler eventHandler, Selector selector) {
        this.selector = selector;
        this.eventHandler = eventHandler;
        this.ioBuffer = ByteBuffer.allocateDirect(262144);
    }

    public ByteBuffer getIoBuffer() {
        this.assertOnSelectorThread();
        this.ioBuffer.clear();
        return this.ioBuffer;
    }

    public TaskScheduler getTaskScheduler() {
        return this.taskScheduler;
    }

    public Selector rawSelector() {
        return this.selector;
    }

    public boolean isOpen() {
        return !this.isClosed.get();
    }

    public boolean isRunning() {
        return this.runLock.isLocked();
    }

    Future<Void> isRunningFuture() {
        return this.isRunningFuture;
    }

    void setThread() {
        boolean wasSet = this.thread.compareAndSet(null, Thread.currentThread());
        assert (wasSet) : "Failed to set thread as it was already set. Should only set once.";
    }

    public boolean isOnCurrentThread() {
        return Thread.currentThread() == this.thread.get();
    }

    public void assertOnSelectorThread() {
        assert (this.isOnCurrentThread()) : "Must be on selector thread [" + this.thread.get().getName() + "} to perform this operation. Currently on thread [" + Thread.currentThread().getName() + "].";
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    public void runLoop() {
        if (!this.runLock.tryLock()) {
            throw new IllegalStateException("selector is already running");
        }
        this.isRunningFuture.complete(null);
        try {
            this.setThread();
            while (this.isOpen()) {
                this.singleLoop();
            }
            return;
        }
        finally {
            try {
                this.cleanupAndCloseChannels();
            }
            finally {
                try {
                    this.selector.close();
                }
                catch (IOException e) {
                    this.eventHandler.selectorException(e);
                }
                finally {
                    this.runLock.unlock();
                    this.exitedLoop.countDown();
                }
            }
        }
    }

    void singleLoop() {
        try {
            int ready;
            this.closePendingChannels();
            this.preSelect();
            long nanosUntilNextTask = this.taskScheduler.nanosUntilNextTask(System.nanoTime());
            if (this.wokenUp.getAndSet(false) || nanosUntilNextTask == 0L) {
                ready = this.selector.selectNow();
            } else {
                long millisUntilNextTask = TimeUnit.NANOSECONDS.toMillis(nanosUntilNextTask);
                ready = this.selector.select(Math.min(300L, Math.max(millisUntilNextTask, 1L)));
            }
            if (ready > 0) {
                Set<SelectionKey> selectionKeys = this.selector.selectedKeys();
                Iterator<SelectionKey> keyIterator = selectionKeys.iterator();
                while (keyIterator.hasNext()) {
                    SelectionKey sk = keyIterator.next();
                    keyIterator.remove();
                    if (sk.isValid()) {
                        try {
                            this.processKey(sk);
                        }
                        catch (CancelledKeyException cke) {
                            this.eventHandler.genericChannelException((ChannelContext)sk.attachment(), cke);
                        }
                        continue;
                    }
                    this.eventHandler.genericChannelException((ChannelContext)sk.attachment(), new CancelledKeyException());
                }
            }
            this.handleScheduledTasks(System.nanoTime());
        }
        catch (ClosedSelectorException e) {
            if (this.isOpen()) {
                throw e;
            }
        }
        catch (IOException e) {
            this.eventHandler.selectorException(e);
        }
        catch (Exception e) {
            this.eventHandler.uncaughtException(e);
        }
    }

    void cleanupAndCloseChannels() {
        this.cleanupPendingWrites();
        this.channelsToClose.addAll(this.channelsToRegister);
        this.channelsToRegister.clear();
        this.channelsToClose.addAll(this.selector.keys().stream().map(sk -> (ChannelContext)sk.attachment()).filter(Objects::nonNull).collect(Collectors.toList()));
        this.closePendingChannels();
    }

    @Override
    public void close() throws IOException {
        if (this.isClosed.compareAndSet(false, true)) {
            this.wakeup();
            if (this.isRunning()) {
                try {
                    this.exitedLoop.await();
                }
                catch (InterruptedException e) {
                    Thread.currentThread().interrupt();
                    throw new IllegalStateException("Thread was interrupted while waiting for selector to close", e);
                }
            } else if (this.selector.isOpen()) {
                this.selector.close();
            }
        }
    }

    void processKey(SelectionKey selectionKey) {
        ChannelContext context = (ChannelContext)selectionKey.attachment();
        if (selectionKey.isAcceptable()) {
            assert (context instanceof ServerChannelContext) : "Only server channels can receive accept events";
            ServerChannelContext serverChannelContext = (ServerChannelContext)context;
            try {
                this.eventHandler.acceptChannel(serverChannelContext);
            }
            catch (IOException e) {
                this.eventHandler.acceptException(serverChannelContext, e);
            }
        } else {
            assert (context instanceof SocketChannelContext) : "Only sockets channels can receive non-accept events";
            SocketChannelContext channelContext = (SocketChannelContext)context;
            int ops = selectionKey.readyOps();
            if ((ops & 8) != 0) {
                this.attemptConnect(channelContext, true);
            }
            if (channelContext.isConnectComplete() && !channelContext.selectorShouldClose()) {
                if ((ops & 4) != 0) {
                    this.handleWrite(channelContext);
                }
                if (!channelContext.selectorShouldClose() && (ops & 1) != 0) {
                    this.handleRead(channelContext);
                }
            }
            this.eventHandler.postHandling(channelContext);
        }
    }

    void preSelect() {
        this.setUpNewChannels();
        this.handleQueuedWrites();
    }

    private void handleScheduledTasks(long nanoTime) {
        Runnable task;
        while ((task = this.taskScheduler.pollTask(nanoTime)) != null) {
            this.handleTask(task);
        }
    }

    private void handleTask(Runnable task) {
        try {
            this.eventHandler.handleTask(task);
        }
        catch (Exception e) {
            this.eventHandler.taskException(e);
        }
    }

    public void queueWrite(WriteOperation writeOperation) {
        if (this.isOnCurrentThread()) {
            this.writeToChannel(writeOperation);
        } else {
            this.queuedWrites.offer(writeOperation);
            if (!this.isOpen()) {
                boolean wasRemoved = this.queuedWrites.remove(writeOperation);
                if (wasRemoved) {
                    writeOperation.getListener().accept(null, new ClosedSelectorException());
                }
            } else {
                this.wakeup();
            }
        }
    }

    public void queueChannelClose(NioChannel channel) {
        ChannelContext<?> context = channel.getContext();
        assert (context.getSelector() == this) : "Must schedule a channel for closure with its selector";
        if (!this.isOnCurrentThread()) {
            this.channelsToClose.offer(context);
            this.ensureSelectorOpenForEnqueuing(this.channelsToClose, context);
            this.wakeup();
        } else {
            this.closeChannel(context);
        }
    }

    public void scheduleForRegistration(NioChannel channel) {
        ChannelContext<?> context = channel.getContext();
        if (!this.isOnCurrentThread()) {
            this.channelsToRegister.add(context);
            this.ensureSelectorOpenForEnqueuing(this.channelsToRegister, context);
            this.wakeup();
        } else {
            this.registerChannel(context);
        }
    }

    private void writeToChannel(WriteOperation writeOperation) {
        this.assertOnSelectorThread();
        SocketChannelContext context = writeOperation.getChannel();
        if (!context.isOpen()) {
            this.executeFailedListener(writeOperation.getListener(), new ClosedChannelException());
        } else if (context.getSelectionKey() == null) {
            this.executeFailedListener(writeOperation.getListener(), new IllegalStateException("Channel not registered"));
        } else {
            boolean shouldFlushAfterQueuing = !context.readyForFlush();
            try {
                context.queueWriteOperation(writeOperation);
            }
            catch (Exception e) {
                shouldFlushAfterQueuing = false;
                this.executeFailedListener(writeOperation.getListener(), e);
            }
            if (shouldFlushAfterQueuing) {
                if (context.isConnectComplete() && !context.selectorShouldClose()) {
                    this.handleWrite(context);
                }
                this.eventHandler.postHandling(context);
            }
        }
    }

    public <V> void executeListener(BiConsumer<V, Exception> listener, V value) {
        this.assertOnSelectorThread();
        this.handleTask(() -> listener.accept(value, null));
    }

    public <V> void executeFailedListener(BiConsumer<V, Exception> listener, Exception exception) {
        this.assertOnSelectorThread();
        this.handleTask(() -> listener.accept(null, exception));
    }

    private void cleanupPendingWrites() {
        WriteOperation op;
        while ((op = this.queuedWrites.poll()) != null) {
            this.executeFailedListener(op.getListener(), new ClosedSelectorException());
        }
    }

    private void wakeup() {
        assert (!this.isOnCurrentThread());
        if (this.wokenUp.compareAndSet(false, true)) {
            this.selector.wakeup();
        }
    }

    private void handleWrite(SocketChannelContext context) {
        try {
            this.eventHandler.handleWrite(context);
        }
        catch (Exception e) {
            this.eventHandler.writeException(context, e);
        }
    }

    private void handleRead(SocketChannelContext context) {
        try {
            this.eventHandler.handleRead(context);
        }
        catch (Exception e) {
            this.eventHandler.readException(context, e);
        }
    }

    private void attemptConnect(SocketChannelContext context, boolean connectEvent) {
        try {
            this.eventHandler.handleConnect(context);
            if (connectEvent && !context.isConnectComplete()) {
                this.eventHandler.connectException(context, new IOException("Received OP_CONNECT but connect failed"));
            }
        }
        catch (Exception e) {
            this.eventHandler.connectException(context, e);
        }
    }

    private void setUpNewChannels() {
        ChannelContext<?> newChannel;
        while ((newChannel = this.channelsToRegister.poll()) != null) {
            this.registerChannel(newChannel);
        }
    }

    private void registerChannel(ChannelContext<?> newChannel) {
        assert (newChannel.getSelector() == this) : "The channel must be registered with the selector with which it was created";
        try {
            if (newChannel.isOpen()) {
                this.eventHandler.handleRegistration(newChannel);
                this.channelActive(newChannel);
                if (newChannel instanceof SocketChannelContext) {
                    this.attemptConnect((SocketChannelContext)newChannel, false);
                }
            } else {
                this.eventHandler.registrationException(newChannel, new ClosedChannelException());
                this.closeChannel(newChannel);
            }
        }
        catch (Exception e) {
            this.eventHandler.registrationException(newChannel, e);
            this.closeChannel(newChannel);
        }
    }

    private void channelActive(ChannelContext<?> newChannel) {
        try {
            this.eventHandler.handleActive(newChannel);
        }
        catch (IOException e) {
            this.eventHandler.activeException(newChannel, e);
        }
    }

    private void closePendingChannels() {
        ChannelContext<?> channelContext;
        while ((channelContext = this.channelsToClose.poll()) != null) {
            this.closeChannel(channelContext);
        }
    }

    private void closeChannel(ChannelContext<?> channelContext) {
        try {
            this.eventHandler.handleClose(channelContext);
        }
        catch (Exception e) {
            this.eventHandler.closeException(channelContext, e);
        }
    }

    private void handleQueuedWrites() {
        WriteOperation writeOperation;
        while ((writeOperation = this.queuedWrites.poll()) != null) {
            this.writeToChannel(writeOperation);
        }
    }

    private <O> void ensureSelectorOpenForEnqueuing(ConcurrentLinkedQueue<O> queue, O objectAdded) {
        if (!this.isOpen() && queue.remove(objectAdded)) {
            throw new IllegalStateException("selector is already closed");
        }
    }
}

