/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.common.cache;

import java.lang.reflect.Array;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;
import java.util.Objects;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.CompletionStage;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.atomic.LongAdder;
import java.util.concurrent.locks.ReadWriteLock;
import java.util.concurrent.locks.ReentrantLock;
import java.util.concurrent.locks.ReentrantReadWriteLock;
import java.util.function.BiConsumer;
import java.util.function.BiFunction;
import java.util.function.ToLongBiFunction;
import org.elasticsearch.common.cache.CacheLoader;
import org.elasticsearch.common.cache.RemovalListener;
import org.elasticsearch.common.cache.RemovalNotification;
import org.elasticsearch.common.util.concurrent.ReleasableLock;
import org.elasticsearch.core.Tuple;

public class Cache<K, V> {
    private final LongAdder hits = new LongAdder();
    private final LongAdder misses = new LongAdder();
    private final LongAdder evictions = new LongAdder();
    private long expireAfterAccessNanos = -1L;
    private boolean entriesExpireAfterAccess;
    private long expireAfterWriteNanos = -1L;
    private boolean entriesExpireAfterWrite;
    private int count = 0;
    private long weight = 0L;
    private long maximumWeight = -1L;
    private ToLongBiFunction<K, V> weigher = (k, v) -> 1L;
    private RemovalListener<K, V> removalListener = notification -> {};
    public static final int NUMBER_OF_SEGMENTS = 256;
    private final CacheSegment[] segments = (CacheSegment[])Array.newInstance(CacheSegment.class, 256);
    Entry<K, V> head;
    Entry<K, V> tail;
    private final ReleasableLock lruLock;

    Cache() {
        for (int i = 0; i < this.segments.length; ++i) {
            this.segments[i] = new CacheSegment();
        }
        this.lruLock = new ReleasableLock(new ReentrantLock());
    }

    void setExpireAfterAccessNanos(long expireAfterAccessNanos) {
        if (expireAfterAccessNanos <= 0L) {
            throw new IllegalArgumentException("expireAfterAccessNanos <= 0");
        }
        this.expireAfterAccessNanos = expireAfterAccessNanos;
        this.entriesExpireAfterAccess = true;
    }

    long getExpireAfterAccessNanos() {
        return this.expireAfterAccessNanos;
    }

    void setExpireAfterWriteNanos(long expireAfterWriteNanos) {
        if (expireAfterWriteNanos <= 0L) {
            throw new IllegalArgumentException("expireAfterWriteNanos <= 0");
        }
        this.expireAfterWriteNanos = expireAfterWriteNanos;
        this.entriesExpireAfterWrite = true;
    }

    long getExpireAfterWriteNanos() {
        return this.expireAfterWriteNanos;
    }

    void setMaximumWeight(long maximumWeight) {
        if (maximumWeight < 0L) {
            throw new IllegalArgumentException("maximumWeight < 0");
        }
        this.maximumWeight = maximumWeight;
    }

    void setWeigher(ToLongBiFunction<K, V> weigher) {
        Objects.requireNonNull(weigher);
        this.weigher = weigher;
    }

    void setRemovalListener(RemovalListener<K, V> removalListener) {
        Objects.requireNonNull(removalListener);
        this.removalListener = removalListener;
    }

    protected long now() {
        return this.entriesExpireAfterAccess || this.entriesExpireAfterWrite ? System.nanoTime() : 0L;
    }

    public V get(K key) {
        return this.get(key, this.now(), false);
    }

    private V get(K key, long now, boolean eagerEvict) {
        CacheSegment segment = this.getCacheSegment(key);
        Entry entry = segment.get(key, now, eagerEvict);
        if (entry == null) {
            return null;
        }
        this.promote(entry, now);
        return entry.value;
    }

    public V computeIfAbsent(K key, CacheLoader<K, V> loader) throws ExecutionException {
        long now = this.now();
        Object value = this.get(key, now, true);
        if (value == null) {
            CompletionStage completableValue;
            CompletableFuture<Entry<K, V>> future;
            CacheSegment segment = this.getCacheSegment(key);
            CompletableFuture completableFuture = new CompletableFuture();
            try (ReleasableLock ignored = segment.writeLock.acquire();){
                if (segment.map == null) {
                    segment.map = new HashMap();
                }
                future = segment.map.putIfAbsent(key, completableFuture);
            }
            BiFunction<Entry, Throwable, Object> handler = (ok, ex) -> {
                if (ok != null) {
                    this.promote((Entry<K, V>)ok, now);
                    return ok.value;
                }
                try (ReleasableLock ignored = segment.writeLock.acquire();){
                    CompletableFuture sanity;
                    CompletableFuture completableFuture = sanity = segment.map == null ? null : segment.map.get(key);
                    if (sanity != null && sanity.isCompletedExceptionally()) {
                        segment.map.remove(key);
                        if (segment.map.isEmpty()) {
                            segment.map = null;
                        }
                    }
                }
                return null;
            };
            if (future == null) {
                V loaded;
                future = completableFuture;
                completableValue = future.handle(handler);
                try {
                    loaded = loader.load(key);
                }
                catch (Exception e) {
                    future.completeExceptionally(e);
                    throw new ExecutionException(e);
                }
                if (loaded == null) {
                    NullPointerException npe = new NullPointerException("loader returned a null value");
                    future.completeExceptionally(npe);
                    throw new ExecutionException(npe);
                }
                future.complete(new Entry<K, V>(key, loaded, now));
            } else {
                completableValue = future.handle(handler);
            }
            try {
                value = ((CompletableFuture)completableValue).get();
                if (future.isCompletedExceptionally()) {
                    future.get();
                    throw new IllegalStateException("the future was completed exceptionally but no exception was thrown");
                }
            }
            catch (InterruptedException e) {
                throw new IllegalStateException(e);
            }
        }
        return value;
    }

    public void put(K key, V value) {
        long now = this.now();
        this.put(key, value, now);
    }

    private void put(K key, V value, long now) {
        CacheSegment segment = this.getCacheSegment(key);
        Tuple tuple = segment.put(key, value, now);
        boolean replaced = false;
        try (ReleasableLock ignored = this.lruLock.acquire();){
            if (tuple.v2() != null && tuple.v2().state == State.EXISTING && this.unlink(tuple.v2())) {
                replaced = true;
            }
            this.promote(tuple.v1(), now);
        }
        if (replaced) {
            this.removalListener.onRemoval(new RemovalNotification(tuple.v2().key, tuple.v2().value, RemovalNotification.RemovalReason.REPLACED));
        }
    }

    private void notifyWithInvalidated(CompletableFuture<Entry<K, V>> f) {
        try {
            Entry<K, V> entry = f.get();
            try (ReleasableLock ignored = this.lruLock.acquire();){
                this.delete(entry, RemovalNotification.RemovalReason.INVALIDATED);
            }
        }
        catch (ExecutionException entry) {
        }
        catch (InterruptedException e) {
            throw new IllegalStateException(e);
        }
    }

    public void invalidate(K key) {
        CacheSegment segment = this.getCacheSegment(key);
        segment.remove(key);
    }

    public void invalidate(K key, V value) {
        CacheSegment segment = this.getCacheSegment(key);
        segment.remove(key, value, true);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void invalidateAll() {
        Entry<K, V> h;
        boolean[] haveSegmentLock = new boolean[256];
        try {
            for (int i = 0; i < 256; ++i) {
                this.segments[i].segmentLock.writeLock().lock();
                haveSegmentLock[i] = true;
            }
            try (ReleasableLock ignored = this.lruLock.acquire();){
                h = this.head;
                for (CacheSegment segment : this.segments) {
                    segment.map = null;
                }
                Entry<K, V> current = this.head;
                while (current != null) {
                    current.state = State.DELETED;
                    current = current.after;
                }
                this.tail = null;
                this.head = null;
                this.count = 0;
                this.weight = 0L;
            }
        }
        catch (Throwable throwable) {
            for (int i = 255; i >= 0; --i) {
                if (!haveSegmentLock[i]) continue;
                this.segments[i].segmentLock.writeLock().unlock();
            }
            throw throwable;
        }
        for (int i = 255; i >= 0; --i) {
            if (!haveSegmentLock[i]) continue;
            this.segments[i].segmentLock.writeLock().unlock();
        }
        while (h != null) {
            this.removalListener.onRemoval(new RemovalNotification(h.key, h.value, RemovalNotification.RemovalReason.INVALIDATED));
            h = h.after;
        }
    }

    public void refresh() {
        long now = this.now();
        try (ReleasableLock ignored = this.lruLock.acquire();){
            this.evict(now);
        }
    }

    public int count() {
        return this.count;
    }

    public long weight() {
        return this.weight;
    }

    public Iterable<K> keys() {
        return () -> new Iterator<K>(){
            private final CacheIterator iterator;
            {
                this.iterator = new CacheIterator(Cache.this.head);
            }

            @Override
            public boolean hasNext() {
                return this.iterator.hasNext();
            }

            @Override
            public K next() {
                return ((Entry)this.iterator.next()).key;
            }

            @Override
            public void remove() {
                this.iterator.remove();
            }
        };
    }

    public Iterable<V> values() {
        return () -> new Iterator<V>(){
            private final CacheIterator iterator;
            {
                this.iterator = new CacheIterator(Cache.this.head);
            }

            @Override
            public boolean hasNext() {
                return this.iterator.hasNext();
            }

            @Override
            public V next() {
                return ((Entry)this.iterator.next()).value;
            }

            @Override
            public void remove() {
                this.iterator.remove();
            }
        };
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    public void forEach(BiConsumer<K, V> consumer) {
        CacheSegment[] cacheSegmentArray = this.segments;
        int n = cacheSegmentArray.length;
        int n2 = 0;
        while (n2 < n) {
            CacheSegment segment = cacheSegmentArray[n2];
            try (ReleasableLock ignored = segment.readLock.acquire();){
                if (segment.map != null) {
                    for (CompletableFuture future : segment.map.values()) {
                        try {
                            if (future == null || !future.isDone()) continue;
                            Entry entry = future.get();
                            consumer.accept(entry.key, entry.value);
                        }
                        catch (InterruptedException | ExecutionException e) {
                            throw new IllegalStateException(e);
                        }
                    }
                }
            }
            ++n2;
        }
        return;
    }

    public CacheStats stats() {
        return new CacheStats(this.hits.sum(), this.misses.sum(), this.evictions.sum());
    }

    private void promote(Entry<K, V> entry, long now) {
        boolean promoted = true;
        try (ReleasableLock ignored = this.lruLock.acquire();){
            switch (entry.state) {
                case DELETED: {
                    promoted = false;
                    break;
                }
                case EXISTING: {
                    this.relinkAtHead(entry);
                    break;
                }
                case NEW: {
                    this.linkAtHead(entry);
                }
            }
            if (promoted) {
                this.evict(now);
            }
        }
    }

    private void evict(long now) {
        assert (this.lruLock.isHeldByCurrentThread());
        while (this.tail != null && this.shouldPrune(this.tail, now)) {
            this.evictEntry(this.tail);
        }
    }

    private void evictEntry(Entry<K, V> entry) {
        assert (this.lruLock.isHeldByCurrentThread());
        CacheSegment segment = this.getCacheSegment(entry.key);
        if (segment != null) {
            segment.remove(entry.key, entry.value, false);
        }
        this.delete(entry, RemovalNotification.RemovalReason.EVICTED);
    }

    private void delete(Entry<K, V> entry, RemovalNotification.RemovalReason removalReason) {
        assert (this.lruLock.isHeldByCurrentThread());
        if (this.unlink(entry)) {
            this.removalListener.onRemoval(new RemovalNotification(entry.key, entry.value, removalReason));
        }
    }

    private boolean shouldPrune(Entry<K, V> entry, long now) {
        return this.exceedsWeight() || this.isExpired(entry, now);
    }

    private boolean exceedsWeight() {
        return this.maximumWeight != -1L && this.weight > this.maximumWeight;
    }

    private boolean isExpired(Entry<K, V> entry, long now) {
        return this.entriesExpireAfterAccess && now - entry.accessTime > this.expireAfterAccessNanos || this.entriesExpireAfterWrite && now - entry.writeTime > this.expireAfterWriteNanos;
    }

    private boolean unlink(Entry<K, V> entry) {
        assert (this.lruLock.isHeldByCurrentThread());
        if (entry.state == State.EXISTING) {
            Entry before = entry.before;
            Entry after = entry.after;
            if (before == null) {
                assert (this.head == entry);
                this.head = after;
                if (this.head != null) {
                    this.head.before = null;
                }
            } else {
                before.after = after;
                entry.before = null;
            }
            if (after == null) {
                assert (this.tail == entry);
                this.tail = before;
                if (this.tail != null) {
                    this.tail.after = null;
                }
            } else {
                after.before = before;
                entry.after = null;
            }
            --this.count;
            this.weight -= this.weigher.applyAsLong(entry.key, entry.value);
            entry.state = State.DELETED;
            return true;
        }
        return false;
    }

    private void linkAtHead(Entry<K, V> entry) {
        assert (this.lruLock.isHeldByCurrentThread());
        Entry<K, V> h = this.head;
        entry.before = null;
        entry.after = this.head;
        this.head = entry;
        if (h == null) {
            this.tail = entry;
        } else {
            h.before = entry;
        }
        ++this.count;
        this.weight += this.weigher.applyAsLong(entry.key, entry.value);
        entry.state = State.EXISTING;
    }

    private void relinkAtHead(Entry<K, V> entry) {
        assert (this.lruLock.isHeldByCurrentThread());
        if (this.head != entry) {
            this.unlink(entry);
            this.linkAtHead(entry);
        }
    }

    private CacheSegment getCacheSegment(K key) {
        return this.segments[key.hashCode() & 0xFF];
    }

    private final class CacheSegment {
        ReadWriteLock segmentLock = new ReentrantReadWriteLock();
        ReleasableLock readLock = new ReleasableLock(this.segmentLock.readLock());
        ReleasableLock writeLock = new ReleasableLock(this.segmentLock.writeLock());
        Map<K, CompletableFuture<Entry<K, V>>> map;

        private CacheSegment() {
        }

        Entry<K, V> get(K key, long now, boolean eagerEvict) {
            CompletableFuture future;
            try (ReleasableLock ignored = this.readLock.acquire();){
                future = this.map == null ? null : this.map.get(key);
            }
            if (future != null) {
                Entry entry;
                try {
                    entry = future.get();
                }
                catch (ExecutionException e) {
                    assert (future.isCompletedExceptionally());
                    Cache.this.misses.increment();
                    return null;
                }
                catch (InterruptedException e) {
                    throw new IllegalStateException(e);
                }
                if (Cache.this.isExpired(entry, now)) {
                    Cache.this.misses.increment();
                    if (eagerEvict) {
                        try (ReleasableLock ignored = Cache.this.lruLock.acquire();){
                            Cache.this.evictEntry(entry);
                        }
                    }
                    return null;
                }
                Cache.this.hits.increment();
                entry.accessTime = now;
                return entry;
            }
            Cache.this.misses.increment();
            return null;
        }

        Tuple<Entry<K, V>, Entry<K, V>> put(K key, V value, long now) {
            Entry entry = new Entry(key, value, now);
            Entry existing = null;
            try (ReleasableLock ignored = this.writeLock.acquire();){
                try {
                    CompletableFuture future;
                    if (this.map == null) {
                        this.map = new HashMap();
                    }
                    if ((future = this.map.put(key, CompletableFuture.completedFuture(entry))) != null) {
                        existing = (Entry)((CompletableFuture)future.handle((ok, ex) -> ok)).get();
                    }
                }
                catch (InterruptedException | ExecutionException e) {
                    throw new IllegalStateException(e);
                }
            }
            return Tuple.tuple(entry, existing);
        }

        void remove(K key) {
            CompletableFuture future;
            try (ReleasableLock ignored = this.writeLock.acquire();){
                if (this.map == null) {
                    future = null;
                } else {
                    future = this.map.remove(key);
                    if (this.map.isEmpty()) {
                        this.map = null;
                    }
                }
            }
            if (future != null) {
                Cache.this.evictions.increment();
                Cache.this.notifyWithInvalidated(future);
            }
        }

        void remove(K key, V value, boolean notify) {
            CompletableFuture future;
            boolean removed = false;
            try (ReleasableLock ignored = this.writeLock.acquire();){
                future = this.map == null ? null : this.map.get(key);
                try {
                    if (future != null && future.isDone()) {
                        Entry entry = future.get();
                        if (Objects.equals(value, entry.value)) {
                            removed = this.map.remove(key, future);
                            if (this.map.isEmpty()) {
                                this.map = null;
                            }
                        }
                    }
                }
                catch (InterruptedException | ExecutionException e) {
                    throw new IllegalStateException(e);
                }
            }
            if (future != null && removed) {
                Cache.this.evictions.increment();
                if (notify) {
                    Cache.this.notifyWithInvalidated(future);
                }
            }
        }
    }

    private static final class Entry<K, V> {
        final K key;
        final V value;
        final long writeTime;
        volatile long accessTime;
        Entry<K, V> before;
        Entry<K, V> after;
        State state = State.NEW;

        Entry(K key, V value, long writeTime) {
            this.key = key;
            this.value = value;
            this.writeTime = this.accessTime = writeTime;
        }
    }

    static enum State {
        NEW,
        EXISTING,
        DELETED;

    }

    public static class CacheStats {
        private final long hits;
        private final long misses;
        private final long evictions;

        public CacheStats(long hits, long misses, long evictions) {
            this.hits = hits;
            this.misses = misses;
            this.evictions = evictions;
        }

        public long getHits() {
            return this.hits;
        }

        public long getMisses() {
            return this.misses;
        }

        public long getEvictions() {
            return this.evictions;
        }
    }

    private class CacheIterator
    implements Iterator<Entry<K, V>> {
        private Entry<K, V> current = null;
        private Entry<K, V> next;

        CacheIterator(Entry<K, V> head) {
            this.next = head;
        }

        @Override
        public boolean hasNext() {
            return this.next != null;
        }

        @Override
        public Entry<K, V> next() {
            this.current = this.next;
            this.next = this.next.after;
            return this.current;
        }

        @Override
        public void remove() {
            Entry entry = this.current;
            if (entry != null) {
                CacheSegment segment = Cache.this.getCacheSegment(entry.key);
                segment.remove(entry.key, entry.value, false);
                try (ReleasableLock ignored = Cache.this.lruLock.acquire();){
                    this.current = null;
                    Cache.this.delete(entry, RemovalNotification.RemovalReason.INVALIDATED);
                }
            }
        }
    }
}

