/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.cluster.metadata;

import java.io.IOException;
import java.io.UnsupportedEncodingException;
import java.nio.file.Path;
import java.time.Instant;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.function.BiConsumer;
import java.util.function.BiFunction;
import java.util.function.Function;
import java.util.function.Predicate;
import java.util.function.Supplier;
import java.util.stream.Collectors;
import java.util.stream.IntStream;
import org.apache.logging.log4j.Level;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.apache.logging.log4j.message.ParameterizedMessage;
import org.elasticsearch.ElasticsearchException;
import org.elasticsearch.ResourceAlreadyExistsException;
import org.elasticsearch.Version;
import org.elasticsearch.action.ActionListener;
import org.elasticsearch.action.admin.indices.alias.Alias;
import org.elasticsearch.action.admin.indices.create.CreateIndexClusterStateUpdateRequest;
import org.elasticsearch.action.admin.indices.shrink.ResizeType;
import org.elasticsearch.action.support.ActiveShardCount;
import org.elasticsearch.action.support.ActiveShardsObserver;
import org.elasticsearch.action.support.master.AcknowledgedResponse;
import org.elasticsearch.action.support.master.ShardsAcknowledgedResponse;
import org.elasticsearch.cluster.AckedClusterStateUpdateTask;
import org.elasticsearch.cluster.ClusterState;
import org.elasticsearch.cluster.block.ClusterBlock;
import org.elasticsearch.cluster.block.ClusterBlockLevel;
import org.elasticsearch.cluster.block.ClusterBlocks;
import org.elasticsearch.cluster.metadata.AliasMetadata;
import org.elasticsearch.cluster.metadata.AliasValidator;
import org.elasticsearch.cluster.metadata.ComponentTemplate;
import org.elasticsearch.cluster.metadata.ComposableIndexTemplate;
import org.elasticsearch.cluster.metadata.IndexAbstraction;
import org.elasticsearch.cluster.metadata.IndexMetadata;
import org.elasticsearch.cluster.metadata.IndexTemplateMetadata;
import org.elasticsearch.cluster.metadata.MappingMetadata;
import org.elasticsearch.cluster.metadata.Metadata;
import org.elasticsearch.cluster.metadata.MetadataCreateDataStreamService;
import org.elasticsearch.cluster.metadata.MetadataIndexTemplateService;
import org.elasticsearch.cluster.node.DiscoveryNodes;
import org.elasticsearch.cluster.routing.IndexRoutingTable;
import org.elasticsearch.cluster.routing.RoutingTable;
import org.elasticsearch.cluster.routing.ShardRouting;
import org.elasticsearch.cluster.routing.ShardRoutingState;
import org.elasticsearch.cluster.routing.allocation.AllocationService;
import org.elasticsearch.cluster.routing.allocation.DataTier;
import org.elasticsearch.cluster.service.ClusterService;
import org.elasticsearch.common.Priority;
import org.elasticsearch.common.Strings;
import org.elasticsearch.common.UUIDs;
import org.elasticsearch.common.ValidationException;
import org.elasticsearch.common.compress.CompressedXContent;
import org.elasticsearch.common.logging.DeprecationCategory;
import org.elasticsearch.common.logging.DeprecationLogger;
import org.elasticsearch.common.settings.IndexScopedSettings;
import org.elasticsearch.common.settings.Setting;
import org.elasticsearch.common.settings.Settings;
import org.elasticsearch.common.xcontent.XContentHelper;
import org.elasticsearch.core.Nullable;
import org.elasticsearch.core.PathUtils;
import org.elasticsearch.env.Environment;
import org.elasticsearch.index.Index;
import org.elasticsearch.index.IndexModule;
import org.elasticsearch.index.IndexNotFoundException;
import org.elasticsearch.index.IndexService;
import org.elasticsearch.index.IndexSettings;
import org.elasticsearch.index.mapper.DocumentMapper;
import org.elasticsearch.index.mapper.MapperService;
import org.elasticsearch.index.query.SearchExecutionContext;
import org.elasticsearch.index.shard.IndexSettingProvider;
import org.elasticsearch.indices.IndexCreationException;
import org.elasticsearch.indices.IndicesService;
import org.elasticsearch.indices.InvalidIndexNameException;
import org.elasticsearch.indices.ShardLimitValidator;
import org.elasticsearch.indices.SystemIndices;
import org.elasticsearch.snapshots.SearchableSnapshotsSettings;
import org.elasticsearch.threadpool.ThreadPool;
import org.elasticsearch.xcontent.NamedXContentRegistry;

public class MetadataCreateIndexService {
    private static final Logger logger = LogManager.getLogger(MetadataCreateIndexService.class);
    private static final DeprecationLogger deprecationLogger = DeprecationLogger.getLogger(MetadataCreateIndexService.class);
    public static final int MAX_INDEX_NAME_BYTES = 255;
    private final Settings settings;
    private final ClusterService clusterService;
    private final IndicesService indicesService;
    private final AllocationService allocationService;
    private final AliasValidator aliasValidator;
    private final Environment env;
    private final IndexScopedSettings indexScopedSettings;
    private final ActiveShardsObserver activeShardsObserver;
    private final NamedXContentRegistry xContentRegistry;
    private final SystemIndices systemIndices;
    private final ShardLimitValidator shardLimitValidator;
    private final boolean forbidPrivateIndexSettings;
    private final Set<IndexSettingProvider> indexSettingProviders = new HashSet<IndexSettingProvider>();

    public MetadataCreateIndexService(Settings settings, ClusterService clusterService, IndicesService indicesService, AllocationService allocationService, AliasValidator aliasValidator, ShardLimitValidator shardLimitValidator, Environment env, IndexScopedSettings indexScopedSettings, ThreadPool threadPool, NamedXContentRegistry xContentRegistry, SystemIndices systemIndices, boolean forbidPrivateIndexSettings) {
        this.settings = settings;
        this.clusterService = clusterService;
        this.indicesService = indicesService;
        this.allocationService = allocationService;
        this.aliasValidator = aliasValidator;
        this.env = env;
        this.indexScopedSettings = indexScopedSettings;
        this.activeShardsObserver = new ActiveShardsObserver(clusterService, threadPool);
        this.xContentRegistry = xContentRegistry;
        this.systemIndices = systemIndices;
        this.forbidPrivateIndexSettings = forbidPrivateIndexSettings;
        this.shardLimitValidator = shardLimitValidator;
    }

    public void addAdditionalIndexSettingProvider(IndexSettingProvider provider) {
        if (provider == null) {
            throw new IllegalArgumentException("provider must not be null");
        }
        if (this.indexSettingProviders.contains(provider)) {
            throw new IllegalArgumentException("provider already added");
        }
        this.indexSettingProviders.add(provider);
    }

    public void validateIndexName(String index, ClusterState state) {
        MetadataCreateIndexService.validateIndexOrAliasName(index, InvalidIndexNameException::new);
        if (!index.toLowerCase(Locale.ROOT).equals(index)) {
            throw new InvalidIndexNameException(index, "must be lowercase");
        }
        if (state.routingTable().hasIndex(index)) {
            throw new ResourceAlreadyExistsException(state.routingTable().index(index).getIndex());
        }
        if (state.metadata().hasIndex(index)) {
            throw new ResourceAlreadyExistsException(state.metadata().index(index).getIndex());
        }
        if (state.metadata().hasAlias(index)) {
            throw new InvalidIndexNameException(index, "already exists as alias");
        }
    }

    public boolean validateDotIndex(String index, @Nullable Boolean isHidden) {
        boolean isSystem = false;
        if (index.charAt(0) == '.') {
            isSystem = this.systemIndices.isSystemName(index);
            if (isSystem) {
                logger.trace("index [{}] is a system index", (Object)index);
            } else if (isHidden.booleanValue()) {
                logger.trace("index [{}] is a hidden index", (Object)index);
            } else {
                deprecationLogger.critical(DeprecationCategory.INDICES, "index_name_starts_with_dot", "index name [{}] starts with a dot '.', in the next major version, index names starting with a dot are reserved for hidden indices and system indices", index);
            }
        }
        return isSystem;
    }

    public SystemIndices getSystemIndices() {
        return this.systemIndices;
    }

    public static void validateIndexOrAliasName(String index, BiFunction<String, String, ? extends RuntimeException> exceptionCtor) {
        if (!Strings.validFileName(index)) {
            throw exceptionCtor.apply(index, "must not contain the following characters " + Strings.INVALID_FILENAME_CHARS);
        }
        if (index.contains("#")) {
            throw exceptionCtor.apply(index, "must not contain '#'");
        }
        if (index.contains(":")) {
            throw exceptionCtor.apply(index, "must not contain ':'");
        }
        if (index.charAt(0) == '_' || index.charAt(0) == '-' || index.charAt(0) == '+') {
            throw exceptionCtor.apply(index, "must not start with '_', '-', or '+'");
        }
        int byteCount = 0;
        try {
            byteCount = index.getBytes("UTF-8").length;
        }
        catch (UnsupportedEncodingException e) {
            throw new ElasticsearchException("Unable to determine length of index name", (Throwable)e, new Object[0]);
        }
        if (byteCount > 255) {
            throw exceptionCtor.apply(index, "index name is too long, (" + byteCount + " > 255)");
        }
        if (index.equals(".") || index.equals("..")) {
            throw exceptionCtor.apply(index, "must not be '.' or '..'");
        }
    }

    public void createIndex(CreateIndexClusterStateUpdateRequest request, ActionListener<ShardsAcknowledgedResponse> listener) {
        logger.trace("createIndex[{}]", (Object)request);
        this.onlyCreateIndex(request, ActionListener.wrap(response -> {
            if (response.isAcknowledged()) {
                logger.trace("[{}] index creation acknowledged, waiting for active shards [{}]", (Object)request.index(), (Object)request.waitForActiveShards());
                this.activeShardsObserver.waitForActiveShards(new String[]{request.index()}, request.waitForActiveShards(), request.ackTimeout(), shardsAcknowledged -> {
                    if (!shardsAcknowledged.booleanValue()) {
                        logger.debug("[{}] index created, but the operation timed out while waiting for enough shards to be started.", (Object)request.index());
                    } else {
                        logger.trace("[{}] index created and shards acknowledged", (Object)request.index());
                    }
                    listener.onResponse(ShardsAcknowledgedResponse.of(true, shardsAcknowledged));
                }, listener::onFailure);
            } else {
                logger.trace("index creation not acknowledged for [{}]", (Object)request);
                listener.onResponse(ShardsAcknowledgedResponse.NOT_ACKNOWLEDGED);
            }
        }, listener::onFailure));
    }

    private void onlyCreateIndex(final CreateIndexClusterStateUpdateRequest request, ActionListener<AcknowledgedResponse> listener) {
        this.normalizeRequestSetting(request);
        this.clusterService.submitStateUpdateTask("create-index [" + request.index() + "], cause [" + request.cause() + "]", new AckedClusterStateUpdateTask(Priority.URGENT, request, listener){

            @Override
            public ClusterState execute(ClusterState currentState) throws Exception {
                return MetadataCreateIndexService.this.applyCreateIndexRequest(currentState, request, false);
            }

            @Override
            public void onFailure(String source, Exception e) {
                if (e instanceof ResourceAlreadyExistsException) {
                    logger.trace(() -> new ParameterizedMessage("[{}] failed to create", (Object)request.index()), (Throwable)e);
                } else {
                    logger.debug(() -> new ParameterizedMessage("[{}] failed to create", (Object)request.index()), (Throwable)e);
                }
                super.onFailure(source, e);
            }
        });
    }

    private void normalizeRequestSetting(CreateIndexClusterStateUpdateRequest createIndexClusterStateRequest) {
        Settings.Builder updatedSettingsBuilder = Settings.builder();
        Settings build = updatedSettingsBuilder.put(createIndexClusterStateRequest.settings()).normalizePrefix("index.").build();
        this.indexScopedSettings.validate(build, true);
        createIndexClusterStateRequest.settings(build);
    }

    public ClusterState applyCreateIndexRequest(ClusterState currentState, CreateIndexClusterStateUpdateRequest request, boolean silent, BiConsumer<Metadata.Builder, IndexMetadata> metadataTransformer) throws Exception {
        String name;
        IndexMetadata sourceMetadata;
        this.normalizeRequestSetting(request);
        logger.trace("executing IndexCreationTask for [{}] against cluster state version [{}]", (Object)request, (Object)currentState.version());
        this.validate(request, currentState);
        Index recoverFromIndex = request.recoverFrom();
        IndexMetadata indexMetadata = sourceMetadata = recoverFromIndex == null ? null : currentState.metadata().getIndexSafe(recoverFromIndex);
        if (sourceMetadata != null) {
            return this.applyCreateIndexRequestWithExistingMetadata(currentState, request, silent, sourceMetadata, metadataTransformer);
        }
        String string = name = request.dataStreamName() != null ? request.dataStreamName() : request.index();
        if (request.systemDataStreamDescriptor() != null) {
            return this.applyCreateIndexRequestForSystemDataStream(currentState, request, silent, metadataTransformer);
        }
        Boolean isHiddenFromRequest = IndexMetadata.INDEX_HIDDEN_SETTING.exists(request.settings()) ? IndexMetadata.INDEX_HIDDEN_SETTING.get(request.settings()) : null;
        String v2Template = MetadataIndexTemplateService.findV2Template(currentState.metadata(), name, isHiddenFromRequest == null ? false : isHiddenFromRequest);
        if (v2Template != null) {
            return this.applyCreateIndexRequestWithV2Template(currentState, request, silent, v2Template, metadataTransformer);
        }
        List<IndexTemplateMetadata> v1Templates = MetadataIndexTemplateService.findV1Templates(currentState.metadata(), request.index(), isHiddenFromRequest);
        if (v1Templates.size() > 1) {
            deprecationLogger.critical(DeprecationCategory.TEMPLATES, "index_template_multiple_match", "index [{}] matches multiple legacy templates [{}], composable templates will only match a single template", request.index(), v1Templates.stream().map(IndexTemplateMetadata::name).sorted().collect(Collectors.joining(", ")));
        }
        return this.applyCreateIndexRequestWithV1Templates(currentState, request, silent, v1Templates, metadataTransformer);
    }

    public ClusterState applyCreateIndexRequest(ClusterState currentState, CreateIndexClusterStateUpdateRequest request, boolean silent) throws Exception {
        return this.applyCreateIndexRequest(currentState, request, silent, null);
    }

    private ClusterState applyCreateIndexWithTemporaryService(ClusterState currentState, CreateIndexClusterStateUpdateRequest request, boolean silent, IndexMetadata sourceMetadata, IndexMetadata temporaryIndexMeta, List<Map<String, Object>> mappings, Function<IndexService, List<AliasMetadata>> aliasSupplier, List<String> templatesApplied, BiConsumer<Metadata.Builder, IndexMetadata> metadataTransformer) throws Exception {
        return (ClusterState)this.indicesService.withTempIndexService(temporaryIndexMeta, indexService -> {
            IndexMetadata indexMetadata;
            try {
                MetadataCreateIndexService.updateIndexMappingsAndBuildSortOrder(indexService, request, mappings, sourceMetadata);
            }
            catch (Exception e) {
                logger.log(silent ? Level.DEBUG : Level.INFO, "failed on parsing mappings on index creation [{}]", (Object)request.index(), (Object)e);
                throw e;
            }
            List aliases = (List)aliasSupplier.apply((IndexService)indexService);
            try {
                indexMetadata = MetadataCreateIndexService.buildIndexMetadata(request.index(), aliases, indexService.mapperService()::documentMapper, temporaryIndexMeta.getSettings(), temporaryIndexMeta.getRoutingNumShards(), sourceMetadata, temporaryIndexMeta.isSystem());
            }
            catch (Exception e) {
                logger.info("failed to build index metadata [{}]", (Object)request.index());
                throw e;
            }
            logger.log(silent ? Level.DEBUG : Level.INFO, "[{}] creating index, cause [{}], templates {}, shards [{}]/[{}]", (Object)request.index(), (Object)request.cause(), (Object)templatesApplied, (Object)indexMetadata.getNumberOfShards(), (Object)indexMetadata.getNumberOfReplicas());
            indexService.getIndexEventListener().beforeIndexAddedToCluster(indexMetadata.getIndex(), indexMetadata.getSettings());
            return MetadataCreateIndexService.clusterStateCreateIndex(currentState, request.blocks(), indexMetadata, this.allocationService::reroute, metadataTransformer);
        });
    }

    private IndexMetadata buildAndValidateTemporaryIndexMetadata(Settings aggregatedIndexSettings, CreateIndexClusterStateUpdateRequest request, int routingNumShards) {
        boolean isHiddenAfterTemplates = IndexMetadata.INDEX_HIDDEN_SETTING.get(aggregatedIndexSettings);
        boolean isSystem = this.validateDotIndex(request.index(), isHiddenAfterTemplates);
        Settings.Builder settingsBuilder = Settings.builder().put(aggregatedIndexSettings);
        settingsBuilder.remove(IndexMetadata.INDEX_NUMBER_OF_ROUTING_SHARDS_SETTING.getKey());
        Settings indexSettings = settingsBuilder.build();
        IndexMetadata.Builder tmpImdBuilder = IndexMetadata.builder(request.index());
        tmpImdBuilder.setRoutingNumShards(routingNumShards);
        tmpImdBuilder.settings(indexSettings);
        tmpImdBuilder.system(isSystem);
        IndexMetadata tempMetadata = tmpImdBuilder.build();
        MetadataCreateIndexService.validateActiveShardCount(request.waitForActiveShards(), tempMetadata);
        return tempMetadata;
    }

    private ClusterState applyCreateIndexRequestWithV1Templates(ClusterState currentState, CreateIndexClusterStateUpdateRequest request, boolean silent, List<IndexTemplateMetadata> templates, BiConsumer<Metadata.Builder, IndexMetadata> metadataTransformer) throws Exception {
        logger.debug("applying create index request using legacy templates {}", templates.stream().map(IndexTemplateMetadata::name).collect(Collectors.toList()));
        Map<String, Object> mappings = Collections.unmodifiableMap(MetadataCreateIndexService.parseV1Mappings(request.mappings(), templates.stream().map(IndexTemplateMetadata::getMappings).collect(Collectors.toList()), this.xContentRegistry));
        Settings aggregatedIndexSettings = MetadataCreateIndexService.aggregateIndexSettings(currentState, request, MetadataIndexTemplateService.resolveSettings(templates), null, this.settings, this.indexScopedSettings, this.shardLimitValidator, this.indexSettingProviders);
        int routingNumShards = MetadataCreateIndexService.getIndexNumberOfRoutingShards(aggregatedIndexSettings, null);
        IndexMetadata tmpImd = this.buildAndValidateTemporaryIndexMetadata(aggregatedIndexSettings, request, routingNumShards);
        return this.applyCreateIndexWithTemporaryService(currentState, request, silent, null, tmpImd, List.of(mappings), indexService -> MetadataCreateIndexService.resolveAndValidateAliases(request.index(), request.aliases(), MetadataIndexTemplateService.resolveAliases(templates), currentState.metadata(), this.aliasValidator, this.xContentRegistry, indexService.newSearchExecutionContext(0, 0, null, () -> 0L, null, Collections.emptyMap()), indexService.dateMathExpressionResolverAt(request.getNameResolvedAt())), templates.stream().map(IndexTemplateMetadata::getName).collect(Collectors.toList()), metadataTransformer);
    }

    private ClusterState applyCreateIndexRequestWithV2Template(ClusterState currentState, CreateIndexClusterStateUpdateRequest request, boolean silent, String templateName, BiConsumer<Metadata.Builder, IndexMetadata> metadataTransformer) throws Exception {
        boolean isDataStream;
        logger.debug("applying create index request using composable template [{}]", (Object)templateName);
        ComposableIndexTemplate template = currentState.getMetadata().templatesV2().get(templateName);
        boolean bl = isDataStream = template.getDataStreamTemplate() != null;
        if (isDataStream && request.dataStreamName() == null) {
            throw new IllegalArgumentException("cannot create index with name [" + request.index() + "], because it matches with template [" + templateName + "] that creates data streams only, use create data stream api instead");
        }
        List<Map<String, Object>> mappings = MetadataCreateIndexService.collectV2Mappings(request.mappings(), currentState, templateName, this.xContentRegistry, request.index());
        Settings aggregatedIndexSettings = MetadataCreateIndexService.aggregateIndexSettings(currentState, request, MetadataIndexTemplateService.resolveSettings(currentState.metadata(), templateName), null, this.settings, this.indexScopedSettings, this.shardLimitValidator, this.indexSettingProviders);
        int routingNumShards = MetadataCreateIndexService.getIndexNumberOfRoutingShards(aggregatedIndexSettings, null);
        IndexMetadata tmpImd = this.buildAndValidateTemporaryIndexMetadata(aggregatedIndexSettings, request, routingNumShards);
        return this.applyCreateIndexWithTemporaryService(currentState, request, silent, null, tmpImd, mappings, indexService -> MetadataCreateIndexService.resolveAndValidateAliases(request.index(), isDataStream ? Set.of() : request.aliases(), isDataStream ? List.of() : MetadataIndexTemplateService.resolveAliases(currentState.metadata(), templateName), currentState.metadata(), this.aliasValidator, this.xContentRegistry, indexService.newSearchExecutionContext(0, 0, null, () -> 0L, null, Collections.emptyMap()), indexService.dateMathExpressionResolverAt(request.getNameResolvedAt())), Collections.singletonList(templateName), metadataTransformer);
    }

    private ClusterState applyCreateIndexRequestForSystemDataStream(ClusterState currentState, CreateIndexClusterStateUpdateRequest request, boolean silent, BiConsumer<Metadata.Builder, IndexMetadata> metadataTransformer) throws Exception {
        Objects.requireNonNull(request.systemDataStreamDescriptor());
        logger.debug("applying create index request for system data stream [{}]", (Object)request.systemDataStreamDescriptor());
        ComposableIndexTemplate template = request.systemDataStreamDescriptor().getComposableIndexTemplate();
        if (request.dataStreamName() == null && template.getDataStreamTemplate() != null) {
            throw new IllegalArgumentException("cannot create index with name [" + request.index() + "], because it matches with a system data stream");
        }
        Map<String, ComponentTemplate> componentTemplates = request.systemDataStreamDescriptor().getComponentTemplates();
        List<Map<String, Object>> mappings = MetadataCreateIndexService.collectSystemV2Mappings(template, componentTemplates, this.xContentRegistry, request.index());
        Settings aggregatedIndexSettings = MetadataCreateIndexService.aggregateIndexSettings(currentState, request, MetadataIndexTemplateService.resolveSettings(template, componentTemplates), null, this.settings, this.indexScopedSettings, this.shardLimitValidator, this.indexSettingProviders);
        int routingNumShards = MetadataCreateIndexService.getIndexNumberOfRoutingShards(aggregatedIndexSettings, null);
        IndexMetadata tmpImd = this.buildAndValidateTemporaryIndexMetadata(aggregatedIndexSettings, request, routingNumShards);
        return this.applyCreateIndexWithTemporaryService(currentState, request, silent, null, tmpImd, mappings, indexService -> MetadataCreateIndexService.resolveAndValidateAliases(request.index(), request.aliases(), MetadataIndexTemplateService.resolveAliases(template, componentTemplates), currentState.metadata(), this.aliasValidator, this.xContentRegistry, indexService.newSearchExecutionContext(0, 0, null, () -> 0L, null, Collections.emptyMap()), indexService.dateMathExpressionResolverAt(request.getNameResolvedAt())), List.of(), metadataTransformer);
    }

    private static List<Map<String, Object>> collectSystemV2Mappings(ComposableIndexTemplate composableIndexTemplate, Map<String, ComponentTemplate> componentTemplates, NamedXContentRegistry xContentRegistry, String indexName) throws Exception {
        List<CompressedXContent> templateMappings = MetadataIndexTemplateService.collectMappings(composableIndexTemplate, componentTemplates, indexName, xContentRegistry);
        return MetadataCreateIndexService.collectV2Mappings("{}", templateMappings, xContentRegistry);
    }

    public static List<Map<String, Object>> collectV2Mappings(String requestMappings, ClusterState currentState, String templateName, NamedXContentRegistry xContentRegistry, String indexName) throws Exception {
        List<CompressedXContent> templateMappings = MetadataIndexTemplateService.collectMappings(currentState, templateName, indexName, xContentRegistry);
        return MetadataCreateIndexService.collectV2Mappings(requestMappings, templateMappings, xContentRegistry);
    }

    public static List<Map<String, Object>> collectV2Mappings(String requestMappings, List<CompressedXContent> templateMappings, NamedXContentRegistry xContentRegistry) throws Exception {
        ArrayList<Map<String, Object>> result = new ArrayList<Map<String, Object>>();
        for (CompressedXContent templateMapping : templateMappings) {
            Map<String, Object> parsedTemplateMapping = MapperService.parseMapping(xContentRegistry, templateMapping.string());
            result.add(parsedTemplateMapping);
        }
        Map<String, Object> parsedRequestMappings = MapperService.parseMapping(xContentRegistry, requestMappings);
        result.add(parsedRequestMappings);
        return result;
    }

    private ClusterState applyCreateIndexRequestWithExistingMetadata(ClusterState currentState, CreateIndexClusterStateUpdateRequest request, boolean silent, IndexMetadata sourceMetadata, BiConsumer<Metadata.Builder, IndexMetadata> metadataTransformer) throws Exception {
        logger.info("applying create index request using existing index [{}] metadata", (Object)sourceMetadata.getIndex().getName());
        Map<String, Object> mappings = MapperService.parseMapping(this.xContentRegistry, request.mappings());
        if (!mappings.isEmpty()) {
            throw new IllegalArgumentException("mappings are not allowed when creating an index from a source index, all mappings are copied from the source index");
        }
        Settings aggregatedIndexSettings = MetadataCreateIndexService.aggregateIndexSettings(currentState, request, Settings.EMPTY, sourceMetadata, this.settings, this.indexScopedSettings, this.shardLimitValidator, this.indexSettingProviders);
        int routingNumShards = MetadataCreateIndexService.getIndexNumberOfRoutingShards(aggregatedIndexSettings, sourceMetadata);
        IndexMetadata tmpImd = this.buildAndValidateTemporaryIndexMetadata(aggregatedIndexSettings, request, routingNumShards);
        return this.applyCreateIndexWithTemporaryService(currentState, request, silent, sourceMetadata, tmpImd, List.of(mappings), indexService -> MetadataCreateIndexService.resolveAndValidateAliases(request.index(), request.aliases(), Collections.emptyList(), currentState.metadata(), this.aliasValidator, this.xContentRegistry, indexService.newSearchExecutionContext(0, 0, null, () -> 0L, null, Collections.emptyMap()), indexService.dateMathExpressionResolverAt(request.getNameResolvedAt())), List.of(), metadataTransformer);
    }

    static Map<String, Object> parseV1Mappings(String mappingsJson, List<CompressedXContent> templateMappings, NamedXContentRegistry xContentRegistry) throws Exception {
        Map<String, Object> mappings = MapperService.parseMapping(xContentRegistry, mappingsJson);
        for (CompressedXContent mapping : templateMappings) {
            Map<String, Object> templateMapping;
            if (mapping == null || (templateMapping = MapperService.parseMapping(xContentRegistry, mapping.string())).isEmpty()) continue;
            assert (templateMapping.size() == 1) : "expected exactly one mapping value, got: " + templateMapping;
            templateMapping = Collections.singletonMap("_doc", templateMapping.values().iterator().next());
            if (mappings.isEmpty()) {
                mappings = templateMapping;
                continue;
            }
            XContentHelper.mergeDefaults(mappings, templateMapping);
        }
        return mappings;
    }

    static Settings aggregateIndexSettings(ClusterState currentState, CreateIndexClusterStateUpdateRequest request, Settings combinedTemplateSettings, @Nullable IndexMetadata sourceMetadata, Settings settings, IndexScopedSettings indexScopedSettings, ShardLimitValidator shardLimitValidator, Set<IndexSettingProvider> indexSettingProviders) {
        String currentTierPreference;
        boolean isDataStreamIndex = request.dataStreamName() != null;
        Settings.Builder templateSettings = Settings.builder().put(combinedTemplateSettings);
        Settings.Builder requestSettings = Settings.builder().put(request.settings());
        Settings.Builder indexSettingsBuilder = Settings.builder();
        if (sourceMetadata == null) {
            Settings.Builder additionalIndexSettings = Settings.builder();
            Settings templateAndRequestSettings = Settings.builder().put(combinedTemplateSettings).put(request.settings()).build();
            for (IndexSettingProvider provider : indexSettingProviders) {
                additionalIndexSettings.put(provider.getAdditionalIndexSettings(request.index(), isDataStreamIndex, templateAndRequestSettings));
            }
            for (String explicitSetting : additionalIndexSettings.keys()) {
                if (templateSettings.keys().contains(explicitSetting) && templateSettings.get(explicitSetting) == null) {
                    logger.debug("removing default [{}] setting as it in set to null in a template for [{}] creation", (Object)explicitSetting, (Object)request.index());
                    additionalIndexSettings.remove(explicitSetting);
                    templateSettings.remove(explicitSetting);
                }
                if (!requestSettings.keys().contains(explicitSetting) || requestSettings.get(explicitSetting) != null) continue;
                logger.debug("removing default [{}] setting as it in set to null in the request for [{}] creation", (Object)explicitSetting, (Object)request.index());
                additionalIndexSettings.remove(explicitSetting);
                requestSettings.remove(explicitSetting);
            }
            indexSettingsBuilder.put(additionalIndexSettings.build());
            indexSettingsBuilder.put(templateSettings.build());
        }
        indexSettingsBuilder.put(requestSettings.build());
        if (sourceMetadata == null && DataTier.parseTierList(currentTierPreference = indexSettingsBuilder.get("index.routing.allocation.include._tier_preference")).isEmpty()) {
            String newTierPreference = isDataStreamIndex ? "data_hot" : "data_content";
            logger.debug("enforcing default [{}] setting for [{}] creation, replacing [{}] with [{}]", (Object)"index.routing.allocation.include._tier_preference", (Object)request.index(), (Object)currentTierPreference, (Object)newTierPreference);
            indexSettingsBuilder.put("index.routing.allocation.include._tier_preference", newTierPreference);
        }
        if (indexSettingsBuilder.get("index.version.created") == null) {
            DiscoveryNodes nodes = currentState.nodes();
            Version createdVersion = Version.min(Version.CURRENT, nodes.getSmallestNonClientNodeVersion());
            indexSettingsBuilder.put("index.version.created", createdVersion);
        }
        if (!IndexMetadata.INDEX_NUMBER_OF_SHARDS_SETTING.exists(indexSettingsBuilder)) {
            indexSettingsBuilder.put("index.number_of_shards", IndexMetadata.INDEX_NUMBER_OF_SHARDS_SETTING.get(settings));
        }
        if (!IndexMetadata.INDEX_NUMBER_OF_REPLICAS_SETTING.exists(indexSettingsBuilder)) {
            indexSettingsBuilder.put("index.number_of_replicas", IndexMetadata.INDEX_NUMBER_OF_REPLICAS_SETTING.get(settings));
        }
        if (settings.get("index.auto_expand_replicas") != null && indexSettingsBuilder.get("index.auto_expand_replicas") == null) {
            indexSettingsBuilder.put("index.auto_expand_replicas", settings.get("index.auto_expand_replicas"));
        }
        if (indexSettingsBuilder.get("index.creation_date") == null) {
            indexSettingsBuilder.put("index.creation_date", Instant.now().toEpochMilli());
        }
        indexSettingsBuilder.put("index.provided_name", request.getProvidedName());
        indexSettingsBuilder.put("index.uuid", UUIDs.randomBase64UUID());
        if (sourceMetadata != null) {
            assert (request.resizeType() != null);
            MetadataCreateIndexService.prepareResizeIndexSettings(currentState, indexSettingsBuilder, request.recoverFrom(), request.index(), request.resizeType(), request.copySettings(), indexScopedSettings);
        }
        Settings indexSettings = indexSettingsBuilder.build();
        shardLimitValidator.validateShardLimit(indexSettings, currentState);
        MetadataCreateIndexService.validateSoftDeleteSettings(indexSettings);
        MetadataCreateIndexService.validateTranslogRetentionSettings(indexSettings);
        MetadataCreateIndexService.validateStoreTypeSetting(indexSettings);
        return indexSettings;
    }

    private static void validateSoftDeleteSettings(Settings indexSettings) {
        if (!IndexSettings.INDEX_SOFT_DELETES_SETTING.get(indexSettings).booleanValue() && IndexMetadata.SETTING_INDEX_VERSION_CREATED.get(indexSettings).onOrAfter(Version.V_8_0_0)) {
            throw new IllegalArgumentException("Creating indices with soft-deletes disabled is no longer supported. Please do not specify a value for setting [index.soft_deletes.enabled].");
        }
    }

    static int getIndexNumberOfRoutingShards(Settings indexSettings, @Nullable IndexMetadata sourceMetadata) {
        int routingNumShards;
        int numTargetShards = IndexMetadata.INDEX_NUMBER_OF_SHARDS_SETTING.get(indexSettings);
        Version indexVersionCreated = IndexMetadata.SETTING_INDEX_VERSION_CREATED.get(indexSettings);
        if (sourceMetadata == null || sourceMetadata.getNumberOfShards() == 1) {
            routingNumShards = indexSettings.get(IndexMetadata.INDEX_NUMBER_OF_ROUTING_SHARDS_SETTING.getKey()) != null ? IndexMetadata.INDEX_NUMBER_OF_ROUTING_SHARDS_SETTING.get(indexSettings) : MetadataCreateIndexService.calculateNumRoutingShards(numTargetShards, indexVersionCreated);
        } else {
            assert (!IndexMetadata.INDEX_NUMBER_OF_ROUTING_SHARDS_SETTING.exists(indexSettings)) : "index.number_of_routing_shards should not be present on the target index on resize";
            routingNumShards = sourceMetadata.getRoutingNumShards();
        }
        return routingNumShards;
    }

    public static List<AliasMetadata> resolveAndValidateAliases(String index, Set<Alias> aliases, List<Map<String, AliasMetadata>> templateAliases, Metadata metadata, AliasValidator aliasValidator, NamedXContentRegistry xContentRegistry, SearchExecutionContext searchExecutionContext, Function<String, String> indexNameExpressionResolver) {
        HashSet<Alias> resolvedExpressions = new HashSet<Alias>();
        ArrayList<AliasMetadata> resolvedAliases = new ArrayList<AliasMetadata>();
        for (Alias alias : aliases) {
            String resolvedExpression = indexNameExpressionResolver.apply(alias.name());
            alias = alias.name(resolvedExpression);
            aliasValidator.validateAlias(alias, index, metadata);
            if (Strings.hasLength(alias.filter())) {
                aliasValidator.validateAliasFilter(alias.name(), alias.filter(), searchExecutionContext, xContentRegistry);
            }
            AliasMetadata aliasMetadata = AliasMetadata.builder(alias.name()).filter(alias.filter()).indexRouting(alias.indexRouting()).searchRouting(alias.searchRouting()).writeIndex(alias.writeIndex()).isHidden(alias.isHidden()).build();
            resolvedAliases.add(aliasMetadata);
            resolvedExpressions.add(new Alias(resolvedExpression));
        }
        HashMap<String, AliasMetadata> templatesAliases = new HashMap<String, AliasMetadata>();
        for (Map<String, AliasMetadata> templateAliasConfig : templateAliases) {
            for (Map.Entry<String, AliasMetadata> entry : templateAliasConfig.entrySet()) {
                String resolvedTemplateExpression = indexNameExpressionResolver.apply(entry.getValue().alias());
                AliasMetadata aliasMetadata = AliasMetadata.newAliasMetadata(entry.getValue(), resolvedTemplateExpression);
                if (resolvedExpressions.contains(new Alias(aliasMetadata.alias())) || templatesAliases.containsKey(entry.getKey())) continue;
                if (aliasMetadata.alias().contains("{index}")) {
                    String templatedAlias = aliasMetadata.alias().replace("{index}", index);
                    aliasMetadata = AliasMetadata.newAliasMetadata(aliasMetadata, templatedAlias);
                }
                aliasValidator.validateAliasMetadata(aliasMetadata, index, metadata);
                if (aliasMetadata.filter() != null) {
                    aliasValidator.validateAliasFilter(aliasMetadata.alias(), aliasMetadata.filter().uncompressed(), searchExecutionContext, xContentRegistry);
                }
                templatesAliases.put(aliasMetadata.alias(), aliasMetadata);
                resolvedAliases.add(aliasMetadata);
            }
        }
        return resolvedAliases;
    }

    static ClusterState clusterStateCreateIndex(ClusterState currentState, Set<ClusterBlock> clusterBlocks, IndexMetadata indexMetadata, BiFunction<ClusterState, String, ClusterState> rerouteRoutingTable, BiConsumer<Metadata.Builder, IndexMetadata> metadataTransformer) {
        Metadata.Builder builder = Metadata.builder(currentState.metadata()).put(indexMetadata, false);
        if (metadataTransformer != null) {
            metadataTransformer.accept(builder, indexMetadata);
        }
        Metadata newMetadata = builder.build();
        String indexName = indexMetadata.getIndex().getName();
        ClusterBlocks.Builder blocks = MetadataCreateIndexService.createClusterBlocksBuilder(currentState, indexName, clusterBlocks);
        blocks.updateBlocks(indexMetadata);
        ClusterState updatedState = ClusterState.builder(currentState).blocks(blocks).metadata(newMetadata).build();
        RoutingTable.Builder routingTableBuilder = RoutingTable.builder(updatedState.routingTable()).addAsNew(updatedState.metadata().index(indexName));
        updatedState = ClusterState.builder(updatedState).routingTable(routingTableBuilder.build()).build();
        return rerouteRoutingTable.apply(updatedState, "index [" + indexName + "] created");
    }

    static IndexMetadata buildIndexMetadata(String indexName, List<AliasMetadata> aliases, Supplier<DocumentMapper> documentMapperSupplier, Settings indexSettings, int routingNumShards, @Nullable IndexMetadata sourceMetadata, boolean isSystem) {
        IndexMetadata.Builder indexMetadataBuilder = MetadataCreateIndexService.createIndexMetadataBuilder(indexName, sourceMetadata, indexSettings, routingNumShards);
        indexMetadataBuilder.system(isSystem);
        HashMap<String, Object> mappingsMetadata = new HashMap<String, Object>();
        DocumentMapper mapper = documentMapperSupplier.get();
        if (mapper != null) {
            MappingMetadata mappingMd = new MappingMetadata(mapper);
            mappingsMetadata.put(mapper.type(), mappingMd);
        }
        for (MappingMetadata mappingMd : mappingsMetadata.values()) {
            indexMetadataBuilder.putMapping(mappingMd);
        }
        for (int i = aliases.size() - 1; i >= 0; --i) {
            indexMetadataBuilder.putAlias(aliases.get(i));
        }
        indexMetadataBuilder.state(IndexMetadata.State.OPEN);
        return indexMetadataBuilder.build();
    }

    private static IndexMetadata.Builder createIndexMetadataBuilder(String indexName, @Nullable IndexMetadata sourceMetadata, Settings indexSettings, int routingNumShards) {
        IndexMetadata.Builder builder = IndexMetadata.builder(indexName);
        builder.setRoutingNumShards(routingNumShards);
        builder.settings(indexSettings);
        if (sourceMetadata != null) {
            long primaryTerm = IntStream.range(0, sourceMetadata.getNumberOfShards()).mapToLong(sourceMetadata::primaryTerm).max().getAsLong();
            for (int shardId = 0; shardId < builder.numberOfShards(); ++shardId) {
                builder.primaryTerm(shardId, primaryTerm);
            }
        }
        return builder;
    }

    private static ClusterBlocks.Builder createClusterBlocksBuilder(ClusterState currentState, String index, Set<ClusterBlock> blocks) {
        ClusterBlocks.Builder blocksBuilder = ClusterBlocks.builder().blocks(currentState.blocks());
        if (!blocks.isEmpty()) {
            for (ClusterBlock block : blocks) {
                blocksBuilder.addIndexBlock(index, block);
            }
        }
        return blocksBuilder;
    }

    private static void updateIndexMappingsAndBuildSortOrder(IndexService indexService, CreateIndexClusterStateUpdateRequest request, List<Map<String, Object>> mappings, @Nullable IndexMetadata sourceMetadata) throws IOException {
        MapperService mapperService = indexService.mapperService();
        for (Map<String, Object> mapping : mappings) {
            if (mapping.isEmpty()) continue;
            mapperService.merge("_doc", mapping, MapperService.MergeReason.INDEX_TEMPLATE);
        }
        if (sourceMetadata == null) {
            indexService.getIndexSortSupplier().get();
        }
        if (request.dataStreamName() != null) {
            MetadataCreateDataStreamService.validateTimestampFieldMapping(mapperService.mappingLookup());
        }
    }

    private static void validateActiveShardCount(ActiveShardCount waitForActiveShards, IndexMetadata indexMetadata) {
        if (waitForActiveShards == ActiveShardCount.DEFAULT) {
            waitForActiveShards = indexMetadata.getWaitForActiveShards();
        }
        if (!waitForActiveShards.validate(indexMetadata.getNumberOfReplicas())) {
            throw new IllegalArgumentException("invalid wait_for_active_shards[" + waitForActiveShards + "]: cannot be greater than number of shard copies [" + (indexMetadata.getNumberOfReplicas() + 1) + "]");
        }
    }

    private void validate(CreateIndexClusterStateUpdateRequest request, ClusterState state) {
        this.validateIndexName(request.index(), state);
        this.validateIndexSettings(request.index(), request.settings(), this.forbidPrivateIndexSettings);
    }

    public void validateIndexSettings(String indexName, Settings settings, boolean forbidPrivateIndexSettings) throws IndexCreationException {
        List<String> validationErrors = this.getIndexSettingsValidationErrors(settings, forbidPrivateIndexSettings);
        if (!validationErrors.isEmpty()) {
            ValidationException validationException = new ValidationException();
            validationException.addValidationErrors(validationErrors);
            throw new IndexCreationException(indexName, validationException);
        }
    }

    List<String> getIndexSettingsValidationErrors(Settings settings, boolean forbidPrivateIndexSettings) {
        List<String> validationErrors = MetadataCreateIndexService.validateIndexCustomPath(settings, this.env.sharedDataFile());
        if (forbidPrivateIndexSettings) {
            validationErrors.addAll(MetadataCreateIndexService.validatePrivateSettingsNotExplicitlySet(settings, this.indexScopedSettings));
        }
        return validationErrors;
    }

    private static List<String> validatePrivateSettingsNotExplicitlySet(Settings settings, IndexScopedSettings indexScopedSettings) {
        ArrayList<String> validationErrors = new ArrayList<String>();
        for (String key : settings.keySet()) {
            Setting<?> setting = indexScopedSettings.get(key);
            if (setting == null) {
                assert (indexScopedSettings.isPrivateSetting(key)) : "expected [" + key + "] to be private but it was not";
                continue;
            }
            if (!setting.isPrivateIndex()) continue;
            validationErrors.add("private index setting [" + key + "] can not be set explicitly");
        }
        return validationErrors;
    }

    private static List<String> validateIndexCustomPath(Settings settings, @Nullable Path sharedDataPath) {
        String customPath = IndexMetadata.INDEX_DATA_PATH_SETTING.get(settings);
        ArrayList<String> validationErrors = new ArrayList<String>();
        if (!Strings.isEmpty(customPath)) {
            if (sharedDataPath == null) {
                validationErrors.add("path.shared_data must be set in order to use custom data paths");
            } else {
                Path resolvedPath = PathUtils.get((Path[])new Path[]{sharedDataPath}, (String)customPath);
                if (resolvedPath == null) {
                    validationErrors.add("custom path [" + customPath + "] is not a sub-path of path.shared_data [" + sharedDataPath + "]");
                }
            }
        }
        return validationErrors;
    }

    static List<String> validateShrinkIndex(ClusterState state, String sourceIndex, String targetIndexName, Settings targetIndexSettings) {
        IndexMetadata sourceMetadata = MetadataCreateIndexService.validateResize(state, sourceIndex, targetIndexName, targetIndexSettings);
        if (SearchableSnapshotsSettings.isSearchableSnapshotStore(sourceMetadata.getSettings())) {
            throw new IllegalArgumentException("can't shrink searchable snapshot index [" + sourceIndex + "]");
        }
        assert (IndexMetadata.INDEX_NUMBER_OF_SHARDS_SETTING.exists(targetIndexSettings));
        IndexMetadata.selectShrinkShards(0, sourceMetadata, IndexMetadata.INDEX_NUMBER_OF_SHARDS_SETTING.get(targetIndexSettings));
        if (sourceMetadata.getNumberOfShards() == 1) {
            throw new IllegalArgumentException("can't shrink an index with only one shard");
        }
        IndexRoutingTable table = state.routingTable().index(sourceIndex);
        HashMap<String, AtomicInteger> nodesToNumRouting = new HashMap<String, AtomicInteger>();
        int numShards = sourceMetadata.getNumberOfShards();
        for (ShardRouting routing : table.shardsWithState(ShardRoutingState.STARTED)) {
            nodesToNumRouting.computeIfAbsent(routing.currentNodeId(), s -> new AtomicInteger(0)).incrementAndGet();
        }
        ArrayList<String> nodesToAllocateOn = new ArrayList<String>();
        for (Map.Entry entries : nodesToNumRouting.entrySet()) {
            int numAllocations = ((AtomicInteger)entries.getValue()).get();
            assert (numAllocations <= numShards) : "wait what? " + numAllocations + " is > than num shards " + numShards;
            if (numAllocations != numShards) continue;
            nodesToAllocateOn.add((String)entries.getKey());
        }
        if (nodesToAllocateOn.isEmpty()) {
            throw new IllegalStateException("index " + sourceIndex + " must have all shards allocated on the same node to shrink index");
        }
        return nodesToAllocateOn;
    }

    static void validateSplitIndex(ClusterState state, String sourceIndex, String targetIndexName, Settings targetIndexSettings) {
        IndexMetadata sourceMetadata = MetadataCreateIndexService.validateResize(state, sourceIndex, targetIndexName, targetIndexSettings);
        if (SearchableSnapshotsSettings.isSearchableSnapshotStore(sourceMetadata.getSettings())) {
            throw new IllegalArgumentException("can't split searchable snapshot index [" + sourceIndex + "]");
        }
        IndexMetadata.selectSplitShard(0, sourceMetadata, IndexMetadata.INDEX_NUMBER_OF_SHARDS_SETTING.get(targetIndexSettings));
    }

    static void validateCloneIndex(ClusterState state, String sourceIndex, String targetIndexName, Settings targetIndexSettings) {
        IndexMetadata sourceMetadata = MetadataCreateIndexService.validateResize(state, sourceIndex, targetIndexName, targetIndexSettings);
        if (SearchableSnapshotsSettings.isSearchableSnapshotStore(sourceMetadata.getSettings())) {
            for (Setting nonCloneableSetting : Arrays.asList(IndexModule.INDEX_STORE_TYPE_SETTING, IndexModule.INDEX_RECOVERY_TYPE_SETTING)) {
                if (nonCloneableSetting.exists(targetIndexSettings)) continue;
                throw new IllegalArgumentException("can't clone searchable snapshot index [" + sourceIndex + "]; setting [" + nonCloneableSetting.getKey() + "] should be overridden");
            }
        }
        IndexMetadata.selectCloneShard(0, sourceMetadata, IndexMetadata.INDEX_NUMBER_OF_SHARDS_SETTING.get(targetIndexSettings));
    }

    static IndexMetadata validateResize(ClusterState state, String sourceIndex, String targetIndexName, Settings targetIndexSettings) {
        if (state.metadata().hasIndex(targetIndexName)) {
            throw new ResourceAlreadyExistsException(state.metadata().index(targetIndexName).getIndex());
        }
        IndexMetadata sourceMetadata = state.metadata().index(sourceIndex);
        if (sourceMetadata == null) {
            throw new IndexNotFoundException(sourceIndex);
        }
        IndexAbstraction source = (IndexAbstraction)state.metadata().getIndicesLookup().get(sourceIndex);
        assert (source != null);
        if (source.getParentDataStream() != null && source.getParentDataStream().getWriteIndex().equals(sourceMetadata.getIndex())) {
            throw new IllegalArgumentException(String.format(Locale.ROOT, "cannot resize the write index [%s] for data stream [%s]", sourceIndex, source.getParentDataStream().getName()));
        }
        if (!state.blocks().indexBlocked(ClusterBlockLevel.WRITE, sourceIndex)) {
            throw new IllegalStateException("index " + sourceIndex + " must be read-only to resize index. use \"index.blocks.write=true\"");
        }
        if (IndexMetadata.INDEX_NUMBER_OF_SHARDS_SETTING.exists(targetIndexSettings)) {
            IndexMetadata.getRoutingFactor(sourceMetadata.getNumberOfShards(), IndexMetadata.INDEX_NUMBER_OF_SHARDS_SETTING.get(targetIndexSettings));
        }
        return sourceMetadata;
    }

    static void prepareResizeIndexSettings(ClusterState currentState, Settings.Builder indexSettingsBuilder, Index resizeSourceIndex, String resizeIntoName, ResizeType type, boolean copySettings, IndexScopedSettings indexScopedSettings) {
        String initialRecoveryIdFilter = IndexMetadata.INDEX_ROUTING_INITIAL_RECOVERY_GROUP_SETTING.getKey() + "_id";
        IndexMetadata sourceMetadata = currentState.metadata().index(resizeSourceIndex.getName());
        if (type == ResizeType.SHRINK) {
            List<String> nodesToAllocateOn = MetadataCreateIndexService.validateShrinkIndex(currentState, resizeSourceIndex.getName(), resizeIntoName, indexSettingsBuilder.build());
            indexSettingsBuilder.put(initialRecoveryIdFilter, Strings.arrayToCommaDelimitedString(nodesToAllocateOn.toArray()));
        } else if (type == ResizeType.SPLIT) {
            MetadataCreateIndexService.validateSplitIndex(currentState, resizeSourceIndex.getName(), resizeIntoName, indexSettingsBuilder.build());
            indexSettingsBuilder.putNull(initialRecoveryIdFilter);
        } else if (type == ResizeType.CLONE) {
            MetadataCreateIndexService.validateCloneIndex(currentState, resizeSourceIndex.getName(), resizeIntoName, indexSettingsBuilder.build());
            indexSettingsBuilder.putNull(initialRecoveryIdFilter);
        } else {
            throw new IllegalStateException("unknown resize type is " + type);
        }
        Settings.Builder builder = Settings.builder();
        if (copySettings) {
            for (String key : sourceMetadata.getSettings().keySet()) {
                Setting<?> setting = indexScopedSettings.get(key);
                if (setting == null) {
                    assert (indexScopedSettings.isPrivateSetting(key)) : key;
                } else if (setting.getProperties().contains((Object)Setting.Property.NotCopyableOnResize)) continue;
                if (indexSettingsBuilder.keys().contains(key)) continue;
                builder.copy(key, sourceMetadata.getSettings());
            }
        } else {
            Predicate<String> sourceSettingsPredicate = s -> (s.startsWith("index.similarity.") || s.startsWith("index.analysis.") || s.startsWith("index.sort.") || s.equals("index.soft_deletes.enabled")) && !indexSettingsBuilder.keys().contains(s);
            builder.put(sourceMetadata.getSettings().filter(sourceSettingsPredicate));
        }
        indexSettingsBuilder.put("index.version.created", sourceMetadata.getCreationVersion()).put(builder.build()).put("index.routing_partition_size", sourceMetadata.getRoutingPartitionSize()).put(IndexMetadata.INDEX_RESIZE_SOURCE_NAME.getKey(), resizeSourceIndex.getName()).put(IndexMetadata.INDEX_RESIZE_SOURCE_UUID.getKey(), resizeSourceIndex.getUUID());
    }

    public static int calculateNumRoutingShards(int numShards, Version indexVersionCreated) {
        if (indexVersionCreated.onOrAfter(Version.V_7_0_0)) {
            int log2MaxNumShards = 10;
            int log2NumShards = 32 - Integer.numberOfLeadingZeros(numShards - 1);
            int numSplits = log2MaxNumShards - log2NumShards;
            numSplits = Math.max(1, numSplits);
            return numShards * 1 << numSplits;
        }
        return numShards;
    }

    public static void validateTranslogRetentionSettings(Settings indexSettings) {
        if (IndexMetadata.SETTING_INDEX_VERSION_CREATED.get(indexSettings).onOrAfter(Version.V_8_0_0) && (IndexSettings.INDEX_TRANSLOG_RETENTION_AGE_SETTING.exists(indexSettings) || IndexSettings.INDEX_TRANSLOG_RETENTION_SIZE_SETTING.exists(indexSettings))) {
            throw new IllegalArgumentException("Translog retention settings [index.translog.retention.age] and [index.translog.retention.size] are no longer supported. Please do not specify values for these settings");
        }
        if (IndexSettings.INDEX_SOFT_DELETES_SETTING.get(indexSettings).booleanValue() && (IndexSettings.INDEX_TRANSLOG_RETENTION_AGE_SETTING.exists(indexSettings) || IndexSettings.INDEX_TRANSLOG_RETENTION_SIZE_SETTING.exists(indexSettings))) {
            deprecationLogger.critical(DeprecationCategory.SETTINGS, "translog_retention", "Translog retention settings [index.translog.retention.age] and [index.translog.retention.size] are deprecated and effectively ignored. They will be removed in a future version.", new Object[0]);
        }
    }

    public static void validateStoreTypeSetting(Settings indexSettings) {
        String storeType = IndexModule.INDEX_STORE_TYPE_SETTING.get(indexSettings);
        if (IndexModule.Type.SIMPLEFS.match(storeType)) {
            deprecationLogger.critical(DeprecationCategory.SETTINGS, "store_type_setting", "[simplefs] is deprecated and will be removed in 8.0. Use [niofs] or other file systems instead. Elasticsearch 7.15 or later uses [niofs] for the [simplefs] store type as it offers superior or equivalent performance to [simplefs].", new Object[0]);
        }
    }
}

