/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.cluster.routing;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import org.elasticsearch.cluster.node.DiscoveryNodes;
import org.elasticsearch.cluster.routing.PlainShardIterator;
import org.elasticsearch.cluster.routing.RotationShardShuffler;
import org.elasticsearch.cluster.routing.ShardIterator;
import org.elasticsearch.cluster.routing.ShardRouting;
import org.elasticsearch.cluster.routing.ShardRoutingState;
import org.elasticsearch.cluster.routing.ShardShuffler;
import org.elasticsearch.common.ExponentiallyWeightedMovingAverage;
import org.elasticsearch.common.Randomness;
import org.elasticsearch.common.io.stream.StreamInput;
import org.elasticsearch.common.io.stream.StreamOutput;
import org.elasticsearch.common.util.CollectionUtils;
import org.elasticsearch.common.util.set.Sets;
import org.elasticsearch.core.Nullable;
import org.elasticsearch.index.Index;
import org.elasticsearch.index.shard.ShardId;
import org.elasticsearch.node.ResponseCollectorService;

public class IndexShardRoutingTable
implements Iterable<ShardRouting> {
    final ShardShuffler shuffler;
    final ShardId shardId;
    final ShardRouting primary;
    final List<ShardRouting> replicas;
    final List<ShardRouting> shards;
    final List<ShardRouting> activeShards;
    final List<ShardRouting> assignedShards;
    final Set<String> allAllocationIds;
    final boolean allShardsStarted;
    final List<ShardRouting> allInitializingShards;

    IndexShardRoutingTable(ShardId shardId, List<ShardRouting> shards) {
        this.shardId = shardId;
        this.shuffler = new RotationShardShuffler(Randomness.get().nextInt());
        this.shards = Collections.unmodifiableList(shards);
        ShardRouting primary = null;
        ArrayList<ShardRouting> replicas = new ArrayList<ShardRouting>();
        ArrayList<ShardRouting> activeShards = new ArrayList<ShardRouting>();
        ArrayList<ShardRouting> assignedShards = new ArrayList<ShardRouting>();
        ArrayList<ShardRouting> allInitializingShards = new ArrayList<ShardRouting>();
        HashSet<String> allAllocationIds = new HashSet<String>();
        boolean allShardsStarted = true;
        for (ShardRouting shard : shards) {
            if (shard.primary()) {
                primary = shard;
            } else {
                replicas.add(shard);
            }
            if (shard.active()) {
                activeShards.add(shard);
            }
            if (shard.initializing()) {
                allInitializingShards.add(shard);
            }
            if (shard.relocating()) {
                allInitializingShards.add(shard.getTargetRelocatingShard());
                allAllocationIds.add(shard.getTargetRelocatingShard().allocationId().getId());
                assert (shard.assignedToNode()) : "relocating from unassigned " + shard;
                assert (shard.getTargetRelocatingShard().assignedToNode()) : "relocating to unassigned " + shard.getTargetRelocatingShard();
                assignedShards.add(shard.getTargetRelocatingShard());
            }
            if (shard.assignedToNode()) {
                assignedShards.add(shard);
                allAllocationIds.add(shard.allocationId().getId());
            }
            if (shard.state() == ShardRoutingState.STARTED) continue;
            allShardsStarted = false;
        }
        this.allShardsStarted = allShardsStarted;
        this.primary = primary;
        this.replicas = CollectionUtils.wrapUnmodifiableOrEmptySingleton(replicas);
        this.activeShards = CollectionUtils.wrapUnmodifiableOrEmptySingleton(activeShards);
        this.assignedShards = CollectionUtils.wrapUnmodifiableOrEmptySingleton(assignedShards);
        this.allInitializingShards = CollectionUtils.wrapUnmodifiableOrEmptySingleton(allInitializingShards);
        this.allAllocationIds = Collections.unmodifiableSet(allAllocationIds);
    }

    public ShardId shardId() {
        return this.shardId;
    }

    public ShardId getShardId() {
        return this.shardId();
    }

    @Override
    public Iterator<ShardRouting> iterator() {
        return this.shards.iterator();
    }

    public int size() {
        return this.shards.size();
    }

    public int getSize() {
        return this.size();
    }

    public List<ShardRouting> shards() {
        return this.shards;
    }

    public List<ShardRouting> getShards() {
        return this.shards();
    }

    public List<ShardRouting> activeShards() {
        return this.activeShards;
    }

    public List<ShardRouting> getAllInitializingShards() {
        return this.allInitializingShards;
    }

    public List<ShardRouting> getActiveShards() {
        return this.activeShards();
    }

    public List<ShardRouting> assignedShards() {
        return this.assignedShards;
    }

    public ShardIterator shardsRandomIt() {
        return new PlainShardIterator(this.shardId, this.shuffler.shuffle(this.shards));
    }

    public ShardIterator shardsIt() {
        return new PlainShardIterator(this.shardId, this.shards);
    }

    public ShardIterator shardsIt(int seed) {
        return new PlainShardIterator(this.shardId, this.shuffler.shuffle(this.shards, seed));
    }

    public ShardIterator activeInitializingShardsRandomIt() {
        return this.activeInitializingShardsIt(this.shuffler.nextSeed());
    }

    public ShardIterator activeInitializingShardsIt(int seed) {
        if (this.allInitializingShards.isEmpty()) {
            return new PlainShardIterator(this.shardId, this.shuffler.shuffle(this.activeShards, seed));
        }
        ArrayList<ShardRouting> ordered = new ArrayList<ShardRouting>(this.activeShards.size() + this.allInitializingShards.size());
        ordered.addAll(this.shuffler.shuffle(this.activeShards, seed));
        ordered.addAll(this.allInitializingShards);
        return new PlainShardIterator(this.shardId, ordered);
    }

    public ShardIterator activeInitializingShardsRankedIt(@Nullable ResponseCollectorService collector, @Nullable Map<String, Long> nodeSearchCounts) {
        int seed = this.shuffler.nextSeed();
        if (this.allInitializingShards.isEmpty()) {
            return new PlainShardIterator(this.shardId, IndexShardRoutingTable.rankShardsAndUpdateStats(this.shuffler.shuffle(this.activeShards, seed), collector, nodeSearchCounts));
        }
        ArrayList<ShardRouting> ordered = new ArrayList<ShardRouting>(this.activeShards.size() + this.allInitializingShards.size());
        List<ShardRouting> rankedActiveShards = IndexShardRoutingTable.rankShardsAndUpdateStats(this.shuffler.shuffle(this.activeShards, seed), collector, nodeSearchCounts);
        ordered.addAll(rankedActiveShards);
        List<ShardRouting> rankedInitializingShards = IndexShardRoutingTable.rankShardsAndUpdateStats(this.allInitializingShards, collector, nodeSearchCounts);
        ordered.addAll(rankedInitializingShards);
        return new PlainShardIterator(this.shardId, ordered);
    }

    private static Set<String> getAllNodeIds(List<ShardRouting> shards) {
        HashSet<String> nodeIds = new HashSet<String>();
        for (ShardRouting shard : shards) {
            nodeIds.add(shard.currentNodeId());
        }
        return nodeIds;
    }

    private static Map<String, Optional<ResponseCollectorService.ComputedNodeStats>> getNodeStats(Set<String> nodeIds, ResponseCollectorService collector) {
        HashMap<String, Optional<ResponseCollectorService.ComputedNodeStats>> nodeStats = new HashMap<String, Optional<ResponseCollectorService.ComputedNodeStats>>(nodeIds.size());
        for (String nodeId : nodeIds) {
            nodeStats.put(nodeId, collector.getNodeStatistics(nodeId));
        }
        return nodeStats;
    }

    private static Map<String, Double> rankNodes(Map<String, Optional<ResponseCollectorService.ComputedNodeStats>> nodeStats, Map<String, Long> nodeSearchCounts) {
        HashMap<String, Double> nodeRanks = new HashMap<String, Double>(nodeStats.size());
        for (Map.Entry<String, Optional<ResponseCollectorService.ComputedNodeStats>> entry : nodeStats.entrySet()) {
            Optional<ResponseCollectorService.ComputedNodeStats> maybeStats = entry.getValue();
            maybeStats.ifPresent(stats -> {
                String nodeId = (String)entry.getKey();
                nodeRanks.put(nodeId, stats.rank(nodeSearchCounts.getOrDefault(nodeId, 0L)));
            });
        }
        return nodeRanks;
    }

    private static void adjustStats(ResponseCollectorService collector, Map<String, Optional<ResponseCollectorService.ComputedNodeStats>> nodeStats, String minNodeId, ResponseCollectorService.ComputedNodeStats minStats) {
        if (minNodeId != null) {
            for (Map.Entry<String, Optional<ResponseCollectorService.ComputedNodeStats>> entry : nodeStats.entrySet()) {
                String nodeId = entry.getKey();
                Optional<ResponseCollectorService.ComputedNodeStats> maybeStats = entry.getValue();
                if (nodeId.equals(minNodeId) || !maybeStats.isPresent()) continue;
                ResponseCollectorService.ComputedNodeStats stats = maybeStats.get();
                int updatedQueue = (minStats.queueSize + stats.queueSize) / 2;
                long updatedResponse = (long)(minStats.responseTime + stats.responseTime) / 2L;
                ExponentiallyWeightedMovingAverage avgServiceTime = new ExponentiallyWeightedMovingAverage(0.3, stats.serviceTime);
                avgServiceTime.addValue((minStats.serviceTime + stats.serviceTime) / 2.0);
                long updatedService = (long)avgServiceTime.getAverage();
                collector.addNodeStatistics(nodeId, updatedQueue, updatedResponse, updatedService);
            }
        }
    }

    private static List<ShardRouting> rankShardsAndUpdateStats(List<ShardRouting> shards, ResponseCollectorService collector, Map<String, Long> nodeSearchCounts) {
        String minNodeId;
        Optional<ResponseCollectorService.ComputedNodeStats> maybeMinStats;
        ShardRouting minShard;
        if (collector == null || nodeSearchCounts == null || shards.size() <= 1) {
            return shards;
        }
        Set<String> nodeIds = IndexShardRoutingTable.getAllNodeIds(shards);
        Map<String, Optional<ResponseCollectorService.ComputedNodeStats>> nodeStats = IndexShardRoutingTable.getNodeStats(nodeIds, collector);
        Map<String, Double> nodeRanks = IndexShardRoutingTable.rankNodes(nodeStats, nodeSearchCounts);
        ArrayList<ShardRouting> sortedShards = new ArrayList<ShardRouting>(shards);
        Collections.sort(sortedShards, new NodeRankComparator(nodeRanks));
        if (sortedShards.size() > 1 && (minShard = sortedShards.get(0)).started() && (maybeMinStats = nodeStats.get(minNodeId = minShard.currentNodeId())).isPresent()) {
            IndexShardRoutingTable.adjustStats(collector, nodeStats, minNodeId, maybeMinStats.get());
            nodeSearchCounts.compute(minNodeId, (id, conns) -> conns == null ? 1L : conns + 1L);
        }
        return sortedShards;
    }

    public ShardIterator primaryShardIt() {
        if (this.primary != null) {
            return new PlainShardIterator(this.shardId, Collections.singletonList(this.primary));
        }
        return new PlainShardIterator(this.shardId, Collections.emptyList());
    }

    public ShardIterator onlyNodeActiveInitializingShardsIt(String nodeId) {
        ArrayList<ShardRouting> ordered = new ArrayList<ShardRouting>(this.activeShards.size() + this.allInitializingShards.size());
        int seed = this.shuffler.nextSeed();
        for (ShardRouting shardRouting : this.shuffler.shuffle(this.activeShards, seed)) {
            if (!nodeId.equals(shardRouting.currentNodeId())) continue;
            ordered.add(shardRouting);
        }
        for (ShardRouting shardRouting : this.shuffler.shuffle(this.allInitializingShards, seed)) {
            if (!nodeId.equals(shardRouting.currentNodeId())) continue;
            ordered.add(shardRouting);
        }
        return new PlainShardIterator(this.shardId, ordered);
    }

    public ShardIterator onlyNodeSelectorActiveInitializingShardsIt(String nodeAttributes, DiscoveryNodes discoveryNodes) {
        return this.onlyNodeSelectorActiveInitializingShardsIt(new String[]{nodeAttributes}, discoveryNodes);
    }

    public ShardIterator onlyNodeSelectorActiveInitializingShardsIt(String[] nodeAttributes, DiscoveryNodes discoveryNodes) {
        ArrayList<ShardRouting> ordered = new ArrayList<ShardRouting>(this.activeShards.size() + this.allInitializingShards.size());
        HashSet<String> selectedNodes = Sets.newHashSet(discoveryNodes.resolveNodes(nodeAttributes));
        int seed = this.shuffler.nextSeed();
        for (ShardRouting shardRouting : this.shuffler.shuffle(this.activeShards, seed)) {
            if (!selectedNodes.contains(shardRouting.currentNodeId())) continue;
            ordered.add(shardRouting);
        }
        for (ShardRouting shardRouting : this.shuffler.shuffle(this.allInitializingShards, seed)) {
            if (!selectedNodes.contains(shardRouting.currentNodeId())) continue;
            ordered.add(shardRouting);
        }
        if (ordered.isEmpty()) {
            String message = String.format(Locale.ROOT, "no data nodes with %s [%s] found for shard: %s", nodeAttributes.length == 1 ? "criteria" : "criterion", String.join((CharSequence)",", nodeAttributes), this.shardId());
            throw new IllegalArgumentException(message);
        }
        return new PlainShardIterator(this.shardId, ordered);
    }

    public ShardIterator preferNodeActiveInitializingShardsIt(Set<String> nodeIds) {
        ArrayList<ShardRouting> preferred = new ArrayList<ShardRouting>(this.activeShards.size() + this.allInitializingShards.size());
        ArrayList<ShardRouting> notPreferred = new ArrayList<ShardRouting>(this.activeShards.size() + this.allInitializingShards.size());
        for (ShardRouting shardRouting : this.shuffler.shuffle(this.activeShards)) {
            if (nodeIds.contains(shardRouting.currentNodeId())) {
                preferred.add(shardRouting);
                continue;
            }
            notPreferred.add(shardRouting);
        }
        preferred.addAll(notPreferred);
        if (!this.allInitializingShards.isEmpty()) {
            preferred.addAll(this.allInitializingShards);
        }
        return new PlainShardIterator(this.shardId, preferred);
    }

    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || this.getClass() != o.getClass()) {
            return false;
        }
        IndexShardRoutingTable that = (IndexShardRoutingTable)o;
        if (!this.shardId.equals(that.shardId)) {
            return false;
        }
        return this.shards.equals(that.shards);
    }

    public int hashCode() {
        int result = this.shardId.hashCode();
        result = 31 * result + this.shards.hashCode();
        return result;
    }

    public boolean allShardsStarted() {
        return this.allShardsStarted;
    }

    @Nullable
    public ShardRouting getByAllocationId(String allocationId) {
        for (ShardRouting shardRouting : this.assignedShards()) {
            if (!shardRouting.allocationId().getId().equals(allocationId)) continue;
            return shardRouting;
        }
        return null;
    }

    public Set<String> getAllAllocationIds() {
        return this.allAllocationIds;
    }

    public ShardRouting primaryShard() {
        return this.primary;
    }

    public List<ShardRouting> replicaShards() {
        return this.replicas;
    }

    public List<ShardRouting> replicaShardsWithState(ShardRoutingState ... states) {
        ArrayList<ShardRouting> shards = new ArrayList<ShardRouting>();
        for (ShardRouting shardEntry : this.replicas) {
            for (ShardRoutingState state : states) {
                if (shardEntry.state() != state) continue;
                shards.add(shardEntry);
            }
        }
        return shards;
    }

    public List<ShardRouting> shardsWithState(ShardRoutingState state) {
        if (state == ShardRoutingState.INITIALIZING) {
            return this.allInitializingShards;
        }
        ArrayList<ShardRouting> shards = new ArrayList<ShardRouting>();
        for (ShardRouting shardEntry : this) {
            if (shardEntry.state() != state) continue;
            shards.add(shardEntry);
        }
        return shards;
    }

    public String toString() {
        StringBuilder sb = new StringBuilder();
        sb.append("IndexShardRoutingTable(").append(this.shardId()).append("){");
        int numShards = this.shards.size();
        for (int i = 0; i < numShards; ++i) {
            sb.append(this.shards.get(i).shortSummary());
            if (i >= numShards - 1) continue;
            sb.append(", ");
        }
        sb.append("}");
        return sb.toString();
    }

    private static class NodeRankComparator
    implements Comparator<ShardRouting> {
        private final Map<String, Double> nodeRanks;

        NodeRankComparator(Map<String, Double> nodeRanks) {
            this.nodeRanks = nodeRanks;
        }

        @Override
        public int compare(ShardRouting s1, ShardRouting s2) {
            if (s1.currentNodeId().equals(s2.currentNodeId())) {
                return 0;
            }
            Double shard1rank = this.nodeRanks.get(s1.currentNodeId());
            Double shard2rank = this.nodeRanks.get(s2.currentNodeId());
            if (shard1rank != null) {
                if (shard2rank != null) {
                    return shard1rank.compareTo(shard2rank);
                }
                return 1;
            }
            if (shard2rank != null) {
                return -1;
            }
            return 0;
        }
    }

    public static class Builder {
        private ShardId shardId;
        private final List<ShardRouting> shards;

        public Builder(IndexShardRoutingTable indexShard) {
            this.shardId = indexShard.shardId;
            this.shards = new ArrayList<ShardRouting>(indexShard.shards);
        }

        public Builder(ShardId shardId) {
            this.shardId = shardId;
            this.shards = new ArrayList<ShardRouting>();
        }

        public Builder addShard(ShardRouting shardEntry) {
            this.shards.add(shardEntry);
            return this;
        }

        public Builder removeShard(ShardRouting shardEntry) {
            this.shards.remove(shardEntry);
            return this;
        }

        public IndexShardRoutingTable build() {
            assert (Builder.distinctNodes(this.shards)) : "more than one shard with same id assigned to same node (shards: " + this.shards + ")";
            return new IndexShardRoutingTable(this.shardId, List.copyOf(this.shards));
        }

        static boolean distinctNodes(List<ShardRouting> shards) {
            HashSet<String> nodes = new HashSet<String>();
            for (ShardRouting shard : shards) {
                if (!shard.assignedToNode()) continue;
                if (!nodes.add(shard.currentNodeId())) {
                    return false;
                }
                if (!shard.relocating() || nodes.add(shard.relocatingNodeId())) continue;
                return false;
            }
            return true;
        }

        public static IndexShardRoutingTable readFrom(StreamInput in) throws IOException {
            Index index = new Index(in);
            return Builder.readFromThin(in, index);
        }

        public static IndexShardRoutingTable readFromThin(StreamInput in, Index index) throws IOException {
            int iShardId = in.readVInt();
            ShardId shardId = new ShardId(index, iShardId);
            Builder builder = new Builder(shardId);
            int size = in.readVInt();
            for (int i = 0; i < size; ++i) {
                ShardRouting shard = new ShardRouting(shardId, in);
                builder.addShard(shard);
            }
            return builder.build();
        }

        public static void writeTo(IndexShardRoutingTable indexShard, StreamOutput out) throws IOException {
            indexShard.shardId().getIndex().writeTo(out);
            Builder.writeToThin(indexShard, out);
        }

        public static void writeToThin(IndexShardRoutingTable indexShard, StreamOutput out) throws IOException {
            out.writeVInt(indexShard.shardId.id());
            out.writeVInt(indexShard.shards.size());
            for (ShardRouting entry : indexShard) {
                entry.writeToThin(out);
            }
        }
    }

    static class AttributesKey {
        final List<String> attributes;

        AttributesKey(List<String> attributes) {
            this.attributes = attributes;
        }

        public int hashCode() {
            return this.attributes.hashCode();
        }

        public boolean equals(Object obj) {
            return obj instanceof AttributesKey && this.attributes.equals(((AttributesKey)obj).attributes);
        }
    }
}

