/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.repositories.encrypted;

import java.io.IOException;
import java.io.InputStream;
import java.util.Objects;

public final class BufferOnMarkInputStream
extends InputStream {
    final InputStream source;
    final RingBuffer ringBuffer;
    boolean storeToBuffer;
    boolean replayFromBuffer;
    boolean closed;

    public BufferOnMarkInputStream(InputStream source, int bufferSize) {
        this.source = source;
        this.ringBuffer = new RingBuffer(bufferSize);
        this.replayFromBuffer = false;
        this.storeToBuffer = false;
        this.closed = false;
    }

    @Override
    public int read(byte[] b, int off, int len) throws IOException {
        int bytesRead;
        this.ensureOpen();
        Objects.checkFromIndexSize(off, len, b.length);
        if (len == 0) {
            return 0;
        }
        if (this.replayFromBuffer) {
            bytesRead = this.ringBuffer.read(b, off, len);
            if (bytesRead == 0) {
                this.replayFromBuffer = false;
            } else {
                return bytesRead;
            }
        }
        if ((bytesRead = this.source.read(b, off, len)) <= 0) {
            return bytesRead;
        }
        if (this.storeToBuffer) {
            if (bytesRead > this.ringBuffer.getAvailableToWriteByteCount()) {
                this.storeToBuffer = false;
                this.ringBuffer.clear();
            } else {
                this.ringBuffer.write(b, off, bytesRead);
            }
        }
        return bytesRead;
    }

    @Override
    public int read() throws IOException {
        this.ensureOpen();
        byte[] arr = new byte[1];
        int readResult = this.read(arr, 0, arr.length);
        if (readResult == -1) {
            return -1;
        }
        return arr[0];
    }

    @Override
    public long skip(long n) throws IOException {
        int bytesRead;
        long remaining;
        this.ensureOpen();
        if (n <= 0L) {
            return 0L;
        }
        if (!this.storeToBuffer) {
            if (this.replayFromBuffer) {
                throw new IllegalStateException("Reset cannot be called without a preceding mark invocation");
            }
            return this.source.skip(n);
        }
        int size = (int)Math.min(2048L, remaining);
        byte[] skipBuffer = new byte[size];
        for (remaining = n; remaining > 0L && (bytesRead = this.read(skipBuffer, 0, (int)Math.min((long)size, remaining))) >= 0; remaining -= (long)bytesRead) {
        }
        return n - remaining;
    }

    @Override
    public int available() throws IOException {
        this.ensureOpen();
        int bytesAvailable = 0;
        if (this.replayFromBuffer) {
            bytesAvailable += this.ringBuffer.getAvailableToReadByteCount();
        }
        return bytesAvailable += this.source.available();
    }

    @Override
    public boolean markSupported() {
        return true;
    }

    @Override
    public void mark(int readlimit) {
        if (readlimit > this.ringBuffer.getBufferSize()) {
            throw new IllegalArgumentException("Readlimit value [" + readlimit + "] exceeds the maximum value of [" + this.ringBuffer.getBufferSize() + "]");
        }
        if (readlimit < 0) {
            throw new IllegalArgumentException("Readlimit value [" + readlimit + "] cannot be negative");
        }
        if (this.closed) {
            return;
        }
        this.storeToBuffer = true;
        if (this.replayFromBuffer) {
            this.ringBuffer.mark();
        } else {
            this.ringBuffer.clear();
        }
    }

    @Override
    public void reset() throws IOException {
        this.ensureOpen();
        if (!this.storeToBuffer) {
            throw new IOException("Mark not called or has been invalidated");
        }
        this.replayFromBuffer = true;
        this.ringBuffer.reset();
    }

    @Override
    public void close() throws IOException {
        if (!this.closed) {
            this.closed = true;
            this.source.close();
        }
    }

    public int getMaxMarkReadlimit() {
        return this.ringBuffer.getBufferSize();
    }

    private void ensureOpen() throws IOException {
        if (this.closed) {
            throw new IOException("Stream has been closed");
        }
    }

    static class RingBuffer {
        private final int bufferSize;
        byte[] buffer;
        int head;
        int tail;
        int position;

        RingBuffer(int bufferSize) {
            if (bufferSize <= 0) {
                throw new IllegalArgumentException("The buffersize constructor argument must be a strictly positive value");
            }
            this.bufferSize = bufferSize;
        }

        int getBufferSize() {
            return this.bufferSize;
        }

        void reset() {
            this.position = this.head;
        }

        void mark() {
            this.head = this.position;
        }

        void clear() {
            this.tail = 0;
            this.position = 0;
            this.head = 0;
        }

        int read(byte[] b, int off, int len) {
            Objects.requireNonNull(b);
            Objects.checkFromIndexSize(off, len, b.length);
            if (this.position == this.tail || len == 0) {
                return 0;
            }
            int readLength = this.position <= this.tail ? Math.min(len, this.tail - this.position) : Math.min(len, this.buffer.length - this.position);
            System.arraycopy(this.buffer, this.position, b, off, readLength);
            this.position += readLength;
            if (this.position == this.buffer.length) {
                this.position = 0;
                return readLength + this.read(b, off + readLength, len - readLength);
            }
            return readLength;
        }

        void write(byte[] b, int off, int len) {
            Objects.requireNonNull(b);
            Objects.checkFromIndexSize(off, len, b.length);
            if (this.buffer == null && len > 0) {
                this.buffer = new byte[this.bufferSize + 1];
                this.tail = 0;
                this.position = 0;
                this.head = 0;
            }
            if (len > this.getAvailableToWriteByteCount()) {
                throw new IllegalArgumentException("Not enough remaining space in the ring buffer");
            }
            while (len > 0) {
                int writeLength = this.head <= this.tail ? Math.min(len, this.buffer.length - this.tail - (this.head == 0 ? 1 : 0)) : Math.min(len, this.head - this.tail - 1);
                if (writeLength <= 0) {
                    throw new IllegalStateException("No space left in the ring buffer");
                }
                System.arraycopy(b, off, this.buffer, this.tail, writeLength);
                this.tail += writeLength;
                off += writeLength;
                len -= writeLength;
                if (this.tail != this.buffer.length) continue;
                this.tail = 0;
                if (this.head != 0) continue;
                throw new IllegalStateException("Possible overflow of the ring buffer");
            }
        }

        int getAvailableToWriteByteCount() {
            if (this.buffer == null) {
                return this.bufferSize;
            }
            if (this.head == this.tail) {
                return this.buffer.length - 1;
            }
            if (this.head < this.tail) {
                return this.buffer.length - this.tail + this.head - 1;
            }
            return this.head - this.tail - 1;
        }

        int getAvailableToReadByteCount() {
            if (this.buffer == null) {
                return 0;
            }
            if (this.head <= this.tail) {
                return this.tail - this.position;
            }
            if (this.position >= this.head) {
                return this.buffer.length - this.position + this.tail;
            }
            return this.tail - this.position;
        }
    }
}

