/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.transport;

import java.io.IOException;
import java.util.function.Consumer;
import org.apache.lucene.util.BytesRef;
import org.elasticsearch.Version;
import org.elasticsearch.common.bytes.BytesReference;
import org.elasticsearch.common.bytes.ReleasableBytesReference;
import org.elasticsearch.common.io.stream.StreamInput;
import org.elasticsearch.common.recycler.Recycler;
import org.elasticsearch.core.Releasable;
import org.elasticsearch.core.Releasables;
import org.elasticsearch.transport.Header;
import org.elasticsearch.transport.TcpHeader;
import org.elasticsearch.transport.TcpTransport;
import org.elasticsearch.transport.TransportDecompressor;

public class InboundDecoder
implements Releasable {
    static final Object PING = new Object();
    static final Object END_CONTENT = new Object();
    private final Version version;
    private final Recycler<BytesRef> recycler;
    private TransportDecompressor decompressor;
    private int totalNetworkSize = -1;
    private int bytesConsumed = 0;
    private boolean isCompressed = false;
    private boolean isClosed = false;

    public InboundDecoder(Version version, Recycler<BytesRef> recycler) {
        this.version = version;
        this.recycler = recycler;
    }

    public int decode(ReleasableBytesReference reference, Consumer<Object> fragmentConsumer) throws IOException {
        this.ensureOpen();
        try {
            return this.internalDecode(reference, fragmentConsumer);
        }
        catch (Exception e) {
            this.cleanDecodeState();
            throw e;
        }
    }

    public int internalDecode(ReleasableBytesReference reference, Consumer<Object> fragmentConsumer) throws IOException {
        if (this.isOnHeader()) {
            int messageLength = TcpTransport.readMessageLength(reference);
            if (messageLength == -1) {
                return 0;
            }
            if (messageLength == 0) {
                fragmentConsumer.accept(PING);
                return 6;
            }
            int headerBytesToRead = this.headerBytesToRead(reference);
            if (headerBytesToRead == 0) {
                return 0;
            }
            this.totalNetworkSize = messageLength + 6;
            Header header = InboundDecoder.readHeader(this.version, messageLength, reference);
            this.bytesConsumed += headerBytesToRead;
            if (header.isCompressed()) {
                this.isCompressed = true;
            }
            fragmentConsumer.accept(header);
            if (this.isDone()) {
                this.finishMessage(fragmentConsumer);
            }
            return headerBytesToRead;
        }
        if (this.isCompressed && this.decompressor == null) {
            TransportDecompressor decompressor = TransportDecompressor.getDecompressor(this.recycler, reference);
            if (decompressor == null) {
                return 0;
            }
            this.decompressor = decompressor;
            fragmentConsumer.accept((Object)this.decompressor.getScheme());
        }
        int remainingToConsume = this.totalNetworkSize - this.bytesConsumed;
        int maxBytesToConsume = Math.min(reference.length(), remainingToConsume);
        ReleasableBytesReference retainedContent = maxBytesToConsume == remainingToConsume ? reference.retainedSlice(0, maxBytesToConsume) : reference.retain();
        int bytesConsumedThisDecode = 0;
        if (this.decompressor != null) {
            ReleasableBytesReference decompressed;
            this.bytesConsumed += (bytesConsumedThisDecode += this.decompress(retainedContent));
            while ((decompressed = this.decompressor.pollDecompressedPage(this.isDone())) != null) {
                fragmentConsumer.accept(decompressed);
            }
        } else {
            bytesConsumedThisDecode += maxBytesToConsume;
            this.bytesConsumed += maxBytesToConsume;
            fragmentConsumer.accept(retainedContent);
        }
        if (this.isDone()) {
            this.finishMessage(fragmentConsumer);
        }
        return bytesConsumedThisDecode;
    }

    @Override
    public void close() {
        this.isClosed = true;
        this.cleanDecodeState();
    }

    private void finishMessage(Consumer<Object> fragmentConsumer) {
        this.cleanDecodeState();
        fragmentConsumer.accept(END_CONTENT);
    }

    private void cleanDecodeState() {
        try {
            Releasables.closeExpectNoException((Releasable)this.decompressor);
        }
        finally {
            this.isCompressed = false;
            this.decompressor = null;
            this.totalNetworkSize = -1;
            this.bytesConsumed = 0;
        }
    }

    private int decompress(ReleasableBytesReference content) throws IOException {
        try (ReleasableBytesReference releasableBytesReference = content;){
            int n = this.decompressor.decompress(content);
            return n;
        }
    }

    private boolean isDone() {
        return this.bytesConsumed == this.totalNetworkSize;
    }

    private int headerBytesToRead(BytesReference reference) {
        if (reference.length() < 19) {
            return 0;
        }
        Version remoteVersion = Version.fromId(reference.getInt(15));
        int fixedHeaderSize = TcpHeader.headerSize(remoteVersion);
        if (fixedHeaderSize > reference.length()) {
            return 0;
        }
        if (remoteVersion.before(TcpHeader.VERSION_WITH_HEADER_SIZE)) {
            return fixedHeaderSize;
        }
        int variableHeaderSize = reference.getInt(19);
        int totalHeaderSize = fixedHeaderSize + variableHeaderSize;
        if (totalHeaderSize > reference.length()) {
            return 0;
        }
        return totalHeaderSize;
    }

    static Header readHeader(Version version, int networkMessageSize, BytesReference bytesReference) throws IOException {
        try (StreamInput streamInput = bytesReference.streamInput();){
            streamInput.skip(6L);
            long requestId = streamInput.readLong();
            byte status = streamInput.readByte();
            Version remoteVersion = Version.fromId(streamInput.readInt());
            Header header = new Header(networkMessageSize, requestId, status, remoteVersion);
            IllegalStateException invalidVersion = InboundDecoder.ensureVersionCompatibility(remoteVersion, version, header.isHandshake());
            if (invalidVersion != null) {
                throw invalidVersion;
            }
            if (remoteVersion.onOrAfter(TcpHeader.VERSION_WITH_HEADER_SIZE)) {
                streamInput.readInt();
                header.finishParsingHeader(streamInput);
            }
            Header header2 = header;
            return header2;
        }
    }

    private boolean isOnHeader() {
        return this.totalNetworkSize == -1;
    }

    private void ensureOpen() {
        if (this.isClosed) {
            throw new IllegalStateException("Decoder is already closed");
        }
    }

    static IllegalStateException ensureVersionCompatibility(Version remoteVersion, Version currentVersion, boolean isHandshake) {
        Version compatibilityVersion;
        Version version = compatibilityVersion = isHandshake ? currentVersion.minimumCompatibilityVersion() : currentVersion;
        if (!remoteVersion.isCompatible(compatibilityVersion)) {
            Version minCompatibilityVersion = isHandshake ? compatibilityVersion : compatibilityVersion.minimumCompatibilityVersion();
            String msg = "Received " + (isHandshake ? "handshake " : "") + "message from unsupported version: [";
            return new IllegalStateException(msg + remoteVersion + "] minimal compatible version is: [" + minCompatibilityVersion + "]");
        }
        return null;
    }
}

