/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.xpack.core.ssl;

import java.io.IOException;
import java.nio.file.Path;
import java.security.AccessControlException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutionException;
import java.util.function.Consumer;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.elasticsearch.ElasticsearchException;
import org.elasticsearch.common.ssl.SslConfiguration;
import org.elasticsearch.watcher.FileChangesListener;
import org.elasticsearch.watcher.FileWatcher;
import org.elasticsearch.watcher.ResourceWatcherService;
import org.elasticsearch.xpack.core.ssl.SSLService;

public final class SSLConfigurationReloader {
    private static final Logger logger = LogManager.getLogger(SSLConfigurationReloader.class);
    private final CompletableFuture<SSLService> sslServiceFuture = new CompletableFuture();

    public SSLConfigurationReloader(ResourceWatcherService resourceWatcherService, Collection<SslConfiguration> sslConfigurations) {
        SSLConfigurationReloader.startWatching(SSLConfigurationReloader.reloadConsumer(this.sslServiceFuture), resourceWatcherService, sslConfigurations);
    }

    SSLConfigurationReloader(Consumer<SslConfiguration> reloadConsumer, ResourceWatcherService resourceWatcherService, Collection<SslConfiguration> sslConfigurations) {
        SSLConfigurationReloader.startWatching(reloadConsumer, resourceWatcherService, sslConfigurations);
    }

    public void setSSLService(SSLService sslService) {
        boolean completed = this.sslServiceFuture.complete(sslService);
        if (!completed) {
            throw new IllegalStateException("ssl service future was already completed!");
        }
    }

    private static Consumer<SslConfiguration> reloadConsumer(CompletableFuture<SSLService> future) {
        return sslConfiguration -> {
            try {
                SSLService sslService = (SSLService)future.get();
                logger.debug("reloading ssl configuration [{}]", sslConfiguration);
                sslService.reloadSSLContext((SslConfiguration)sslConfiguration);
            }
            catch (InterruptedException e) {
                Thread.currentThread().interrupt();
            }
            catch (ExecutionException e) {
                throw new ElasticsearchException("failed to obtain ssl service", (Throwable)e, new Object[0]);
            }
        };
    }

    private static void startWatching(Consumer<SslConfiguration> reloadConsumer, ResourceWatcherService resourceWatcherService, Collection<SslConfiguration> sslConfigurations) {
        HashMap<Path, List> pathToConfigurationsMap = new HashMap<Path, List>();
        for (SslConfiguration sslConfiguration : sslConfigurations) {
            Collection<Path> filesToMonitor = sslConfiguration.getDependentFiles();
            for (Path directory : SSLConfigurationReloader.directoriesToMonitor(filesToMonitor)) {
                pathToConfigurationsMap.compute(directory, (path, list) -> {
                    if (list == null) {
                        list = new ArrayList<SslConfiguration>();
                    }
                    list.add(sslConfiguration);
                    return list;
                });
            }
        }
        pathToConfigurationsMap.forEach((path, configurations) -> {
            ChangeListener changeListener = new ChangeListener(List.copyOf(configurations), reloadConsumer);
            FileWatcher fileWatcher = new FileWatcher((Path)path);
            fileWatcher.addListener(changeListener);
            try {
                resourceWatcherService.add(fileWatcher, ResourceWatcherService.Frequency.HIGH);
            }
            catch (IOException | AccessControlException e) {
                logger.error("failed to start watching directory [{}] for ssl configurations [{}] - {}", path, configurations, (Object)e);
            }
        });
    }

    private static Set<Path> directoriesToMonitor(Iterable<Path> filePaths) {
        HashSet<Path> paths = new HashSet<Path>();
        for (Path path : filePaths) {
            paths.add(path.getParent());
        }
        return paths;
    }

    private static class ChangeListener
    implements FileChangesListener {
        private final List<SslConfiguration> sslConfigurations;
        private final Consumer<SslConfiguration> reloadConsumer;

        private ChangeListener(List<SslConfiguration> sslConfigurations, Consumer<SslConfiguration> reloadConsumer) {
            this.sslConfigurations = sslConfigurations;
            this.reloadConsumer = reloadConsumer;
        }

        @Override
        public void onFileCreated(Path file) {
            this.onFileChanged(file);
        }

        @Override
        public void onFileDeleted(Path file) {
            this.onFileChanged(file);
        }

        @Override
        public void onFileChanged(Path file) {
            boolean reloaded = false;
            for (SslConfiguration sslConfiguration : this.sslConfigurations) {
                if (!sslConfiguration.getDependentFiles().contains(file)) continue;
                this.reloadConsumer.accept(sslConfiguration);
                reloaded = true;
            }
            if (reloaded) {
                logger.info("reloaded [{}] and updated ssl contexts using this file", (Object)file);
            }
        }
    }
}

