"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.getErrorTypeFromS3Error = exports.getErrorTypeFromRequestError = exports.getErrorTypeFromStatusCode = exports.UplogBuffer = exports.UplogBufferFileLockPath = exports.UplogBufferFilePath = exports.ErrorType = exports.LogType = void 0;
const os_1 = __importDefault(require("os"));
const fs_1 = __importDefault(require("fs"));
const path_1 = __importDefault(require("path"));
const package_json_1 = __importDefault(require("./package.json"));
const lockfile_1 = __importDefault(require("lockfile"));
var LogType;
(function (LogType) {
    LogType["Request"] = "request";
    LogType["SdkApi"] = "sdkapi";
})(LogType = exports.LogType || (exports.LogType = {}));
var ErrorType;
(function (ErrorType) {
    ErrorType["UnknownError"] = "unknown_error";
    ErrorType["NetworkError"] = "network_error";
    ErrorType["Timeout"] = "timeout";
    ErrorType["UnknownHost"] = "unknown_host";
    ErrorType["CannotConnectToHost"] = "cannot_connect_to_host";
    ErrorType["TransmissionError"] = "transmission_error";
    ErrorType["ProxyError"] = "proxy_error";
    ErrorType["SslError"] = "ssl_error";
    ErrorType["ResponseError"] = "response_error";
    ErrorType["ParseError"] = "parse_error";
    ErrorType["MaliciousResponse"] = "malicious_response";
    ErrorType["UserCanceled"] = "user_canceled";
    ErrorType["BadRequest"] = "bad_request";
    ErrorType["UnexpectedSyscallError"] = "unexpected_syscall_error";
})(ErrorType = exports.ErrorType || (exports.ErrorType = {}));
exports.UplogBufferFilePath = path_1.default.join(os_1.default.homedir(), '.kodo-s3-adapter-sdk', 'uplog-buffer');
exports.UplogBufferFileLockPath = path_1.default.join(os_1.default.homedir(), '.kodo-s3-adapter-sdk', 'uplog-buffer.lock');
class UplogBuffer {
    constructor(option) {
        this.option = option;
        if (!UplogBuffer.uploadBufferFd) {
            const dirPath = path_1.default.dirname(exports.UplogBufferFilePath);
            if (!fs_1.default.existsSync(dirPath)) {
                fs_1.default.mkdirSync(dirPath, { recursive: true, mode: 0o700 });
            }
            UplogBuffer.uploadBufferFd = fs_1.default.openSync(exports.UplogBufferFilePath, 'a', 0o600);
        }
    }
    log(entry) {
        UplogBuffer.uploadBufferedEntries.push(this.convertUplogEntryToJSON(entry) + "\n");
        return new Promise((resolve, reject) => {
            this.flushBufferToLogFile().then((fileSize) => {
                var _a;
                if (fileSize && fileSize >= ((_a = this.option.bufferSize) !== null && _a !== void 0 ? _a : 1 << 20)) {
                    this.exportLogs().then(resolve, reject);
                }
                else {
                    resolve();
                }
            }, reject);
        });
    }
    flushBufferToLogFile() {
        return new Promise((resolve, reject) => {
            lockfile_1.default.lock(exports.UplogBufferFileLockPath, this.lockOptions(), (err) => {
                if (err) {
                    if (err.code === 'EEXIST') {
                        resolve(undefined);
                    }
                    else {
                        console.warn("locked fail:", err);
                        reject(err);
                    }
                }
                else {
                    const uploadBufferedEntries = UplogBuffer.uploadBufferedEntries;
                    UplogBuffer.uploadBufferedEntries = [];
                    const writePromise = uploadBufferedEntries.reduce((writePromise, data) => {
                        return writePromise.then(() => {
                            return new Promise((resolve, reject) => {
                                fs_1.default.write(UplogBuffer.uploadBufferFd, data, (err) => {
                                    if (err) {
                                        reject(err);
                                    }
                                    else {
                                        resolve();
                                    }
                                });
                            });
                        }, reject);
                    }, Promise.resolve());
                    writePromise.then(() => {
                        fs_1.default.fstat(UplogBuffer.uploadBufferFd, (err, stats) => {
                            if (err) {
                                lockfile_1.default.unlock(exports.UplogBufferFileLockPath, () => { reject(err); });
                                return;
                            }
                            lockfile_1.default.unlock(exports.UplogBufferFileLockPath, (err) => {
                                if (err) {
                                    reject(err);
                                }
                                else {
                                    resolve(stats.size);
                                }
                            });
                        });
                    }, (err) => {
                        lockfile_1.default.unlock(exports.UplogBufferFileLockPath, () => { reject(err); });
                    });
                }
            });
        });
    }
    exportLogs() {
        if (!this.option.onBufferFull) {
            return Promise.resolve();
        }
        return new Promise((resolve, reject) => {
            lockfile_1.default.lock(exports.UplogBufferFileLockPath, this.lockOptions(), (err) => {
                if (err) {
                    console.warn("locked fail:", err);
                    reject(err);
                }
                else {
                    fs_1.default.readFile(exports.UplogBufferFilePath, (err, buffer) => {
                        if (err) {
                            lockfile_1.default.unlock(exports.UplogBufferFileLockPath, () => { reject(err); });
                            return;
                        }
                        if (this.option.onBufferFull) {
                            this.option.onBufferFull(buffer).then(() => {
                                fs_1.default.truncate(exports.UplogBufferFilePath, (err) => {
                                    if (err) {
                                        lockfile_1.default.unlock(exports.UplogBufferFileLockPath, () => { reject(err); });
                                    }
                                    else {
                                        lockfile_1.default.unlock(exports.UplogBufferFileLockPath, (err) => {
                                            if (err) {
                                                reject(err);
                                            }
                                            else {
                                                resolve();
                                            }
                                        });
                                    }
                                });
                            }, (err) => {
                                lockfile_1.default.unlock(exports.UplogBufferFileLockPath, () => { reject(err); });
                            });
                        }
                        else {
                            lockfile_1.default.unlock(exports.UplogBufferFileLockPath, (err) => {
                                if (err) {
                                    reject(err);
                                }
                                else {
                                    resolve();
                                }
                            });
                        }
                    });
                }
            });
        });
    }
    convertUplogEntryToJSON(entry) {
        entry.os_name = os_1.default.platform();
        entry.os_version = os_1.default.release();
        entry.sdk_name = this.option.appName;
        entry.sdk_version = this.option.appVersion;
        entry.http_client = `kodo-s3-adapter-sdk`;
        entry.http_client_version = package_json_1.default.version;
        entry.up_time = Math.trunc(Date.now() / 1000);
        return JSON.stringify(entry);
    }
    lockOptions() {
        return { retries: 10, retryWait: 100 };
    }
}
exports.UplogBuffer = UplogBuffer;
UplogBuffer.uploadBufferedEntries = [];
UplogBuffer.uploadBufferFd = undefined;
exports.getErrorTypeFromStatusCode = (statusCode) => {
    if (statusCode > 399 && statusCode < 500 ||
        statusCode == 573 || statusCode == 579 ||
        statusCode == 608 || statusCode == 612 ||
        statusCode == 614 || statusCode == 630 ||
        statusCode == 631 || statusCode == 701) {
        return ErrorType.BadRequest;
    }
    else {
        return ErrorType.ResponseError;
    }
};
exports.getErrorTypeFromRequestError = (err) => {
    switch (err.code) {
        case 'ENOTFOUND':
            return ErrorType.UnknownHost;
        case 'ECONNREFUSED':
            return ErrorType.CannotConnectToHost;
        case 'ECONNRESET':
            return ErrorType.CannotConnectToHost;
        case 'EMFILE':
            return ErrorType.UnexpectedSyscallError;
        case 'EACCES':
            return ErrorType.UnexpectedSyscallError;
        case 'ETIMEDOUT':
            return ErrorType.Timeout;
        case 'EPIPE':
            return ErrorType.TransmissionError;
        case 'EPROTO':
            return ErrorType.NetworkError;
        case 'UNABLE_TO_VERIFY_LEAF_SIGNATURE':
            return ErrorType.SslError;
    }
    if (err.name && err.name.endsWith('TimeoutError')) {
        return ErrorType.Timeout;
    }
    switch (err.name) {
        case 'JSONResponseFormatError':
            return ErrorType.ParseError;
        default:
            return ErrorType.UnknownError;
    }
};
exports.getErrorTypeFromS3Error = (err) => {
    switch (err.code) {
        case 'TimeoutError':
            return ErrorType.Timeout;
        case 'NetworkingError':
            return ErrorType.NetworkError;
        case 'UnknownEndpoint':
            return ErrorType.UnknownHost;
        case 'XMLParserError':
            return ErrorType.ParseError;
        case 'CredentialsError':
            return ErrorType.BadRequest;
        case 'InvalidHeader':
            return ErrorType.BadRequest;
        case 'InvalidParameter':
            return ErrorType.BadRequest;
        case 'InvalidDigest':
            return ErrorType.BadRequest;
        case 'RequestAbortedError':
            return ErrorType.UserCanceled;
        default:
            return ErrorType.UnknownError;
    }
};
//# sourceMappingURL=uplog.js.map