"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
class Semaphore {
    /**
     * Creates a new semaphore container.
     * @param count Number of available semaphores
     */
    constructor(count) {
        this._maxSemaphores = count;
        this._currentSemaphores = new Set();
        this._waitingCallers = [];
    }
    tryAcquire() {
        const semaphoreId = Symbol();
        if (this.hasAvailableSemaphores) {
            this._currentSemaphores.add(semaphoreId);
            return this.createReleaseFunction(semaphoreId);
        }
        else {
            throw new Error('No semaphore available.');
        }
    }
    /**
     * Acquire a new semaphore.
     * The returned promise resolves as soon as a semaphore is available and returns a function `release`
     * which has to be used to release the acquired promise.
     * @return {Promise<Function>}
     */
    acquire() {
        const promise = new Promise((resolve) => {
            const semaphoreId = Symbol();
            const release = this.createReleaseFunction(semaphoreId);
            this._waitingCallers.push(() => {
                resolve(release);
                return semaphoreId;
            });
        });
        setTimeout(() => this.treatPendingCallers(), 0);
        return promise;
    }
    /**
     * Release the given semaphore.
     * This is usually done from inside the resolve callback because the ID is only known there.
     */
    release(id) {
        if (this._currentSemaphores.has(id)) {
            this._currentSemaphores.delete(id);
        }
        setTimeout(() => this.treatPendingCallers(), 0);
    }
    get hasAvailableSemaphores() {
        return this._currentSemaphores.size < this._maxSemaphores;
    }
    /**
     * @param id ID of the semaphore which is released when calling this function
     */
    createReleaseFunction(id) {
        return () => this.release(id);
    }
    /**
     * When a semaphore is requested, the request is added to a queue.
     * The queue is processed by this function.
     */
    treatPendingCallers() {
        while (this._waitingCallers.length > 0 && this.hasAvailableSemaphores) {
            const semaphoreId = this._waitingCallers.shift()();
            this._currentSemaphores.add(semaphoreId);
        }
    }
}
exports.Semaphore = Semaphore;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiaW5kZXguanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyIuLi8uLi9zcmMvaW5kZXgudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6Ijs7QUFBQSxNQUFhLFNBQVM7SUFPbEI7OztPQUdHO0lBQ0gsWUFBYSxLQUFjO1FBQ3ZCLElBQUksQ0FBQyxjQUFjLEdBQUcsS0FBSyxDQUFDO1FBQzVCLElBQUksQ0FBQyxrQkFBa0IsR0FBRyxJQUFJLEdBQUcsRUFBRSxDQUFDO1FBQ3BDLElBQUksQ0FBQyxlQUFlLEdBQUcsRUFBRSxDQUFDO0lBQzlCLENBQUM7SUFFRCxVQUFVO1FBQ04sTUFBTSxXQUFXLEdBQUcsTUFBTSxFQUFFLENBQUM7UUFDN0IsSUFBSyxJQUFJLENBQUMsc0JBQXNCLEVBQUc7WUFDL0IsSUFBSSxDQUFDLGtCQUFrQixDQUFDLEdBQUcsQ0FBRSxXQUFXLENBQUUsQ0FBQztZQUMzQyxPQUFPLElBQUksQ0FBQyxxQkFBcUIsQ0FBRSxXQUFXLENBQUUsQ0FBQztTQUNwRDthQUFNO1lBQ0gsTUFBTSxJQUFJLEtBQUssQ0FBRSx5QkFBeUIsQ0FBRSxDQUFDO1NBQ2hEO0lBQ0wsQ0FBQztJQUVEOzs7OztPQUtHO0lBQ0gsT0FBTztRQUNILE1BQU0sT0FBTyxHQUFHLElBQUksT0FBTyxDQUFjLENBQUUsT0FBTyxFQUFHLEVBQUU7WUFFbkQsTUFBTSxXQUFXLEdBQUcsTUFBTSxFQUFFLENBQUM7WUFDN0IsTUFBTSxPQUFPLEdBQUcsSUFBSSxDQUFDLHFCQUFxQixDQUFFLFdBQVcsQ0FBRSxDQUFDO1lBRTFELElBQUksQ0FBQyxlQUFlLENBQUMsSUFBSSxDQUFFLEdBQUcsRUFBRTtnQkFDNUIsT0FBTyxDQUFFLE9BQU8sQ0FBRSxDQUFDO2dCQUNuQixPQUFPLFdBQVcsQ0FBQztZQUN2QixDQUFDLENBQUUsQ0FBQztRQUNSLENBQUMsQ0FBRSxDQUFDO1FBQ0osVUFBVSxDQUFFLEdBQUcsRUFBRSxDQUFDLElBQUksQ0FBQyxtQkFBbUIsRUFBRSxFQUFFLENBQUMsQ0FBRSxDQUFDO1FBQ2xELE9BQU8sT0FBTyxDQUFDO0lBQ25CLENBQUM7SUFFRDs7O09BR0c7SUFDSCxPQUFPLENBQUUsRUFBVztRQUNoQixJQUFLLElBQUksQ0FBQyxrQkFBa0IsQ0FBQyxHQUFHLENBQUUsRUFBRSxDQUFFLEVBQUc7WUFDckMsSUFBSSxDQUFDLGtCQUFrQixDQUFDLE1BQU0sQ0FBRSxFQUFFLENBQUUsQ0FBQztTQUN4QztRQUNELFVBQVUsQ0FBRSxHQUFHLEVBQUUsQ0FBQyxJQUFJLENBQUMsbUJBQW1CLEVBQUUsRUFBRSxDQUFDLENBQUUsQ0FBQztJQUN0RCxDQUFDO0lBRUQsSUFBWSxzQkFBc0I7UUFDOUIsT0FBTyxJQUFJLENBQUMsa0JBQWtCLENBQUMsSUFBSSxHQUFHLElBQUksQ0FBQyxjQUFjLENBQUM7SUFDOUQsQ0FBQztJQUVEOztPQUVHO0lBQ0sscUJBQXFCLENBQUUsRUFBVztRQUN0QyxPQUFPLEdBQUcsRUFBRSxDQUFDLElBQUksQ0FBQyxPQUFPLENBQUUsRUFBRSxDQUFFLENBQUM7SUFDcEMsQ0FBQztJQUdEOzs7T0FHRztJQUNLLG1CQUFtQjtRQUN2QixPQUFRLElBQUksQ0FBQyxlQUFlLENBQUMsTUFBTSxHQUFHLENBQUMsSUFBSSxJQUFJLENBQUMsc0JBQXNCLEVBQUc7WUFDckUsTUFBTSxXQUFXLEdBQUcsSUFBSSxDQUFDLGVBQWUsQ0FBQyxLQUFLLEVBQUUsRUFBRSxDQUFDO1lBQ25ELElBQUksQ0FBQyxrQkFBa0IsQ0FBQyxHQUFHLENBQUUsV0FBVyxDQUFFLENBQUM7U0FDOUM7SUFDTCxDQUFDO0NBRUo7QUFsRkQsOEJBa0ZDIn0=