"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.UplogBuffer = exports.ErrorType = exports.LogType = void 0;
const os_1 = __importDefault(require("os"));
const fs_1 = __importDefault(require("fs"));
const path_1 = __importDefault(require("path"));
const package_json_1 = __importDefault(require("./package.json"));
const platform_folders_1 = require("platform-folders");
const lockfile_1 = __importDefault(require("lockfile"));
var LogType;
(function (LogType) {
    LogType[LogType["Request"] = 0] = "Request";
})(LogType = exports.LogType || (exports.LogType = {}));
var ErrorType;
(function (ErrorType) {
})(ErrorType = exports.ErrorType || (exports.ErrorType = {}));
const UplogBufferFilePath = path_1.default.join(platform_folders_1.getCacheFolder(), '.kodo-s3-adapter-sdk', 'uplog-buffer');
const UplogBufferFileLockPath = path_1.default.join(platform_folders_1.getCacheFolder(), '.kodo-s3-adapter-sdk', 'uplog-buffer.lock');
class UplogBuffer {
    constructor(option) {
        this.option = option;
        fs_1.default.mkdirSync(path_1.default.dirname(UplogBufferFilePath), { recursive: true, mode: 0o700 });
        this.uploadBufferFd = fs_1.default.openSync(UplogBufferFilePath, 'a', 0o600);
        this.uploadBufferedEntries = [];
    }
    log(entry) {
        this.uploadBufferedEntries.push(this.convertUplogEntryToJSON(entry) + "\n");
        return this.flushBufferToLogFile();
    }
    flushBufferToLogFile() {
        return new Promise((resolve, reject) => {
            lockfile_1.default.lock(UplogBufferFileLockPath, {}, (err) => {
                if (err) {
                    if (err.code === 'EEXIST') {
                        console.info("locked fail");
                        resolve();
                        return;
                    }
                    else {
                        console.warn("locked fail:", err);
                        reject(err);
                        return;
                    }
                }
                else {
                    console.info("locked successfully");
                    const uploadBufferedEntries = this.uploadBufferedEntries;
                    this.uploadBufferedEntries = [];
                    const writePromise = uploadBufferedEntries.reduce((writePromise, data) => {
                        return writePromise.then(() => {
                            return new Promise((resolve, reject) => {
                                fs_1.default.write(this.uploadBufferFd, data, (err) => {
                                    if (err) {
                                        reject(err);
                                    }
                                    else {
                                        resolve();
                                    }
                                });
                            });
                        }, reject);
                    }, Promise.resolve());
                    writePromise.then(() => {
                        lockfile_1.default.unlock(UplogBufferFileLockPath, (err) => {
                            if (err) {
                                reject(err);
                            }
                            else {
                                resolve();
                            }
                        });
                    }, (err) => {
                        lockfile_1.default.unlock(UplogBufferFileLockPath, () => {
                            reject(err);
                        });
                    });
                }
            });
        });
    }
    exportLogs(f) {
        return new Promise((resolve, reject) => {
            this.flushBufferToLogFile().then(() => {
                lockfile_1.default.lock(UplogBufferFileLockPath, {}, (err) => {
                    if (err) {
                        console.warn("locked fail:", err);
                        reject(err);
                        return;
                    }
                    else {
                        console.info("locked successfully");
                        fs_1.default.open(UplogBufferFilePath, 'r', 0o600, (err, fd) => {
                            if (err) {
                                reject(err);
                                return;
                            }
                            fs_1.default.read(fd, {}, (err, _, buffer) => {
                                if (err) {
                                    fs_1.default.closeSync(fd);
                                    reject(err);
                                    return;
                                }
                                resolve(buffer);
                                fs_1.default.close(fd);
                            });
                        });
                    }
                });
            }, reject);
        });
    }
    convertUplogEntryToJSON(entry) {
        entry.os_name = os_1.default.platform();
        entry.os_version = os_1.default.release();
        entry.sdk_name = this.option.appName;
        entry.sdk_version = this.option.appVersion;
        entry.http_client = `kodo-s3-adapter-sdk/${this.option.mode}`;
        entry.http_client_version = package_json_1.default.version;
        entry.up_time = Math.trunc(Date.now() / 1000);
        return JSON.stringify(entry);
    }
}
exports.UplogBuffer = UplogBuffer;
//# sourceMappingURL=uplog.js.map