angular
  .module("web", [
    "ui.router",
    "ui.bootstrap",
    "ui.codemirror",
    "pascalprecht.translate",
    "ngSanitize",
    "templates"
  ])
  .config([
    "$stateProvider",
    "$urlRouterProvider",
    "$translateProvider",
    function($stateProvider, $urlRouterProvider, $translateProvider) {
      moment.locale("zh-CN");

      $stateProvider
        .state("files", {
          url: "/",
          templateUrl: "main/files/files.html",
          controller: "filesCtrl"
        })
        .state("login", {
          url: "/login",
          templateUrl: "main/auth/login.html",
          controller: "loginCtrl"
        });

      $urlRouterProvider.otherwise("/");

      //i18n
      for (var k in Global.i18n) {
        $translateProvider.translations(k, Global.i18n[k].content);
      }
      $translateProvider.preferredLanguage("zh-CN");

      $translateProvider.useSanitizeValueStrategy("escapeParameters");
    }
  ])
  .run([
    "$rootScope",
    "$translate",
    "Toast",
    function($rootScope, $translate, Toast) {
      $rootScope.openURL = function(url) {
        openExternal(url);
      };

      // //i18n
      var langMap = {};
      var langList = [];
      angular.forEach(Global.i18n, function(v, k) {
        langMap[k] = v;
        langList.push({
          lang: k,
          label: v.label
        });
      });
      var lang = localStorage.getItem("lang") || langList[0].lang;

      $rootScope.langSettings = {
        langList: langList,
        lang: lang,
        changeLanguage: function(key) {
          key = langMap[key] ? key : langList[0].lang;
          $translate.use(key);
          localStorage.setItem("lang", key);
          $rootScope.langSettings.lang = key;

          Toast.success($translate.instant("setup.success")); //'已经设置成功'
        }
      };

      $translate.use(lang);
    }
  ]);

/*
 * {
 *   "regions": [{
 *      id: "regionA",
 *      label: "区域 A", // Optional
 *      endpoint: "https://s3-region-1.localhost.com"
 *   }],
 *   "uc_url": "https://uc.qbox.me"
 * }
 */

angular.module("web").factory("Config", ["$translate", "$timeout", "$q", "AuthInfo", "Toast",
    ($translate, $timeout, $q, AuthInfo, Toast) => {
        class ConfigError extends Error { }
        class ConfigParseError extends Error { }

        const fs = require('fs'),
              path = require('path'),
              { Region } = require('kodo-s3-adapter-sdk'),
              T = $translate.instant,
              configFilePath = path.join(Global.config_path, 'config.json');
        let configCache = undefined;

        return {
            load: load,
            save: save,
            exists: exists,
        };

        function load(loadDefault) {
            const result = {};

            if (loadDefault === undefined || loadDefault === null) {
                loadDefault = AuthInfo.usePublicCloud();
            }

            if (!loadDefault) {
                try {
                    if (fs.existsSync(configFilePath)) {
                        let config = null;

                        if (configCache) {
                            config = configCache;
                        } else {
                            try {
                                config = JSON.parse(fs.readFileSync(configFilePath));
                                configCache = config;
                            } catch (e) {
                                throw new ConfigParseError(e.message);
                            }
                        }
                        if (config.uc_url) {
                            result.ucUrl = config.uc_url;
                        } else {
                            throw new ConfigError("uc_url is missing or empty");
                        }
                        if (config.regions && config.regions.length) {
                            config.regions.forEach((region) => {
                                if (!region.id) {
                                    throw new ConfigError('id is missing or empty in region');
                                }
                                if (!region.endpoint) {
                                    throw new ConfigError('endpoint is missing or empty in region');
                                }
                            });
                            result.regions = config.regions.map((r) => {
                                const region = new Region('', r.id, r.label);
                                region.ucUrls = [config.uc_url];
                                region.s3Urls = [r.endpoint];
                                return region;
                            });
                        }
                    }
                } catch (e) {
                    if (e instanceof ConfigParseError) {
                        Toast.error(T('config.parse.error'));
                    } else if (e instanceof ConfigError) {
                        Toast.error(T('config.format.error'));
                    }
                    throw e;
                }
            }

            return result;
        }

        function save(ucUrl, regions) {
            if (!ucUrl) {
                throw new ConfigError('ucUrl is missing or empty');
            }

            const newConfig = { uc_url: ucUrl };

            if (regions && regions.length > 0) {
                newConfig.regions = regions.map((region) => {
                    if (!region.s3Id) {
                        throw new ConfigError('id is missing or empty in region');
                    }
                    if (!region.s3Urls || region.s3Urls.length === 0) {
                        throw new ConfigError('endpoint is missing or empty in region');
                    }
                    return { id: region.s3Id, label: region.label, endpoint: region.s3Urls[0] };
                });
            }
            configCache = newConfig;
            fs.writeFileSync(configFilePath, JSON.stringify(newConfig, null, 4), { mode: 0o600 });
        }

        function exists() {
            return fs.existsSync(configFilePath);
        }
    }
]);

angular.module("web").factory("Const", [
  function() {
    return {
      AUTH_INFO_KEY: "auth-info",
      CLOUD_CHOICE_KEY: "cloud-choice",
      EMPTY_FOLDER_UPLOADING: "empty-folder-uploading",
      OVERWRITE_UPLOADING: "overwrite-uploading",
      OVERWRITE_DOWNLOADING: "overwrite-downloading",

      REG: {
        EMAIL: /^\w+([\.-]?\w+)*@\w+([\.-]?\w+)*(\.\w{2,3})+$/
      },

      bucketACL: [
        { acl: "public-read", label: "公共读" }, // 目前仅支持公共读
      ]
    };
  }
]);

angular.module("web").factory("Customize", [
  function() {
    return {
      disable: {
        createBucket: false,
        deleteBucket: false
      },
      upgrade: {
        // Release Notes 目录后缀，里面有 ${version}.md, 如 1.0.0.md
        release_notes_url: "https://kodo-toolbox.qiniu.com/kodobrowser/release-notes/",

        // 升级检测链接
        check_url: "https://kodo-toolbox.qiniu.com/kodobrowser/update.json"
      }
    }
  }
]);

angular.module("web").controller("mainCtrl", [
  "$scope",
  "$timeout",
  "autoUpgradeSvs",
  function (
    $scope,
    $timeout,
    autoUpgradeSvs
  ) {
    angular.extend($scope, {
      upgradeInfo: {
        isLastVersion: true
      }
    });
    load();

    function load() {
      autoUpgradeSvs.load(function(upgradeInfo, success) {
        $scope.$apply(() => {
          $scope.upgradeInfo = upgradeInfo;
        });
        if (success) {
          $timeout(load, 86400 * 1000);
        } else {
          $timeout(load, 30 * 1000);
        }
      });
    }
  }
]);

angular
  .module("web")
  .directive("autoHeight", [
    "$timeout",
    function ($timeout) {
      return {
        link: linkFn,
        restrict: "EA",
        transclude: false,
        scope: {
          autoHeight: "="
          //bottomLoader: '&'
        }
      };

      function linkFn(scope, ele, attr) {
        var h = parseInt(scope.autoHeight);

        ele.css({
          overflow: "auto",
          position: "relative"
        });

        var tid;

        function resize() {
          $timeout.cancel(tid);
          tid = $timeout(function () {
            var v = $(window).height() + h;
            $(ele).height(v);
          }, 300);
        }

        $(window).resize(resize);
        resize();
      }
    }
  ]);
angular
  .module("web")
  .directive("bottomLoader", [
    "$timeout",
    function ($timeout) {
      return {
        link: linkFn,
        restrict: "EA",
        transclude: false,
        scope: {
          bottomLoader: "&"
        }
      };

      function linkFn(scope, ele, attr) {
        ele.css({
          overflow: "auto",
          position: "relative"
        });

        var scrollTid;

        function onScroll() {
          $timeout.cancel(scrollTid);

          scrollTid = $timeout(function () {
            if ($(ele)[0].scrollHeight > 0 &&
              $(ele).parent().height() + $(ele).scrollTop() + 10 >= $(ele)[0].scrollHeight
            ) {
              scope.bottomLoader();
            }
          }, 500);
        }

        onScroll();

        $(window).resize(onScroll);
        $(ele).scroll(onScroll);
      }
    }
  ]);
/*
 <input type="text" ng-model="abc" cleanable-input x="-3" y="-5"/>
 */

angular
  .module("web")
  .directive("cleanableInput", [
    "$timeout",
    function ($timeout) {
      return {
        restrict: "EA",
        require: "ngModel",

        scope: {
          model: "=ngModel",
          ngChange: "&",
          x: "=",
          y: "="
        },
        link: function link(scope, element) {
          var id = "cleanable_inp-" + (Math.random() + "").substring(2);

          element.wrap(
            '<div id="' + id + '" style="position:relative;width:100%;"></div>'
          );
          var btn = $(
            '<a href="" style="font-size:14px;color:#999">' +
            '<i class="glyphicon glyphicon-remove-circle"></i></a>'
          ).appendTo($("#" + id));

          btn
            .css({
              display: "none",
              position: "absolute",
              "z-index": 10
            })
            .click(function (e) {
              scope.model = "";
              if (!scope.$root.$$phase) {
                scope.$apply();
              }

              if (scope.ngChange) {
                scope.$eval(scope.ngChange);
              }

              return false;
            });

          var y = isNaN(scope.y) ? 0 : parseInt(scope.y);
          var x = isNaN(scope.x) ? 0 : parseInt(scope.x);

          function onchange(v) {
            if (v && v !== "") {
              btn
                .css({
                  top: 6 + y,
                  right: 6 - x
                })
                .show();
            } else {
              btn.hide();
            }
          }

          // Listen for any changes to the original model.
          var tid;
          scope.$watch(
            "model",
            function alteredValues(newValue, oldValue) {
              $timeout.cancel(tid);
              tid = $timeout(function () {
                onchange(newValue);
              }, 300);
            },
            true
          );
        }
      };
    }
  ]);
/*
 <input type="text" ng-model="abc" cleanable-input x="-3" y="-5"/>
 */

angular
  .module("web")
  .directive("clipboardButton", [
    "$translate",
    "Toast",
    function ($translate, Toast) {
      var T = $translate.instant;

      return {
        restrict: "EA",
        scope: {
          action: "=",
          target: "=",
          success: "&"
        },
        link: function link(scope, ele) {
          var d = new Clipboard(ele[0], {
            text: function () {
              return $(scope.target).val();
            },
            action: scope.action || "copy"
          });

          d.on("success", function () {
            Toast.success(T("copy.successfully")); //'复制成功'
          });
        }
      };
    }
  ]);
angular
  .module("web")
  .directive("dropZone", function () {
    return {
      link: linkFn,
      restrict: "EA",
      transclude: false,
      scope: {
        dropZone: "="
      }
    };

    function linkFn(scope, ele, attr) {
      $(document).on("dragenter", stopPrev)
        .on("dragover", stopPrev)
        .on("dragleave", stopPrev)
        .on("drop", stopPrev);

      function stopPrev(e) {
        e.originalEvent.stopPropagation();
        e.originalEvent.preventDefault();
      }

      $(ele).on("drop", function(e) {
        scope.dropZone(e);
      });
    }
  });

angular
  .module("web")
  .directive("fileDialogButton", function () {
    return {
      link: linkFn,
      restrict: "EA",
      transclude: false,
      scope: {
        fileChange: "="
      }
    };

    function linkFn(scope, ele, attr) {
      $(ele).on("change", function (e) {
        scope.fileChange.call({}, e.target.files);
      });
    }
  });
angular
  .module("web")
  .directive("flvPlayer", [
    "$timeout",
    function ($timeout) {
      return {
        link: linkFn,
        restrict: "EA",
        transclude: false,
        scope: {
          src: "=",
          autoplay: "=" //autoplay
        }
      };

      function linkFn(scope, ele, attr) {
        scope.$watch("src", init);

        function init() {
          if (!scope.src) return;
          var src =
            "http://localhost:" +
            Global.staticServerPort +
            "/flv-player.html?src=" +
            encodeURIComponent(scope.src) +
            "&autoplay=" +
            (scope.autoplay || "");
          ele.html(
            '<iframe scrolling="no" style="border:0;width:100%;height:460px" src="' +
            src +
            '"><iframe>'
          );
        }
      }
    }
  ]);
angular.module("web").directive("isCloudConfigured", () => {
  return {
    restrict: 'A',
    require: 'ngModel',
    link: (scope, element, attributes, controller) => {
      controller.$validators.isCloudConfigured = promise => {
        promise.then((value) => {
          if (value) {
            element.siblings('i').removeClass('blinking');
            controller.$setValidity('isCloudConfigured', true);
          } else {
            element.siblings('i').addClass('blinking');
            controller.$setValidity('isCloudConfigured', false);
          }
        });
      };
    }
  };
});

angular
  .module("web")
  .directive("isLoading", function () {
    return {
      templateUrl: "components/directives/is-loading.html"
    };
  });
angular.module("web").directive("isMultipleOfFour", () => {
  return {
    restrict: 'A',
    require: 'ngModel',
    link: (scope, element, attrs, controller) => {
      controller.$validators.isMultipleOfFour = value => {
        if (!value) {
            return true;
        }
        return value % 4 == 0;
      };
    }
  };
});

angular
  .module("web")
  .directive("longScrollList", [
    "$timeout",
    function ($timeout) {
      return {
        restrict: "EA",
        transclude: true,
        scope: {
          loadMoreFn: "=loadMore",
          triggerSize: "="
        },
        template: "<div ng-transclude></div>",

        link: function (scope, ele, attr) {
          var t = ele.offset().top;
          var h = $(ele).height();
          var size = scope.triggerSize || 20;

          $(ele).scroll(onScroll);

          var tid;

          function onScroll() {
            $timeout.cancel(tid);
            tid = $timeout(function () {
              effect();
            }, 200);
          }

          function effect() {
            var scrollTop = $(ele).scrollTop();
            var scrollHeight = $(ele)[0].scrollHeight;

            if (scrollTop + h > scrollHeight - size) {
              if (typeof scope.loadMoreFn == "function") scope.loadMoreFn();
            }

          }
        }
      };
    }
  ]);
angular
  .module("web")
  .directive("noData", function () {
    return {
      templateUrl: "components/directives/no-data.html"
    };
  });
angular
  .module("web")
  .directive("qrcode", [
    "$timeout",
    function ($timeout) {
      return {
        link: linkFn,
        restrict: "EA",
        transclude: true,
        template: '<div class="qrcode"><div></div></div><ng-transclude></ng-transclude>',
        scope: {
          text: "=",
          width: "=",
          height: "=",
          label: "="
        }
      };

      function linkFn(scope, ele, attr) {
        scope.$watch("text", function (v) {
          reset(v);
        });

        function reset(v) {
          var w = scope.width || scope.height || 100;
          var h = scope.height || scope.width || 100;
          $(ele)
            .find(".qrcode")
            .html("<div></div>");
          if (v)
            $($(ele).find(".qrcode>div")).qrcode({
              text: v || "",
              width: w,
              height: h
            });
        }
      }
    }
  ]);
/*
usage:

step 1. add element to body:
  <toast-list></toast-list>

step 2: use Toast factory
  Toast.info('test');
*/

angular
  .module("web")
  .directive("toastList", function () {
    return {
      restrict: "EA",
      templateUrl: "components/directives/toast-list.html",
      controller: [
        "$scope",
        "$timeout",
        "$location",
        function ($scope, $timeout, $location) {
          $scope.alerts = [];

          $scope.$on("message", function (evt, data) {
            showMessage(data.message, data.type || "danger", data.ttl || 3000);
          });

          function showMessage(msg, type, ttl) {
            var obj = {
              type: type || "danger",
              msg: msg || "",
              id: Math.random()
            };

            //next tick
            $timeout(function () {
              $scope.alerts.push(obj);
              $timeout(function () {
                for (var i = 0; i < $scope.alerts.length; i++) {
                  if ($scope.alerts[i] == obj) {
                    $scope.alerts.splice(i, 1);
                    break;
                  }
                }
              }, ttl || 3000);
            }, 0);
          }
        }
      ]
    };

    function linkFn(scope, ele, attr) {}
  })
  .factory("Toast", [
    "$rootScope",
    function ($rootScope) {
      return {
        success: function (msg, ttl) {
          sendMessage(msg, "success", ttl);
        },
        info: function (msg, ttl) {
          sendMessage(msg, "info", ttl);
        },
        warn: function (msg, ttl) {
          sendMessage(msg, "warning", ttl);
        },
        warning: function (msg, ttl) {
          sendMessage(msg, "warning", ttl);
        },
        error: function (msg, ttl) {
          sendMessage(msg, "danger", ttl);
        }
      };

      function sendMessage(msg, type, ttl) {
        $rootScope.$broadcast("message", {
          message: msg,
          type: type,
          ttl: ttl
        });
      }
    }
  ]);
"use strict";

angular.module("web")
  .filter("sub", function () {
    return function (s, len) {
      if (!s) {
        return "";
      }

      if (s.length < len) return s;
      else return s.substring(0, len) + "...";
    };
  })
  .filter("hideSecret", function () {
    return function (s) {
      if (!s) {
        return "";
      }

      if (s.length < 6) return "******";
      else return s.substring(0, 3) + "****" + s.substring(s.length - 3);
    };
  })
  .filter("timeFormat", function () {
    return function (d, de) {
      de = de || "";
      try {
        if (!d) return de;
        var s = new Date(d);
        if (s == "Invalid date") {
          return de;
        }
        return moment(s).format("YYYY-MM-DD HH:mm:ss");
      } catch (e) {
        return de;
      }
    };
  })
  .filter("elapse", function () {
    return function (st, et) {
      et = et || new Date().getTime();

      var ms = et - st;

      if (isNaN(ms)) {
        return "";
      }
      if (ms <= 0) return 0;
      else if (ms < 1000) return ms + "ms";

      //return moment.duration(ms).humanize();
      var t = [];
      var h = Math.floor(ms / 3600 / 1000);
      if (h) {
        ms = ms - h * 3600 * 1000;
        t.push(h + "h");
      }
      var m = Math.floor(ms / 60 / 1000);
      if (m) {
        ms = ms - m * 60 * 1000;
        t.push(m + "m");
      }
      var s = Math.floor(ms / 1000);
      if (s) {
        ms = ms - s * 1000;
        t.push(s + "s");
      }
      return t.join("");
    };
  })
  .filter("leftTimeFormat", [
    "utilSvs",
    function (utilSvs) {
      return function (ms) {
        return utilSvs.leftTime(ms);
      };
    }
  ])
  .filter("sizeFormat", function () {
    return function (n, ex) {
      if (n == 0) return 0;
      if (!n) return "0";

      var t = [];
      var left = n;
      var gb = Math.floor(n / Math.pow(1024, 3));
      if (gb > 0) {
        if (ex) {
          t.push(gb + "G");
          left = left % Math.pow(1024, 3);
        } else {
          return Math.round(n * 100 / Math.pow(1024, 3)) / 100 + "GB";
        }
      }

      var mb = Math.floor(left / Math.pow(1024, 2));
      if (mb > 0) {
        if (ex) {
          t.push(mb + "M");
          left = left % Math.pow(1024, 2);
        } else {
          return Math.round(100 * left / Math.pow(1024, 2)) / 100 + "MB";
        }
      }

      var kb = Math.floor(left / 1024);
      if (kb > 0) {
        if (ex) {
          t.push(kb + "K");
          left = left % 1024;
        } else {
          return Math.round(100 * left / 1024) / 100 + "KB";
        }
      }

      if (left > 0) {
        t.push(left + "B");
        if (!ex) return left + "B";
      }
      return t.length > 0 ? t.join("") : 0;
    };
  })
  .filter("persent", function () {
    return function (a, b, status) {
      if (a == 0 && b == 0) {
        if (status == "finished") {
          return 100;
        } else return 0;
      }
      return Math.floor(a / b * 10000) / 100;
    };
  })
  .filter("statusCls", [
    "jobUtil",
    function (jobUtil) {
      return function (s) {
        return jobUtil.getStatusCls(s);
      };
    }
  ])
  .filter("status", [
    "jobUtil",
    function (jobUtil) {
      return function (s, isUp) {
        return jobUtil.getStatusLabel(s, isUp);
      };
    }
  ])
  .filter("fileIcon", [
    "fileSvs",
    function (fileSvs) {
      return function (item) {
        var info = fileSvs.getFileType(item);

        if (info.type == "folder") return "folder";
        if (info.type == "video") return "file-video-o";
        if (info.type == "audio") return "file-audio-o";
        if (info.type == "picture") return "file-image-o";
        if (info.type == "doc") {
          switch (info.ext[0]) {
          case "doc":
          case "docx":
            return "file-word-o";
          case "pdf":
            return "file-pdf-o";
          case "ppt":
          case "pptx":
            return "file-powerpoint-o";
          case "exl":
            return "file-excel-o";
          }
          return "file-o";
        }
        if (info.type == "code") return "file-text-o";
        if (info.type == "others") {
          switch (info.ext[0]) {
          case "gz":
          case "tar":
          case "zip":
          case "jar":
          case "bz":
          case "war":
          case "xz":
            return "file-zip-o";
          case "pkg":
            return "dropbox";
          case "app":
          case "dmg":
            return "apple";
          case "apk":
            return "android";
          case "msi":
          case "deb":
          case "bin":
          case "exe":
            return "cog";
          case "img":
          case "iso":
            return "dot-circle-o";
          case "cmd":
          case "sh":
            return "terminal";
          }
        }
        return "file-o";
      };
    }
  ])
  .filter("htmlEscape", function () {
    return function (html) {
      return html.toString().replace(/[\u00A0-\u9999<>\&\'\"]/gim, function(char) {
        return '&#' + char.charCodeAt(0) + ';';
      });
    };
  });

"use strict";

angular.module("web").filter("listFilter", function() {
  return function(arr, keyFn, value) {
    if (!value) return arr;
    if (arr && arr.length > 0) {
      var t = [];
      if (typeof keyFn == "string") {
        angular.forEach(arr, function(n) {
          if (n[keyFn].indexOf(value) != -1) {
            t.push(n);
          }
        });
      } else if (typeof keyFn == "function") {
        angular.forEach(arr, function(n) {
          if (keyFn(n).indexOf(value) != -1) {
            t.push(n);
          }
        });
      }
      return t;
    }
    return [];
  };
});

angular.module("web").factory("AkHistory", [function() {
    const fs = require('fs'),
          path = require('path');

    class AkHistory {
        constructor(isPublicCloud, accessKeyId, accessKeySecret, description) {
            this.isPublicCloud = isPublicCloud;
            this.accessKeyId = accessKeyId;
            this.accessKeySecret = accessKeySecret;
            this.description = description;
        }
    }

    return {
        list: list,
        add: add,
        remove: remove,
        clearAll: clearAll,
    };

    function list() {
        const filePath = getFilePath();
        try {
            fs.accessSync(filePath, fs.constants.R_OK);
        } catch (err) {
            return [];
        }
        const data = fs.readFileSync(filePath, 'utf8');
        return JSON.parse(data).historyItems || [];
    }

    function add(isPublicCloud, accessKeyId, accessKeySecret, description) {
        const histories = list();
        let found = false;
        for (let i = 0; i < histories.length; i++) {
            if (histories[i].accessKeyId === accessKeyId) {
                histories[i].isPublicCloud = isPublicCloud;
                histories[i].accessKeySecret = accessKeySecret;
                histories[i].description = description;
                found = true;
                break;
            }
        }
        if (!found) {
            histories.push(new AkHistory(isPublicCloud, accessKeyId, accessKeySecret, description));
        }
        writeHistories(histories);
    }

    function remove(accessKeyId) {
        const histories = list();
        for (let i = 0; i < histories.length; i++) {
            if (histories[i].accessKeyId === accessKeyId) {
                histories.splice(i, 1);
                break;
            }
        }
        writeHistories(histories);
    }

    function clearAll() {
        writeHistories([]);
    }

    function getFilePath() {
        const folder = Global.config_path;

        if (!fs.existsSync(folder)) {
            fs.mkdirSync(folder, { recursive: true });
        }

        return path.join(folder, 'ak_histories.json');
    }

    function writeHistories(histories) {
        fs.writeFileSync(getFilePath(), JSON.stringify({historyItems: histories}));
    }
}]);

angular.module("web").factory("AuditLog", [
    "AuthInfo",
    function(AuthInfo) {
        const fs = require('fs'),
              path = require('path'),
              moment = require('moment'),
              expirationMonths = 3;

        return {
            log: log,
        }

        function log(action, params, options) {
            options = options || {};
            fs.appendFileSync(getFilePath(), JSON.stringify({
                time: new Date(),
                appVersion: Global.app.version,
                logVersion: options.logVersion || 1,
                accessKeyId: AuthInfo.get().id,
                action: action,
                params: params || {},
            }) + "\n");
        }

        function getFilePath() {
            const folderPath = path.join(Global.config_path, 'logs');
            const logFilePath = path.join(folderPath, `audit_log_${moment().format('YYYY-MM')}.json`);

            if (!fs.existsSync(folderPath)) {
                fs.mkdirSync(folderPath, { recursive: true });
            }
            if (!fs.existsSync(logFilePath)) {
                cleanLogs();
            }

            return logFilePath;
        }

        function cleanLogs() {
            const folderPath = path.join(Global.config_path, 'logs');
            const now = moment();
            let entries = fs.readdirSync(folderPath, { withFileTypes: true });

            entries = entries.filter((entry) => {
                return entry.isFile() && entry.name.startsWith('audit_log_');
            });
            entries.forEach((entry) => {
                const momentRegexp = /^audit_log_(\d{4}\-\d{2})\.json$/;
                const matchResult = entry.name.match(momentRegexp);
                if (matchResult) {
                    if (now.diff(moment(matchResult[1], 'YYYY-MM')) > expirationMonths) {
                        fs.unlinkSync(path.join(folderPath, entry.name));
                    }
                }
            });
        }
    }
]);

angular.module("web").factory("Auth", [
  "$q",
  "$location",
  "$translate",
  "QiniuClient",
  "AuthInfo",
  function ($q, $location, $translate, QiniuClient, AuthInfo) {
    const T = $translate.instant;

    return {
      login: login,
      logout: logout
    };

    function login(data) {
      return new Promise((resolve, reject) => {
        QiniuClient.listAllBuckets(data).then(() => {
          data.isAuthed = true;
          AuthInfo.save(data);
          resolve();
        }).catch((err) => {
          data.isAuthed = false;
          reject(err);
        });
      });
    }

    function logout() {
      return new Promise((resolve) => {
        QiniuClient.clearAllCache();
        const { ipcRenderer } = require('electron');
        AuthInfo.remove();
        ipcRenderer.send('asynchronous', { key: 'clearCache' });
        ipcRenderer.send('asynchronous-job', { key: 'job-stopall' });
        resolve();
      });
    }
  }
]);

angular.module("web").factory("AuthInfo", [
  "$q",
  "Const",
  "Cipher",
  function ($q, Const, Cipher) {
    var AUTH_INFO = Const.AUTH_INFO_KEY;
    var CLOUD_CHOICE = Const.CLOUD_CHOICE_KEY;

    return {
      get: function () {
        return get(AUTH_INFO);
      },
      save: function (obj) {
        var oldobj = get(AUTH_INFO);
        Object.assign(oldobj, obj);

        save(AUTH_INFO, oldobj);
      },
      remove: function () {
        remove(AUTH_INFO);
      },
      saveToAuthInfo: saveToAuthInfo,

      usePublicCloud: function() {
        return get(CLOUD_CHOICE) === 'default';
      },
      switchToPublicCloud: function() {
        save(CLOUD_CHOICE, 'default');
      },
      switchToPrivateCloud: function() {
        save(CLOUD_CHOICE, 'customized');
      },
    };

    function saveToAuthInfo(opt) {
      var obj = get(AUTH_INFO);
      for (var k in opt) obj[k] = opt[k];
      save(AUTH_INFO, obj);
    }

    ///////////////////////////////
    function remove(key) {
      localStorage.removeItem(key);
    }

    function get(key, defv) {
      var str = localStorage.getItem(key);
      if (str) {
        try {
          str = Cipher.decipher(str);
          return JSON.parse(str);
        } catch (e) {
          console.log(e, str);
        }
      }
      return defv || {};
    }

    function save(key, obj, defv) {
      delete obj["httpOptions"];

      var str = JSON.stringify(obj || defv || {});
      try {
        str = Cipher.cipher(str);
      } catch (e) {
        console.log(e);
      }
      localStorage.setItem(key, str);
    }
  }
]);

angular.module("web").factory("autoUpgradeSvs", [
  "$timeout",
  "Customize",
  function ($timeout, Customize) {
    const NAME = "kodo-browser";
    const util = require("./node/qiniu-store/lib/util");
    const path = require("path");
    const fs = require("fs");
    const request = require("request");
    const downloadsFolder = require("downloads-folder");

    const upgrade_url = Customize.upgrade.check_url;
    const release_notes_url = Customize.upgrade.release_notes_url;
    const gVersion = Global.app.version;

    var upgradeOpt = {
      currentVersion: gVersion,
      isLastVersion: false,
      lastVersion: gVersion,
      fileName: "",
      link: "",
      upgradeJob: {
        pkgLink: "",
        progress: 0,
        status: "waiting"
      }
    };

    return {
      load: load,
      start: start,
      stop: stop,

      compareVersion: compareVersion,
      getLastestReleaseNote: getLastestReleaseNote
    };

    var job;

    function start() {
      if (job) job.start();
    }

    function stop() {
      if (job) job.stop();
    }

    function getLastestReleaseNote(version, fn) {
      if (release_notes_url) {
        $.get(release_notes_url + version + ".md", fn);
      } else {
        $timeout(() => { fn(''); });
      }
    }

    function FlatDownloadJob(name, from, to) {
      console.log("FlatDownloadJob:", from, to);
      this.total = 0;
      this.progress = 0;
      this.name = name;
      this.from = from;
      this.to = to;

      let _statusChangeFn;
      let _progressChangeFn;
      let _request;
      let _stopped;

      this.update = function () {
        //copy
        console.log("copy:", __dirname);
        fs.renameSync(to + ".download", to);
        this._changeStatus("finished");
      };
      this.check = function (expected, callback) {
        //crc
        console.log("etag check");
        return util.getEtag(to + ".download", function(actual) {
          if (expected !== `"${actual}"`) {
            callback(new Error(`Etag check failed, expected: ${expected}, actual: "${actual}"`));
          } else {
            callback();
          }
        });
      };

      this.precheck = function () {
        this.progress = 0;
        this.total = 0;

        if (fs.existsSync(to)) {
          console.log("exists, done");
          this.progress = 100;
          this._changeStatus("finished");
          return false;
        }
        return true;
      };

      this.start = function () {
        if (!this.precheck()) {
          return;
        }
        _stopped = false;

        const that = this;

        console.log("start download ...");
        that._changeStatus("running");

        _request = request
          .head(from)
          .on("error", function (err) {
            console.error(err);
            this._changeStatus("failed", err);
          })
          .on("response", function (response) {
            if (_stopped) {
              return;
            } else if (response.statusCode == 200) {
              that.total = response.headers["content-length"];

              const to_download_target = to + ".download";
              let current = 0;

              if (fs.existsSync(to_download_target)) {
                const stat = fs.statSync(to_download_target);
                if (stat.isFile() && stat.size <= that.total) {
                  current = stat.size;
                  console.log(`resume download from ${current}`);
                } else {
                  fs.unlinkSync(to_download_target);
                }
              }

              that.progress = Math.round(current * 10000 / that.total) / 100;

              var ws = fs.createWriteStream(to_download_target, {
                flags: fs.constants.O_CREAT | fs.constants.O_WRONLY | fs.constants.O_NONBLOCK,
                start: current
              });

              _request = request({url: from, headers: { 'Range': `bytes=${current}-` }})
                .on("error", function (err) {
                  if (_stopped) {
                    return;
                  }
                  console.error(err);
                  that._changeStatus("failed", err);
                })
                .on("data", function (chunk) {
                  if (_stopped) {
                    throw new Error('Manually Stopped, Just ignore this exception');
                  }
                  current += chunk.length;
                  that.progress =
                    Math.round(current * 10000 / that.total) / 100;
                  that._changeProgress(that.progress);
                  return chunk;
                })
                .pipe(ws)
                .on("finish", function () {
                  if (_stopped) {
                    return;
                  }
                  that._changeStatus("verifying");

                  that.check(
                    response.headers["etag"],
                    function (err) {
                      if (err) {
                        console.error("check error:", err);
                        that._changeStatus("failed", err);
                      } else {
                        that.update();
                      }
                    }
                  );
                });
            } else {
              console.error('download upgrade package error', response.statusCode, response.statusMessage, response.headers);
              that._changeStatus("failed", response);
            }
          });
      };
      this.onProgressChange = function (fn) {
        _progressChangeFn = fn;
      };
      this.onStatusChange = function (fn) {
        _statusChangeFn = fn;
      };
      this._changeStatus = function (status, err) {
        this.status = status;
        this.message = err;
        if (_statusChangeFn) _statusChangeFn(status);
      };
      this._changeProgress = function (prog) {
        if (_progressChangeFn) _progressChangeFn(prog);
      };
      this.stop = function() {
        _stopped = true;
        this._changeStatus("stopped");
        if (_request) {
          _request.end();
          _request.destroy();
          _request = null;
        }
      }
    }

    function load(fn) {
      const fallback = {
        currentVersion: Global.app.version,
        isLastVersion: true,
        lastVersion: Global.app.version,
        fileName: "",
        link: "",
        localPath: ""
      };
      if (!upgrade_url) {
        $timeout(() => { fn(fallback, true); });
        return;
      }

      $.ajax(upgrade_url, {
        cache: false,
        method: 'GET',
        dataType: 'json',
        success: function(data) {
          const isLastVersion = compareVersion(gVersion, data.version) >= 0;
          const lastVersion = data.version;
          let downloadUrl = '';

          upgradeOpt.isLastVersion = isLastVersion;
          upgradeOpt.lastVersion = lastVersion;

          if (!isLastVersion && data.downloads) {
            try {
              downloadUrl = data.downloads[process.platform][process.arch];
            } catch (e) {
              console.error(e);
              fn(fallback, true);
              return;
            }

            let jobs = [];
            let fileName = decodeURIComponent(path.basename(downloadUrl));

            upgradeOpt.fileName = fileName;
            upgradeOpt.localPath = path.join(downloadsFolder(), fileName);
            upgradeOpt.link = downloadUrl;
            upgradeOpt.upgradeJob.status = "waiting";
            upgradeOpt.upgradeJob.progress = 0;
            upgradeOpt.upgradeJob.pkgLink = downloadUrl;

            job = new FlatDownloadJob(fileName, downloadUrl, upgradeOpt.localPath);

            job.onStatusChange(function (status) {
              upgradeOpt.upgradeJob.status = status;
            });
            job.onProgressChange(function (progress) {
              upgradeOpt.upgradeJob.progress = progress;
            });
            job.precheck();

            fn(upgradeOpt, true);
          } else {
            fn(fallback, true);
          }
        },
        error: function(xhr, _, error) {
          console.error(error);
          fn(fallback, false);
        },
      });
    }

    function compareVersion(curV, lastV) {
      var arr = curV.split(".");
      var arr2 = lastV.split(".");

      var len = Math.max(arr.length, arr2.length);

      for (var i = 0; i < len; i++) {
        var a = parseInt(arr[i]) || 0;
        var b = parseInt(arr2[i]) || 0;

        if (a > b) {
          return 1;
        } else if (a < b) {
          return -1;
        }
      }
      return 0;
    }
  }
]);

angular.module("web").factory("Bookmark", [
  "AuthInfo",
  function(AuthInfo) {
    const fs = require("fs"),
          path = require("path"),
          moment = require("moment");

    class Bookmark {
      constructor(fullPath, mode, timestamp) {
        this.fullPath = fullPath;
        this.mode = mode;
        this.timestamp = timestamp || moment().unix();
      }

      isBucketsOrFiles() {
        return this.mode.startsWith('local');
      }

      isExternalPathBookmark() {
        return this.mode.startsWith('external');
      }
    }

    return {
      list: list,
      add: add,
      marked: marked,
      remove: remove,
    };

    function add(fullPath, mode) {
      const bookmarks = list();
      let found = false;
      for (let i = 0; i < bookmarks.length; i++) {
        if (bookmarks[i].fullPath === fullPath &&
            bookmarks[i].mode === mode) {
          bookmarks[i].timestamp = moment().unix();
          found = true;
          break;
        }
      }
      if (!found) {
        bookmarks.push(new Bookmark(fullPath, mode));
      }
      writeBookmarks(bookmarks);
    }

    function marked(fullPath, mode) {
      const bookmarks = list();
      for (let i = 0; i < bookmarks.length; i++) {
        if (bookmarks[i].fullPath === fullPath &&
            bookmarks[i].mode === mode) {
          return true;
        }
      }
      return false;
    }

    function remove(fullPath, mode) {
        const bookmarks = list();
        for (let i = 0; i < bookmarks.length; i++) {
          if (bookmarks[i].fullPath === fullPath &&
              bookmarks[i].mode === mode) {
                bookmarks.splice(i, 1);
                break;
            }
        }
        writeBookmarks(bookmarks);
    }

    function list() {
        const filePath = getFilePath();
        try {
            fs.accessSync(filePath, fs.constants.R_OK);
        } catch (err) {
            return [];
        }
        const data = fs.readFileSync(filePath, 'utf8');
        let bookmarks = JSON.parse(data).bookmarks || [];

        bookmarks = bookmarks.map((bookmark) => {
          return new Bookmark(bookmark.fullPath, bookmark.mode, bookmark.timestamp);
        });

        return bookmarks;
      }

    function getFilePath() {
        const folder = Global.config_path;

        if (!fs.existsSync(folder)) {
            fs.mkdirSync(folder, { recursive: true });
        }

        const username = AuthInfo.get().id || 'kodo-browser';
        return path.join(folder, `bookmarks_${username}.json`);
    }

    function writeBookmarks(bookmarks) {
        fs.writeFileSync(getFilePath(), JSON.stringify({bookmarks: bookmarks}));
    }
  }
]);

angular.module("web").factory("Cipher", function () {
  var crypto = require("crypto");
  var ALGORITHM = "aes-256-cbc";
  var KEY = "x82m#*lx8vv";

  return {
    cipher: cipher,
    decipher: decipher
  };

  function cipher(buf, key, algorithm) {
    if (!buf instanceof Buffer) {
      buf = new Buffer(buf);
    }
    var encrypted = "";
    var cip = crypto.createCipher(algorithm || ALGORITHM, key || KEY);
    encrypted += cip.update(buf, "utf8", "hex");
    encrypted += cip.final("hex");
    return encrypted;
  }

  function decipher(encrypted, key, algorithm) {
    var decrypted = "";
    var decipher = crypto.createDecipher(algorithm || ALGORITHM, key || KEY);
    decrypted += decipher.update(encrypted, "hex", "utf8");
    decrypted += decipher.final("utf8");
    return decrypted;
  }
});
angular.module("web").factory("DelayDone", [
  "$timeout",
  function ($timeout) {
    var mDelayCall = {};

    return {
      delayRun: delayRun,
      seriesRun: seriesRun
    };

    /**
     * @param id {String}  uniq
     * @param timeout {int}  ms
     * @param times {int}  超过次数也会调, 然后重新统计
     * @param fn {Function}  callback
     */

    function delayRun(id, timeout, fn, times) {
      if (!mDelayCall[id])
        mDelayCall[id] = {
          tid: "",
          c: 0
        };
      var n = mDelayCall[id];

      n.c++;

      if (n.c >= times) {
        fn();
        n.c = 0;
      } else {
        $timeout.cancel(n.tid);
        n.tid = $timeout(fn, timeout);
      }
    }

    function seriesRun(arr, fn, doneFn) {
      var len = arr.length;
      var c = 0;

      function _dig() {
        var n = arr[c];
        fn(n, function () {
          c++;
          if (c >= len) {
            doneFn();
          } else {
            $timeout(function () {
              _dig();
            }, 1);
          }
        });
      }
      _dig();
    }
  }
]);
angular
  .module("web")
  .factory("Dialog", [
    "$uibModal",
    function ($modal) {
      var dialog = require("electron").remote.dialog;

      return {
        alert: alert,
        confirm: confirm,

        showUploadDialog: showUploadDialog,
        showDownloadDialog: showDownloadDialog
      };

      function showUploadDialog(fn) {
        var isMac = navigator.userAgent.indexOf("Macintosh") != -1;

        dialog.showOpenDialog({
            title: "Upload",
            properties: isMac ?
              ["openFile", "openDirectory", "multiSelections"] :
              ["openFile", "multiSelections"]
          },
          function (filePaths) {
            if (typeof fn == "function") fn(filePaths);
          }
        );
      }

      function showDownloadDialog(fn) {
        dialog.showOpenDialog({
            title: "Download",
            properties: ["openDirectory"]
          },
          function (filePaths) {
            if (typeof fn == "function") fn(filePaths);
          }
        );
      }

      /**
       *
       * @param title
       * @param msg
       * @param fn
       * @param opt
       *    opt.cls: danger success warning info,
       *    opt.hideIcon:     default: false
       */
      function alert(title, msg, fn, opt) {
        opt = opt || {
          cls: "primary"
        };
        if (typeof opt == "number") {
          switch (opt) {
          case 2:
            opt = {
              cls: "warning"
            };
            break;
          case 1:
            opt = {
              cls: "danger"
            };
            break;
          default:
            opt = {
              cls: "primary"
            };
            break;
          }
        } else {
          opt = Object.assign({
            cls: "primary"
          }, opt);
        }
        var putData = {
          title: title,
          message: msg,
          opt: opt,
          callback: fn || function (flag) {}
        };

        $modal.open({
          templateUrl: "components/services/dialog.html",
          controller: "alertDialogCtrl",
          size: opt.size || "md",
          resolve: {
            putData: function () {
              return putData;
            }
          }
        }).result.then(angular.noop, angular.noop);
      }

      function confirm(title, msg, fn, opt) {
        opt = opt || {
          cls: "primary"
        };
        if (typeof opt == "number") {
          switch (opt) {
          case 2:
            opt = {
              cls: "warning"
            };
            break;
          case 1:
            opt = {
              cls: "danger"
            };
            break;
          default:
            opt = {
              cls: "primary"
            };
            break;
          }
        } else {
          opt = Object.assign({
            cls: "primary"
          }, opt);
        }
        var putData = {
          title: title,
          message: msg,
          opt: opt,
          callback: fn || function (flag) {}
        };

        $modal.open({
          templateUrl: "components/services/dialog.html",
          controller: "confirmDialogCtrl",
          size: opt.size || "md",
          resolve: {
            putData: function () {
              return putData;
            }
          }
        }).result.then(angular.noop, angular.noop);
      }
    }
  ])
  .controller("alertDialogCtrl", [
    "$scope",
    "$uibModalInstance",
    "putData",
    function ($scope, $modalInstance, putData) {
      angular.extend($scope, putData);
      $scope.isAlert = true;
      $scope.cancel = function () {
        $modalInstance.dismiss("cancel");
        putData.callback(false);
      };
      $scope.ok = function () {
        $modalInstance.dismiss("cancel");
        putData.callback(true);
      };
    }
  ])
  .controller("confirmDialogCtrl", [
    "$scope",
    "$uibModalInstance",
    "putData",
    function ($scope, $modalInstance, putData) {
      angular.extend($scope, putData);
      $scope.cancel = function () {
        $modalInstance.dismiss("cancel");
        putData.callback(false);
      };
      $scope.ok = function () {
        $modalInstance.dismiss("cancel");
        putData.callback(true);
      };
    }
  ]);
"use strict";

angular
  .module("web")
  .factory("DiffModal", [
    "$uibModal",
    function ($modal) {
      return {
        /**
         * @param title
         * @param originalContent
         * @param content
         * @param callback
         * @param editable
         */
        show: function (title, originalContent, content, callback, editable) {
          editable = editable === false ? false : true;

          $modal.open({
            templateUrl: "components/services/diff-modal.html",
            controller: "diffModalCtrl",
            size: "lg",
            resolve: {
              title: function () {
                return title || "Diff";
              },
              editable: function () {
                return editable;
              },
              originalContent: function () {
                return originalContent;
              },
              content: function () {
                return content;
              },
              callback: function () {
                return function (v) {
                  if (editable) callback(v);
                  else callback();
                };
              }
            }
          }).result.then(angular.noop, angular.noop);
        }
      };
    }
  ])
  .controller("diffModalCtrl", [
    "$scope",
    "$uibModalInstance",
    "$timeout",
    "title",
    "editable",
    "originalContent",
    "content",
    "callback",
    function (
      $scope,
      $modalInstance,
      $timeout,
      title,
      editable,
      originalContent,
      content,
      callback
    ) {
      angular.extend($scope, {
        title: title || "Diff",
        originalContent: originalContent,
        content: content,
        initUI: initUI,
        editable: editable,

        ok: ok,
        cancel: cancel
      });

      var editor;

      function initUI() {
        $timeout(function () {
          editor = CodeMirror.MergeView(document.getElementById("diff-view"), {
            value: content,
            origLeft: originalContent,
            //orig:  content,
            lineNumbers: true,
            mode: "javascript",
            highlightDifferences: true,
            connect: "align",
            collapseIdentical: true,

            //不可编辑
            allowEditingOriginals: false,
            revertButtons: false
          });
        }, 100);
      }

      function cancel() {
        $modalInstance.dismiss("close");
      }

      function ok() {
        callback(editor.editor().getValue());
        $modalInstance.dismiss("close");
      }
    }
  ]);
angular.module("web").factory("Domains", [
  "$q",
  "$timeout",
  "$translate",
  "AuthInfo",
  "QiniuClient",
  function (
    $q,
    $timeout,
    $translate,
    AuthInfo,
    QiniuClient,
  ) {
    const T = $translate.instant,
      { KODO_MODE, S3_MODE } = require('kodo-s3-adapter-sdk');

    class S3Domain {
      constructor(region, bucket) {
        this.region = region;
        this.bucket = bucket;
      }

      default() {
        return true;
      }

      name() {
        return T('no.owned.domain');
      }

      toQiniuDomain() {
        return undefined;
      }

      qiniuBackendMode() {
        return S3_MODE;
      }

      signatureUrl(key, expires, opt) {
        expires = expires || this.maxLifetime();
        const newOpt = Object.assign({}, opt, { preferS3Adapter: true });
        return QiniuClient.signatureUrl(this.region, this.bucket, key, undefined, expires, newOpt);
      }

      getContent(key, opt) {
        const newOpt = Object.assign({}, opt, { preferS3Adapter: true });
        return QiniuClient.getContent(this.region, this.bucket, key, this.toQiniuDomain(), newOpt);
      }

      saveContent(key, content, opt) {
        const getOpt = Object.assign({}, opt, { preferS3Adapter: true });
        return QiniuClient.saveContent(this.region, this.bucket, key, content, this.toQiniuDomain(), getOpt, opt);
      }

      deadlineRequired() {
        return true;
      }

      maxLifetime() {
        return 24 * 60 * 60 * 7;
      }
    }

    class KodoDomain {
      constructor(region, bucket, domain) {
        this.region = region;
        this.bucket = bucket;
        this.domain = domain;
      }

      default() {
        return false;
      }

      name() {
        return this.domain.name;
      }

      toQiniuDomain() {
        return this.domain;
      }

      qiniuBackendMode() {
        return KODO_MODE;
      }

      signatureUrl(key, expires, opt) {
        expires = expires || this.maxLifetime();
        const newOpt = Object.assign({}, opt, { preferKodoAdapter: true });
        return QiniuClient.signatureUrl(this.region, this.bucket, key, this.domain, expires, newOpt);
      }

      getContent(key, opt) {
        const newOpt = Object.assign({}, opt, { preferKodoAdapter: true });
        return QiniuClient.getContent(this.region, this.bucket, key, this.toQiniuDomain(), newOpt);
      }

      saveContent(key, content, opt) {
        const getOpt = Object.assign({}, opt, { preferKodoAdapter: true });
        return QiniuClient.saveContent(this.region, this.bucket, key, content, this.toQiniuDomain(), getOpt, opt);
      }

      deadlineRequired() {
        return this.domain.private;
      }

      maxLifetime() {
        return 24 * 60 * 60 * 365;
      }
    }

    return {
      s3: s3,
      list: list
    };

    function s3(region, bucket) {
      return new S3Domain(region, bucket);
    }

    function list(region, bucket, grantedPermissions) {
      return new Promise((resolve, reject) => {
        let allDomains = [];

        if (!grantedPermissions) {
          allDomains.push(new S3Domain(region, bucket));
        }

        if (AuthInfo.usePublicCloud()) {
          QiniuClient.listDomains(region, bucket).then((domains) => {
            allDomains = allDomains.concat(domains.map((domain) => new KodoDomain(region, bucket, domain)));
            resolve(allDomains);
          }, () => {
            resolve(allDomains);
          });
        } else {
          resolve(allDomains);
        }
      });
    }
  }
]);

angular.module("web").factory("DownloadMgr", [
  "$q",
  "$timeout",
  '$translate',
  "AuthInfo",
  "QiniuClient",
  "Config",
  "Toast",
  "settingsSvs",
  function (
    $q,
    $timeout,
    $translate,
    AuthInfo,
    QiniuClient,
    Config,
    Toast,
    settingsSvs
  ) {
    const T = $translate.instant,
          fs = require("fs"),
          http = require("http"),
          https = require("https"),
          pfs = fs.promises,
          path = require("path"),
          os = require("os"),
          sanitize = require("sanitize-filename"),
          QiniuStore = require("./node/qiniu-store"),
          { KODO_MODE } = require('kodo-s3-adapter-sdk');

    var $scope;
    var concurrency = 0;
    var stopCreatingFlag = false;

    return {
      init: init,
      createDownloadJobs: createDownloadJobs,
      trySchedJob: trySchedJob,
      trySaveProg: trySaveProg,

      stopCreatingJobs: () => {
        stopCreatingFlag = true;
      }
    };

    function init(scope) {
      $scope = scope;
      $scope.lists.downloadJobList = [];

      tryLoadProg().then((progs) => {
        angular.forEach(progs, (prog) => {
          const job = createJob(prog);
          if (job.status === "waiting" || job.status === "running") {
            job.stop();
          }
          addEvents(job);
        });
      });
    }

    /**
     * @param  opt { region, from, to, ...}
     * @param  opt.from {bucket, key}
     * @param  opt.to   {name, path}
     * @return job  { start(), stop(), status, progress }
     */
    function createJob(options) {
      const bucket = options.from.bucket,
            key = options.from.key,
            region = options.region,
            domain = options.domain;

      console.info(
        "GET",
        "::",
        region,
        "::",
        bucket + "/" + key,
        "==>",
        options.to.path + "/" + options.to.name
      );

      const config = Config.load();

      options.clientOptions = {
        accessKey: AuthInfo.get().id,
        secretKey: AuthInfo.get().secret,
        ucUrl: config.ucUrl,
        regions: config.regions || [],
      };
      options.region = region;
      options.domain = domain;
      options.resumeDownload = (settingsSvs.resumeDownload.get() == 1);
      options.multipartDownloadThreshold = settingsSvs.multipartDownloadThreshold.get();
      options.multipartDownloadSize = settingsSvs.multipartDownloadSize.get();
      options.downloadSpeedLimit = (settingsSvs.downloadSpeedLimitEnabled.get() == 1 && settingsSvs.downloadSpeedLimitKBperSec.get());
      options.isDebug = (settingsSvs.isDebug.get() == 1);

      const store = new QiniuStore();
      return store.createDownloadJob(options);
    }

    /**
     * 下载
     * @param bucketInfos {array}  item={region, bucket, path, name, size=0, itemType='file'}  有可能是目录，需要遍历
     * @param toLocalPath {string}
     * @param jobsAddedFn {Function} 加入列表完成回调方法， jobs列表已经稳定
     */
    function createDownloadJobs(bucketInfos, toLocalPath, jobsAddedFn) {
      stopCreatingFlag = false;

      const dirPath = bucketInfos[0].path.parentDirectoryPath();

      loop(bucketInfos, (jobs) => {}, () => {
        if (jobsAddedFn) {
          jobsAddedFn();
        }
      });

      function loop(arr, callFn, callFn2) {
        const t = [];
        const len = arr.length;
        let c = 0;
        let c2 = 0;

        if (len == 0) {
          callFn(t);
          callFn2(t);
          return;
        }

        _kdig();

        function _kdig() {
          dig(arr[c], t, () => {

          }, () => {
            c2++;
            if (c2 >= len) {
              callFn2(t);
            }
          });

          c++;
          if (c == len) {
            callFn(t);
          } else {
            if (stopCreatingFlag) {
              return;
            }

            $timeout(_kdig, 0);
          }
        }
      }

      function dig(qiniuInfo, t, callFn, callFn2) {
        if (stopCreatingFlag) {
          return;
        }

        const fileName = sanitize(qiniuInfo.path.basename() || qiniuInfo.path.directoryBasename());
        let filePath = '';
        if (path.sep == '\\') {
          angular.forEach(path.relative(dirPath.toString().replace(/\\/g, '/'), qiniuInfo.path.toString()).replace(/\\/g, '/').split('/'), (folder) => {
            filePath = path.join(filePath, sanitize(folder));
          });
        } else {
          angular.forEach(path.relative(dirPath.toString(), qiniuInfo.path.toString()).split('/'), (folder) => {
            filePath = path.join(filePath, sanitize(folder));
          });
        }

        if (qiniuInfo.itemType === 'folder') {
          // list all files under qiniuInfo.path
          function tryLoadFiles(marker) {
            QiniuClient
              .listFiles(qiniuInfo.region, qiniuInfo.bucket, qiniuInfo.path.toString(), marker, {
                maxKeys: 1000,
                minKeys: 0,
              })
              .then((result) => {
                var files = result.data;
                files.forEach((f) => {
                  f.region = qiniuInfo.region;
                  f.bucket = qiniuInfo.bucket;
                  f.domain = qiniuInfo.domain;
                  f.qiniuBackendMode = qiniuInfo.qiniuBackendMode;
                });

                loop(files, (jobs) => {
                  t = t.concat(jobs);
                  if (result.marker) {
                    $timeout(() => { tryLoadFiles(result.marker); }, 10);
                  } else {
                    if (callFn) callFn();
                  }
                }, callFn2);
              });
          }

          if (!qiniuInfo.path.directoryBasename()) {
            Toast.error(T('download.emptyNameFolder.forbidden', { path: qiniuInfo.path.toString() }));
            return;
          }

          tryLoadFiles();
        } else {
          let fileFolders = '';
          if (path.sep == '\\') {
            fileFolders = path.dirname(filePath.replace(/\\/g, '/')).split('/');
          } else {
            fileFolders = path.dirname(filePath.replace(path.sep, '/')).split('/');
          }

          fileFolders.reduce((prevPromise, folder) => {
            return prevPromise.then((localFolder) => {
              const absfolder = localFolder.joinFolder(folder);

              return pfs.stat(absfolder.toString()).then((stat) => {
                if (stat.isDirectory()) {
                  return Promise.resolve(absfolder);
                }

                return pfs.mkdir(absfolder.toString()).then(() => {
                  return Promise.resolve(absfolder);
                }).catch((err) => {
                  if (err.message.indexOf('EEXIST: file already exists') > -1) {
                    return Promise.resolve(absfolder);
                  }

                  throw err;
                });
              }).catch((err) => {
                return pfs.mkdir(absfolder.toString()).then(() => {
                  return Promise.resolve(absfolder);
                }).catch((err) => {
                  if (err.message.indexOf('EEXIST: file already exists') > -1) {
                    return Promise.resolve(absfolder);
                  }

                  throw err;
                });
              });
            });
          }, Promise.resolve(toLocalPath)).then((localPath) => {
            const ext = path.extname(fileName);
            const fileLocalPathWithoutExt = path.normalize(localPath.joinFile(path.basename(fileName, ext)).toString());
            let fileLocalPathWithSuffixWithoutExt = fileLocalPathWithoutExt

            if (!$scope.overwriteDownloading.enabled) {
              for (let i = 1; fs.existsSync(fileLocalPathWithSuffixWithoutExt + ext); i++) {
                fileLocalPathWithSuffixWithoutExt = `${fileLocalPathWithoutExt}.${i}`;
              }
            }

            const job = createJob({
              region: qiniuInfo.region,
              from: {
                bucket: qiniuInfo.bucket,
                key: qiniuInfo.path.toString(),
                size: qiniuInfo.size,
                mtime: qiniuInfo.lastModified.toISOString(),
              },
              to: {
                name: fileName,
                path: fileLocalPathWithSuffixWithoutExt + ext
              },
              domain: qiniuInfo.domain.toQiniuDomain(),
              backendMode: qiniuInfo.domain.qiniuBackendMode(),
            });
            addEvents(job);
            t.push(job);

            if (callFn) callFn();
            if (callFn2) callFn2();
          });
        }
      }
    }

    function addEvents(job) {
      if (!job.downloadedParts) {
        job.downloadedParts = [];
      }

      $scope.lists.downloadJobList.push(job);

      trySchedJob();
      trySaveProg();

      $timeout(() => {
        $scope.calcTotalProg();
      });

      job.on("partcomplete", (prog) => {
        trySaveProg();
      });
      job.on("statuschange", (status) => {
        if (status == "stopped") {
          concurrency--;
          trySchedJob();
        }

        trySaveProg();

        $timeout(() => {
          $scope.calcTotalProg();
        });
      });
      job.on("speedchange", () => {
        $timeout(() => {
          $scope.calcTotalProg();
        });
      });
      job.on("complete", () => {
        concurrency--;
        trySchedJob();

        $timeout(() => {
          $scope.calcTotalProg();
        });
      });
      job.on("error", (err) => {
        if (err) {
          console.error(`download kodo://${job.from.bucket}/${job.from.key} error: ${err}`);
        }

        concurrency--;
        trySchedJob();

        $timeout(() => {
          $scope.calcTotalProg();
        });
      });
    }

    function trySchedJob() {
      var maxConcurrency = settingsSvs.maxDownloadConcurrency.get();
      var isDebug = (settingsSvs.isDebug.get() == 1);

      concurrency = Math.max(0, concurrency);
      if (isDebug) {
        console.log(`[JOB] download max: ${maxConcurrency}, cur: ${concurrency}, jobs: ${$scope.lists.downloadJobList.length}`);
      }

      if (concurrency < maxConcurrency) {
        const jobs = $scope.lists.downloadJobList;

        const startAllJobsFrom = (i) => {
          if (i >= jobs.length) {
            return;
          }
          if (concurrency >= maxConcurrency) {
            return;
          }

          const job = jobs[i];
          if (isDebug) {
            console.log('[JOB] sched ', job.status, ' => ', job._config);
          }
          if (job.status === "waiting") {
            concurrency++;

            if (job.prog.resumable) {
              tryLoadProgForJob(job).then((prog) => {
                if (prog) {
                  job.start(prog);
                } else {
                  job.start();
                }
              }).finally(() => {
                startAllJobsFrom(i + 1);
              });
              return;
            } else {
              job.start();
            }
          }
          $timeout(() => { startAllJobsFrom(i + 1); });
        };
        startAllJobsFrom(0);
      }
    }

    function trySaveProg() {
      var t = {};
      angular.forEach($scope.lists.downloadJobList, function (job) {
        if (job.status == "finished") return;

        t[job.id] = {
          region: job.region,
          to: job.to,
          from: job.from,
          prog: {
            synced: job.prog.synced,
            total: job.prog.total,
            resumable: job.prog.resumable
          },
          backendMode: job.backendMode,
          domain: job.domain,
          status: job.status,
          message: job.message,
        };
      });

      fs.writeFileSync(getDownProgFilePath(), JSON.stringify(t));
    }

    /**
     * resolve prog saved
     */
    function tryLoadProg() {
      let progs = {};
      try {
        const data = fs.readFileSync(getDownProgFilePath());
        progs = JSON.parse(data);
      } catch (e) {}

      if (!progs) {
        return Promise.resolve([]);
      }

      const promises = Object.values(progs).map((job) => tryLoadProgForJob(job));
      return Promise.all(promises);
    }

    function tryLoadProgForJob(job) {
      return new Promise((resolve) => {
        const options = { ignoreError: true };
        if (job.backendMode == KODO_MODE) {
          options.preferKodoAdapter = true;
        } else {
          options.preferS3Adapter = true;
        }
        QiniuClient.headFile(job.region, job.from.bucket, job.from.key, options).then((info) => {
          if (info.size !== job.from.size || info.lastModified.toISOString() !== job.from.mtime) {
            if (job.prog) {
              delete job.prog.synced;
            }
          }
          resolve(job);
        }).catch(() => {
          if (job.prog) {
            delete job.prog.synced;
          }
          resolve(job);
        });
      });
    }

    // prog save path
    function getDownProgFilePath() {
      var folder = Global.config_path;
      if (!fs.existsSync(folder)) {
        fs.mkdirSync(folder);
      }

      var username = AuthInfo.get().id || "kodo-browser";
      return path.join(folder, "downprog_" + username + ".json");
    }
  }
]);

angular.module("web").factory("ExternalPath", [
    "$q",
    "$translate",
    "AuthInfo",
    "Config",
    function($q, $translate, AuthInfo, Config) {
        const fs = require('fs'),
              path = require('path'),
              T = $translate.instant;

        class ExternalPath {
            constructor(bucketId, objectPrefix, regionId) {
                this.bucketId = bucketId;
                this.objectPrefix = objectPrefix;
                this.regionId = regionId;
                this.shortPath = bucketId;
                if (objectPrefix) {
                    this.shortPath += `/${objectPrefix}`;
                }
                this.fullPath = `kodo://${this.shortPath}`;
            }
        }

        return {
            list: list,
            listSync: listSync,
            getRegionByBucketSync: getRegionByBucketSync,
            create: create,
            remove: remove,
            new: newExternalPath,
        };

        function list() {
            const df = $q.defer(),
                  filePath = getFilePath();
            fs.access(filePath, fs.constants.R_OK, (err) => {
                if (err) {
                    df.resolve([]);
                } else {
                    fs.readFile(filePath, 'utf8', (err, data) => {
                        if (err) {
                            df.reject(err);
                        } else {
                            df.resolve(JSON.parse(data));
                        }
                    });
                }
            });
            return df.promise;
        }

        function listSync() {
            const filePath = getFilePath();

            try {
                fs.accessSync(filePath, fs.constants.R_OK);
            } catch (err) {
                return []
            }

            return JSON.parse(fs.readFileSync(filePath, 'utf8'));
        }

        function getRegionByBucketSync(bucketId) {
            const paths = listSync();
            for (let i = 0; i < paths.length; i++) {
                const path = paths[i];
                if (bucketId === path.bucketId) {
                    return path.regionId;
                }
            }
            return null;
        }

        function create(externalPath, regionId) {
            const df = $q.defer();

            list().then((paths) => {
                const newOne = newExternalPath(externalPath, regionId);
                for (let i = 0; i < paths.length; i++) {
                    const path = paths[i];
                    if (newOne.bucketId === path.bucketId && newOne.objectPrefix === path.objectPrefix) {
                        df.reject(new Error("Duplicated external path"));
                        return;
                    }
                }
                paths.push(newOne);
                writeExternalPaths(paths, (err) => {
                    if (err) {
                        df.reject(err);
                    } else {
                        df.resolve(paths);
                    }
                });
            }, (err) => {
                df.reject(err);
            });
            return df.promise;
        }

        function remove(externalPath, regionId) {
            const df = $q.defer(),
                  filePath = getFilePath();

            list().then((paths) => {
                const target = newExternalPath(externalPath, regionId);
                for (let i = 0; i < paths.length; i++) {
                    const path = paths[i];
                    if (target.bucketId === path.bucketId && target.objectPrefix === path.objectPrefix) {
                        paths.splice(i, 1);
                        break;
                    }
                }
                writeExternalPaths(paths, (err) => {
                    if (err) {
                        df.reject(err);
                    } else {
                        df.resolve(paths);
                    }
                });
            }, (err) => {
                df.reject(err);
            });
            return df.promise;
        }

        function newExternalPath(externalPath, regionId) {
            if (externalPath.startsWith('kodo://')) {
                externalPath = externalPath.substring('kodo://'.length);
            }
            let splits = externalPath.split('/', 2);
            return new ExternalPath(splits[0], splits[1] || '', regionId);
        }

        function getFilePath() {
            const username = AuthInfo.get().id || "kodo-browser",
                  folder = Global.config_path;

            if (!fs.existsSync(folder)) {
                fs.mkdirSync(folder, { recursive: true });
            }

            return path.join(folder, `external_paths_${username}.json`);
        }

        function writeExternalPaths(paths, fn) {
            fs.writeFile(getFilePath(), JSON.stringify(paths), 'utf8', fn);
        }
    }
]);

angular.module("web").factory("fileSvs", [
  "$q",
  function ($q) {
    return {
      /**
       * 根据后缀判断
       * @param  item = {name, size}
       * @return obj = {type, ...}
       *     type: [picture|code|others|doc|video|audio]
       */
      getFileType: function (item) {
        if (item.itemType === 'folder') {
          return {
            type: "folder",
            ext: []
          };
        }

        const ext = item.path.extname() ? item.path.extname().substring(1) : '';

        switch (ext) {
        case "png":
        case "jpg":
        case "jpeg":
        case "bmp":
        case "gif":
          return {
            type: "picture",
            ext: [ext]
          };

        case "doc":
        case "docx":
        case "pdf":
          return {
            type: "doc",
            ext: [ext]
          };

        case "mp4":
          return {
            type: "video",
            ext: [ext],
            mineType: "video/mp4"
          };
        case "webm":
          return {
            type: "video",
            ext: [ext],
            mineType: "video/webm"
          };
        case "mov":
          return {
            type: "video",
            ext: [ext],
            mineType: "video/quicktime"
          };
        case "ogv":
          return {
            type: "video",
            ext: [ext],
            mineType: "video/ogg"
          };
        case "flv":
          return {
            type: "video",
            ext: [ext],
            mineType: "video/x-flv"
          };

        case "mp3":
          return {
            type: "audio",
            ext: [ext],
            mineType: "audio/mp3"
          };
        case "ogg":
          return {
            type: "audio",
            ext: [ext],
            mineType: "audio/ogg"
          };
        }

        var codeMode = CodeMirror.findModeByExtension(ext);
        if (codeMode) {
          codeMode.type = "code";

          return codeMode;
        }

        return {
          type: "others",
          ext: [ext]
        };
      }
    };
  }
]);

angular
  .module("web")
  .factory("I18n", [
    function() {
      var defaultLocale = navigator.language;
      console.log("Default Locale:", defaultLocale);

      var _transMap = {}; // locale: kvPairs
      return {
        init: function(locale, kvPairs) {
          _transMap[locale] = kvPairs;
        },
        use: function(locale) {
          defaultLocale = locale;
        },
        getLocale: function() {
          return defaultLocale;
        },
        translate: function(key, options, locale) {
          try {
            if (options) {
              console.log(key, options, locale, defaultLocale, _transMap);
              var msg = _transMap[locale || defaultLocale][key];
              for (var k in options) {
                msg.replace(
                  new Regex("\\{\\{" + k + "\\}\\}", "g"),
                  options[k]
                );
              }
              return msg;
            } else {
              return _transMap[locale || defaultLocale][key];
            }
          } catch (e) {
            console.log("---", e);
            return key;
          }
        }
      };
    }
  ])
  .filter("translate2", [
    "I18n",
    function(I18n) {
      return function(key, options) {
        return I18n.translate(key, options);
      };
    }
  ]);

angular.module("web").factory("indexDBSvs", [
  "$q",
  function($q) {
    return {
      open: open,
      create: create,
      update: update,
      list: list,
      get: get,
      delete: del,
      clear: clear
    };

    /**
      initStore: {storeName: }
      */
    function open(name, version, initStoreMap) {
      var df = $q.defer();
      var version = version || 1;
      var request = window.indexedDB.open(name, version);
      request.onerror = function(e) {
        console.error(e.currentTarget.error.message);
        df.reject(e.currentTarget.error);
      };
      request.onsuccess = function(e) {
        var db = e.target.result;
        df.resolve(db);
      };
      request.onupgradeneeded = function(e) {
        var db = e.target.result;
        if (initStoreMap) {
          for (var storeName in initStoreMap) {
            if (!db.objectStoreNames.contains(storeName)) {
              db.createObjectStore(
                storeName,
                initStoreMap[storeName] || {
                  keyPath: "id",
                  autoIncrement: true
                }
              );
              //db.createObjectStore(storeName, {keyPath: 'id', utoIncrement: true});
            }
          }
        }
        console.log("DB version changed to " + version);
      };
      return df.promise;
    }
    function list(db, storeName) {
      var df = $q.defer();
      var transaction = db.transaction(storeName, "readwrite");
      var store = transaction.objectStore(storeName);
      var cursor = store.openCursor();
      var data = [];
      cursor.onsuccess = function(e) {
        var result = e.target.result;
        if (result && result !== null) {
          data.push(result.value);
          result.continue();
        } else {
          df.resolve(data);
        }
      };
      cursor.onerror = function(e) {
        console.error(e.currentTarget.error.message);
        df.reject(e.currentTarget.error);
      };
      return df.promise;
    }

    function get(db, storeName, key) {
      var df = $q.defer();
      var transaction = db.transaction(storeName, "readwrite");
      var store = transaction.objectStore(storeName);
      var request = store.get(key);
      request.onsuccess = function(e) {
        var item = e.target.result;
        df.resolve(item);
      };
      request.onerror = function(e) {
        console.error(e.currentTarget.error.message);
        df.reject(e.currentTarget.error);
      };
      return df.promise;
    }
    function create(db, storeName, data) {
      var df = $q.defer();
      var transaction = db.transaction(storeName, "readwrite");
      var store = transaction.objectStore(storeName);
      var request = store.add(data);
      request.onsuccess = function(e) {
        df.resolve();
      };
      request.onerror = function(e) {
        console.error(e.currentTarget.error.message);
        df.reject(e.currentTarget.error);
      };
      return df.promise;
    }
    function update(db, storeName, key, data) {
      var df = $q.defer();
      var transaction = db.transaction(storeName, "readwrite");
      var store = transaction.objectStore(storeName);
      var request = store.get(key);
      request.onsuccess = function(e) {
        var item = e.target.result;
        if (item) {
          angular.extend(item, data);
          store.put(item);
        } else {
          store.add(data);
        }
        df.resolve();
      };
      request.onerror = function(e) {
        console.error(e.currentTarget.error.message);
        df.reject(e.currentTarget.error);
      };
      return df.promise;
    }

    function del(db, storeName, key) {
      var df = $q.defer();
      var transaction = db.transaction(storeName, "readwrite");
      var store = transaction.objectStore(storeName);
      var request = store.delete(key);
      request.onsuccess = function(e) {
        df.resolve();
      };
      request.onerror = function(e) {
        console.error(e.currentTarget.error.message);
        df.reject(e.currentTarget.error);
      };
      return df.promise;
    }
    function clear(db, storeName, key) {
      var df = $q.defer();
      var transaction = db.transaction(storeName, "readwrite");
      var store = transaction.objectStore(storeName);
      var request = store.clear();
      request.onsuccess = function(e) {
        df.resolve();
      };
      request.onerror = function(e) {
        console.error(e.currentTarget.error.message);
        df.reject(e.currentTarget.error);
      };
      return df.promise;
    }
  }
]);

angular.module("web").factory("jobUtil", [
  "$q",
  "$state",
  "$timeout",
  "$translate",
  function($q, $state, $timeout, $translate) {
    var T = $translate.instant;

    return {
      getStatusLabel: getStatusLabel,
      getStatusCls: getStatusCls
    };

    function getStatusCls(s) {
      if (!s) return "default";
      switch (s.toLowerCase()) {
        case "running":
          return "info";
        case "verifying":
          return "primary";
        case "failed":
          return "danger";
        case "finished":
          return "success";
        case "stopped":
          return "warning";
        default:
          return "default";
      }
    }

    function getStatusLabel(s, isUp) {
      if (!s) return s;
      switch (s.toLowerCase()) {
        case "running":
          return isUp
            ? T("status.running.uploading")
            : T("status.running.downloading"); //'正在上传':'正在下载';
        case "failed":
          return T("status.failed"); //'失败';
        case "finished":
          return T("status.finished"); // '完成';
        case "stopped":
          return T("status.stopped"); //'暂停';
        case "verifying":
          return T("status.verifying"); //'';
        default:
          return T("status.waiting"); //'等待';
      }
    }
  }
]);

angular.module('web').factory('QiniuClient', [
  '$q',
  '$rootScope',
  '$translate',
  '$timeout',
  '$state',
  'Toast',
  'Config',
  'AuthInfo',
  'settingsSvs',
  function ($q, $rootScope, $translate, $timeout, $state, Toast, Config, AuthInfo, settingsSvs) {
    const { Qiniu, Region, KODO_MODE, S3_MODE, RegionService } = require('kodo-s3-adapter-sdk'),
      path = require('path'),
      qiniuPath = require('qiniu-path'),
      { Semaphore } = require('semaphore-promise'),
      T = $translate.instant,
      kodoAdaptersCache = {},
      s3AdaptersCache = {},
      regionServicesCache = {};

    return {
      isQueryRegionAPIAvaiable: isQueryRegionAPIAvaiable,
      listAllBuckets: listAllBuckets,
      createBucket: createBucket,
      deleteBucket: deleteBucket,

      listFiles: listFiles,
      listDomains: listDomains,
      createFolder: createFolder,

      checkFileExists: checkFileExists,
      checkFolderExists: checkFolderExists,
      getFrozenInfo: getFrozenInfo,
      headFile: headFile,
      setStorageClass: setStorageClass,
      setStorageClassOfFiles: setStorageClassOfFiles,
      stopSetStorageClassOfFiles: stopSetStorageClassOfFiles,

      getContent: getContent,
      saveContent: saveContent,

      //重命名
      moveOrCopyFile: moveOrCopyFile,

      //复制，移动
      moveOrCopyFiles: moveOrCopyFiles,
      stopMoveOrCopyFiles: stopMoveOrCopyFiles,

      //解冻
      restoreFile: restoreFile,
      restoreFiles: restoreFiles,
      stopRestoreFiles: stopRestoreFiles,

      //删除
      deleteFiles: deleteFiles,
      stopDeleteFiles: stopDeleteFiles,

      parseKodoPath: parseKodoPath,
      signatureUrl: signatureUrl,
      getRegions: getRegions,
      clientBackendMode: clientBackendMode,
      clearAllCache: clearAllCache,
    };

    function isQueryRegionAPIAvaiable(ucUrl) {
      return new Promise((resolve) => {
        const option = {
                         accessKey: '', bucketName: '', ucUrl: ucUrl,
                         appName: 'kodo-browser', appVersion: Global.app.version,
                         requestCallback: debugRequest(KODO_MODE),
                         responseCallback: debugResponse(KODO_MODE),
                       };
        Region.query(option).
          then(() => { resolve(true); }).
          catch((err) => {
            if (err.res && err.res.statusCode === 404) {
              resolve(false);
            } else if (err.code && (err.code === 'ENOTFOUND' || err.code === 'ECONNREFUSED')) {
              resolve(false);
            } else {
              resolve(true);
            }
          });
      });
    }

    function listAllBuckets(opt) {
      return new Promise((resolve, reject) => {
        getDefaultClient(opt).enter('listAllBuckets', (client) => client.listBuckets()).
          then(resolve).catch((err) => {
            handleError(err);
            reject(err);
          });
      });
    }

    function createBucket(region, bucket, opt) {
      return new Promise((resolve, reject) => {
        getDefaultClient(opt).enter('createBucket', (client) => client.createBucket(region, bucket)).
          then(resolve).catch((err) => {
            handleError(err);
            reject(err);
          });
      });
    }

    function deleteBucket(region, bucket, opt) {
      return new Promise((resolve, reject) => {
        getDefaultClient(opt).enter('deleteBucket', (client) => client.deleteBucket(region, bucket)).
          then(resolve).catch((err) => {
            handleError(err);
            reject(err);
          });
      });
    }

    function listFiles(region, bucket, key, marker, opt) {
      return new Promise((resolve, reject) => {
        const items = [];
        const option = {
          nextContinuationToken: marker,
          delimiter: '/',
          maxKeys: opt.maxKeys || 1000,
          minKeys: opt.minKeys || 1000,
        };
        let prefix = key.toString();
        if (!prefix.endsWith('/')) {
          prefix = prefix.substring(0, prefix.lastIndexOf('/') + 1);
        }
        getDefaultClient(opt).enter('listFiles', (client) => client.listObjects(region, bucket, key.toString(), option)).
          then((listedObjects) => {
            if (listedObjects.commonPrefixes) {
              listedObjects.commonPrefixes.forEach((item) => {
                const path = new qiniuPath.fromQiniuPath(item.key);
                items.push({
                  bucket: item.bucket,
                  name: path.directoryBasename(),
                  path: path,
                  itemType: 'folder',
                });
              });
            }
            if (listedObjects.objects) {
              const ONE_HOUR = 60 * 60 * 1000;
              listedObjects.objects.forEach((item) => {
                if (!key.toString().endsWith('/') || item.key != key.toString()) {
                  const path = new qiniuPath.fromQiniuPath(item.key);
                  items.push({
                    bucket: bucket,
                    name: path.basename(),
                    path: path,
                    size: item.size,
                    storageClass: item.storageClass,
                    lastModified: item.lastModified,
                    itemType: 'file',
                    withinFourHours: (new Date() - item.lastModified) <= 4 * ONE_HOUR,
                  });
                }
              });
            }
            resolve({ data: items, marker: listedObjects.nextContinuationToken });
          }).catch((err) => {
            handleError(err);
            reject(err);
          });
      });
    }

    function listDomains(region, bucket, opt) {
      return new Promise((resolve, listDomains) => {
        getDefaultClient(opt).enter('listDomains', (client) => client.listDomains(region, bucket)).
          then(resolve).catch((err) => {
            handleError(err);
            reject(err);
          });
      });
    }

    function createFolder(region, bucket, prefix, opt) {
      if (!prefix.directoryBasename) {
          prefix = qiniuPath.fromQiniuPath(prefix);
      }
      return new Promise((resolve, reject) => {
        getDefaultClient(opt).enter('createFolder', (client) => {
          return client.putObject(region, {
            bucket: bucket, key: prefix.toString(),
          }, Buffer.alloc(0), prefix.directoryBasename());
        }).
          then(resolve).
          catch((err) => {
            handleError(err);
            reject(err);
          });
      });
    }

    function checkFileExists(region, bucket, key, opt) {
      return new Promise((resolve, reject) => {
        getDefaultClient(opt).enter('checkFileExists', (client) => client.isExists(region, { bucket: bucket, key: key.toString() })).
          then(resolve).catch((err) => {
            handleError(err);
            reject(err);
          });
      });
    }

    function checkFolderExists(region, bucket, prefix, opt) {
      return new Promise((resolve, reject) => {
        getDefaultClient(opt).enter('checkFolderExists', (client) => client.listObjects(region, bucket, prefix.toString(), { maxKeys: 1 })).
          then((results) => {
            resolve(results.objects && results.objects.length > 0 && results.objects[0].key.startsWith(prefix.toString()));
          }).catch((err) => {
            handleError(err);
            reject(err);
          });
      });
    }

    function getFrozenInfo(region, bucket, key, opt) {
      return new Promise((resolve, reject) => {
        getDefaultClient(opt).enter('getFrozenInfo', (client) => client.getFrozenInfo(region, { bucket: bucket, key: key.toString() })).
          then(resolve).catch((err) => {
            handleError(err);
            reject(err);
          });
      });
    }

    function headFile(region, bucket, key, opt) {
      let promise = getDefaultClient(opt).enter('headFile', (client) => client.getObjectInfo(region, { bucket: bucket, key: key.toString() }));
      if (!opt.ignoreError) {
        promise = promise.catch((err) => {
          handleError(err);
          reject(err);
        });
      }
      delete opt.ignoreError;
      return promise;
    }

    function setStorageClass(region, bucket, key, storageClass, opt) {
      return new Promise((resolve, reject) => {
        getDefaultClient(opt).enter('setStorageClass', (client) => client.setObjectStorageClass(region, { bucket: bucket, key: key }, storageClass)).
          then(resolve).catch((err) => {
            handleError(err);
            reject(err);
          });
      });
    }

    function getContent(region, bucket, key, domain, opt) {
      return new Promise((resolve, reject) => {
        getDefaultClient(opt).enter('getContent', (client) => client.getObject(region, { bucket: bucket, key: key.toString() }, domain)).
          then((result) => {
            resolve(result.data);
          }).catch((err) => {
            handleError(err);
            reject(err);
          });
      });
    }

    function saveContent(region, bucket, key, content, domain, getOpt, putOpt) {
      return new Promise((resolve, reject) => {
        getDefaultClient(opt).enter('saveContent', (client) => {
          return client.getObjectHeader(region, { bucket: bucket, key: key.toString() }, domain).then((headers) => {
            const basename = key.basename();
            client.putObject(region, { bucket: bucket, key: key.toString() }, Buffer.from(content), basename,
              { metadata: headers.metadata },
            ).then(() => { resolve(); }).catch((err) => {
              handleError(err);
              reject(err);
            });
          }).catch((err) => {
            handleError(err);
            reject(err);
          });
        });
      });
    }

    var stopSetStorageClassOfFilesFlag = false;

    function setStorageClassOfFiles(region, bucket, items, storageClass, progressFn, opt) {
      const progress = { total: 0, current: 0, errorCount: 0 };
      stopSetStorageClassOfFilesFlag = false;

      const errorItems = [];

      const newProgressFn = (progress) => {
        if (progressFn) {
          try {
            progressFn(progress);
          } catch (err) {
            handleError(err);
          }
        }
      };
      newProgressFn(progress);

      const restoreCallback = (items) => {
        return (index, error) => {
          if (error) {
            errorItems.push({ item: items[index], error: error });
            progress.errorCount += 1;
          } else {
            progress.current += 1;
          }
          newProgressFn(progress);
          if (stopSetStorageClassOfFilesFlag) {
            return false;
          }
        };
      };

      return new Promise((resolve, reject) => {
        getDefaultClient(opt).enter('setStorageClassOfFiles', (client) => {
          const toSetStorageClassOfObjects = items.filter((item) => item.itemType === 'file');
          let promises = [];
          if (toSetStorageClassOfObjects && toSetStorageClassOfObjects.length > 0) {
            promises.push(client.setObjectsStorageClass(region, bucket, toSetStorageClassOfObjects.map((item) => item.path.toString()), storageClass, restoreCallback(toSetStorageClassOfObjects.map((item) => { return { key: item.path.toString() }; }))));
            progress.total += toSetStorageClassOfObjects.length;
            newProgressFn(progress);
          }

          const toSetStorageClassOfFolders = items.filter((item) => item.itemType === 'folder');
          const semaphore = new Semaphore(3);
          promises = promises.concat(toSetStorageClassOfFolders.map((toSetStorageClassOfFolder) => {
            return doSetStorageClassOfFolder(client, region, toSetStorageClassOfFolder, progress, newProgressFn, semaphore, restoreCallback);
          }));

          return Promise.all(promises).then(() => { resolve(errorItems); }, reject);
        });
      });

      function doSetStorageClassOfFolder(client, region, folderObject, progress, progressFn, semaphore, restoreCallback) {
        return new Promise((resolve, reject) => {
          semaphore.acquire().then((release) => {
            _doSetStorageClassOfFolder(client, region, folderObject, progress, progressFn, undefined, restoreCallback)
              .then(resolve, reject)
              .finally(() => { release(); });
          });
        });

        function _doSetStorageClassOfFolder(client, region, folderObject, progress, progressFn, marker, restoreCallback) {
          return new Promise((resolve, reject) => {
            if (stopSetStorageClassOfFilesFlag) {
              reject(new Error('User Cancelled'));
              return;
            }
            client.listObjects(region, folderObject.bucket, folderObject.path.toString(), { nextContinuationToken: marker, maxKeys: 1000 }).then((listedObjects) => {
              if (stopSetStorageClassOfFilesFlag) {
                reject(new Error('User Cancelled'));
                return;
              } else if (!listedObjects.objects || listedObjects.objects.length === 0) {
                resolve();
                return;
              }

              const objects = listedObjects.objects.filter((object) => !object.key.endsWith('/'));

              let promise;
              if (objects.length > 0) {
                progress.total += objects.length;
                progressFn(progress);
                promise = client.setObjectsStorageClass(region, folderObject.bucket, objects.map((object) => object.key), storageClass, restoreCallback(objects.map((item) => { return { key: item.key }; })));
              } else {
                promise = Promise.resolve();
              }
              if (listedObjects.nextContinuationToken) {
                const promises = [promise];
                promises.push(_doSetStorageClassOfFolder(client, region, folderObject, progress, progressFn, listedObjects.nextContinuationToken, restoreCallback));
                promise = Promise.all(promises);
              }
              promise.then(resolve).catch(reject);
            }).catch(reject);
          });
        }
      }
    }

    function stopSetStorageClassOfFiles() {
      stopSetStorageClassOfFilesFlag = true;
    }

    function moveOrCopyFile(region, bucket, oldKey, newKey, isCopy, opt) {
      return new Promise((resolve, reject) => {
        const transferObject = {
          from: { bucket: bucket, key: oldKey.toString() },
          to: { bucket: bucket, key: newKey.toString() },
        };

        getDefaultClient(opt).enter('moveOrCopyFile', (client) => {
          if (isCopy) {
            return client.copyObject(region, transferObject).then(resolve).catch((err) => {
              handleError(err);
              reject(err);
            });
          } else {
            return client.moveObject(region, transferObject).then(resolve).catch((err) => {
              if (err.code === 'AccessDenied' && err.stage === 'delete') {
                Toast.error(T('permission.denied.move.error_when_delete', {
                  fromKey: oldKey, toKey: newKey,
                }));
                reject(err);
                return;
              }
              handleError(err);
              reject(err);
            });
          }
        });
      });
    }

    var stopCopyFilesFlag = false;

    function moveOrCopyFiles(region, items, target, progressFn, isCopy, renamePrefix, opt) {
      const progress = { total: 0, current: 0, errorCount: 0 };
      stopCopyFilesFlag = false;

      return new Promise((resolve, reject) => {
        getDefaultClient(opt).enter('moveOrCopyFiles', (client) => {
          const newProgressFn = (progress) => {
            if (progressFn) {
              try {
                progressFn(progress);
              } catch (err) {
                handleError(err);
              }
            }
          };
          newProgressFn(progress);

          return deepForEachItem(client, region, items, target, progress, newProgressFn, isCopy, renamePrefix, new Semaphore(3))
            .then(resolve)
            .catch((err) => {
              handleError(err);
              reject(err);
            });
          });
      });

      function deepForEachItem(client, region, items, target, progress, progressFn, isCopy, renamePrefix, semaphore) {
        const errorItems = [];

        if (stopCopyFilesFlag) {
          reject(new Error('User Cancelled'));
          return;
        }

        const transferObjects = items.map((item) => {
          let toPrefix = renamePrefix;

          if (!renamePrefix) {
            if (target.key) {
              toPrefix = target.key;
              if (!target.key.endsWith('/')) {
                toPrefix += '/'
              }
            } else {
              toPrefix = '';
            }
            toPrefix += item.name;
            if (item.itemType === 'folder' && !item.name.endsWith('/')) {
              toPrefix += '/';
            }
          }

          item = Object.assign({ key: item.path.toString() }, item);

          return {
            from: { bucket: item.bucket, key: item.key },
            to: { bucket: target.bucket, key: toPrefix },
            item: item,
          };
        });
        const moveOrCopyFileCallback = (transferObjects) => {
          return (index, err) => {
            if (err) {
              errorItems.push({ item: transferObjects[index].item, error: err });
              progress.errorCount += 1;
              if (err.code === 'AccessDenied' && err.stage === 'delete') {
                err.translated_message = T('permission.denied.move.error_when_delete', {
                  fromKey: transferObjects[index].from.key, toKey: transferObjects[index].to.key,
                });
              }
            } else {
              progress.current += 1;
            }
            progressFn(progress);
            if (stopCopyFilesFlag) {
              return false;
            }
          };
        };

        const transferFileObjects = transferObjects.filter((transferObject) => transferObject.item.itemType === 'file');
        let promises = [];
        if (transferFileObjects && transferFileObjects.length > 0) {
          if (isCopy) {
            promises.push(client.copyObjects(region, transferFileObjects, moveOrCopyFileCallback(transferFileObjects)));
          } else {
            promises.push(client.moveObjects(region, transferFileObjects, moveOrCopyFileCallback(transferFileObjects)));
          }
          progress.total += transferFileObjects.length;
          progressFn(progress);
        }

        const transferFolderObjects = transferObjects.filter((transferObject) => transferObject.item.itemType === 'folder');
        promises = promises.concat(transferFolderObjects.map((transferObject) => {
          return doCopyFolder(client, region, transferObject, progress, progressFn, isCopy, semaphore, moveOrCopyFileCallback);
        }));
        return new Promise((resolve, reject) => {
          Promise.all(promises).then(() => { resolve(errorItems); }).catch(reject);
        });
      }

      function doCopyFolder(client, region, transferObject, progress, progressFn, isCopy, semaphore, moveOrCopyFileCallback) {
        return new Promise((resolve, reject) => {
          semaphore.acquire().then((release) => {
            _doCopyFolder(client, region, transferObject, progress, progressFn, isCopy, undefined, moveOrCopyFileCallback)
              .then(resolve, reject)
              .finally(() => { release(); });
          });
        });

        function _doCopyFolder(client, region, transferObject, progress, progressFn, isCopy, marker, moveOrCopyFileCallback) {
          return new Promise((resolve, reject) => {
            if (stopCopyFilesFlag) {
              reject(new Error('User Cancelled'));
              return;
            }
            client.listObjects(region, transferObject.from.bucket, transferObject.from.key, { nextContinuationToken: marker, maxKeys: 1000 }).then((listedObjects) => {
              if (stopCopyFilesFlag) {
                reject(new Error('User Cancelled'));
                return;
              } else if (!listedObjects.objects || listedObjects.objects.length === 0) {
                resolve();
                return;
              }

              const transferObjects = listedObjects.objects.map((object) => {
                let toKey = transferObject.to.key;
                  if (toKey && !toKey.endsWith('/')) {
                  toKey += '/';
                }
                toKey += object.key.substring(transferObject.from.key.length);
                return { from: object, to: { bucket: transferObject.to.bucket, key: toKey }, item: { key: toKey, itemType: toKey.endsWith('/') ? 'folder' : 'file' } };
              }).filter((object) => object.to.key);

              progress.total += transferObjects.length;
              progressFn(progress);

              let promise;
              if (isCopy) {
                promise = client.copyObjects(region, transferObjects, moveOrCopyFileCallback(transferObjects));
              } else {
                promise = client.moveObjects(region, transferObjects, moveOrCopyFileCallback(transferObjects));
              }
              if (listedObjects.nextContinuationToken) {
                const promises = [promise];
                promises.push(_doCopyFolder(client, region, transferObject, progress, progressFn, isCopy, listedObjects.nextContinuationToken, moveOrCopyFileCallback));
                promise = Promise.all(promises);
              }
              promise.then(resolve).catch(reject);
            }).catch(reject);
          });
        }
      }
    }

    function stopMoveOrCopyFiles() {
      stopCopyFilesFlag = true;
    }

    function restoreFile(region, bucket, key, days, opt) {
      return new Promise((resolve, reject) => {
        getDefaultClient(opt).enter('restoreFile', (client) => client.restoreObject(region, { bucket: bucket, key: key.toString() }, days)).
          then(resolve).catch((err) => {
            handleError(err);
            reject(err);
          });
      });
    }

    var stopRestoreFilesFlag = false;

    function restoreFiles(region, bucket, items, days, progressFn, opt) {
      const progress = { total: 0, current: 0, errorCount: 0 };
      stopRestoreFilesFlag = false;

      const errorItems = [];

      const newProgressFn = (progress) => {
        if (progressFn) {
          try {
            progressFn(progress);
          } catch (err) {
            handleError(err);
          }
        }
      };
      newProgressFn(progress);

      const restoreCallback = (items) => {
        return (index, error) => {
          if (error) {
            errorItems.push({ item: items[index], error: error });
            progress.errorCount += 1;
          } else {
            progress.current += 1;
          }
          newProgressFn(progress);
          if (stopRestoreFilesFlag) {
            return false;
          }
        };
      };

      return new Promise((resolve, reject) => {
        getDefaultClient(opt).enter('restoreFiles', (client) => {
          const toRestoreObjects = items.filter((item) => item.itemType === 'file');
          let promises = [];
          if (toRestoreObjects && toRestoreObjects.length > 0) {
            promises.push(client.restoreObjects(region, bucket, toRestoreObjects.map((item) => item.path.toString()), days, restoreCallback(toRestoreObjects.map((item) => { return { key: item.path.toString() }; }))));
            progress.total += toRestoreObjects.length;
            newProgressFn(progress);
          }

          const toRestoreFolders = items.filter((item) => item.itemType === 'folder');
          const semaphore = new Semaphore(3);
          promises = promises.concat(toRestoreFolders.map((toRestoreFolder) => {
            return doRestoreFolder(client, region, toRestoreFolder, progress, newProgressFn, semaphore, restoreCallback);
          }));

          return Promise.all(promises).then(() => { resolve(errorItems); }, reject);
        });
      });

      function doRestoreFolder(client, region, folderObject, progress, progressFn, semaphore, restoreCallback) {
        return new Promise((resolve, reject) => {
          semaphore.acquire().then((release) => {
            _doRestoreFolder(client, region, folderObject, progress, progressFn, undefined, restoreCallback)
              .then(resolve, reject)
              .finally(() => { release(); });
          });
        });

        function _doRestoreFolder(client, region, folderObject, progress, progressFn, marker, restoreCallback) {
          return new Promise((resolve, reject) => {
            if (stopRestoreFilesFlag) {
              reject(new Error('User Cancelled'));
              return;
            }
            client.listObjects(region, folderObject.bucket, folderObject.path.toString(), { nextContinuationToken: marker, maxKeys: 1000 }).then((listedObjects) => {
              if (stopRestoreFilesFlag) {
                reject(new Error('User Cancelled'));
                return;
              } else if (!listedObjects.objects || listedObjects.objects.length === 0) {
                resolve();
                return;
              }

              const objects = listedObjects.objects.filter((object) => !object.key.endsWith('/'));

              let promise;
              if (objects.length > 0) {
                progress.total += objects.length;
                progressFn(progress);
                promise = client.restoreObjects(region, folderObject.bucket, objects.map((object) => object.key), days, restoreCallback(objects.map((item) => { return { key: item.key }; })));
              } else {
                promise = Promise.resolve();
              }
              if (listedObjects.nextContinuationToken) {
                const promises = [promise];
                promises.push(_doRestoreFolder(client, region, folderObject, progress, progressFn, listedObjects.nextContinuationToken, restoreCallback));
                promise = Promise.all(promises);
              }
              promise.then(resolve).catch(reject);
            }).catch(reject);
          });
        }
      }
    }

    function stopRestoreFiles() {
      stopRestoreFilesFlag = true;
    }

    var stopDeleteFilesFlag = false;

    function deleteFiles(region, bucket, items, progressFn, opt) {
      const progress = { total: 0, current: 0, errorCount: 0 };
      stopDeleteFilesFlag = false;

      const errorItems = [];

      const newProgressFn = (progress) => {
        if (progressFn) {
          try {
            progressFn(progress);
          } catch (err) {
            handleError(err);
          }
        }
      };
      newProgressFn(progress);

      const deleteCallback = (items) => {
        return (index, error) => {
          if (error) {
            errorItems.push({ item: items[index], error: error });
            progress.errorCount += 1;
          } else {
            progress.current += 1;
          }
          newProgressFn(progress);
          if (stopDeleteFilesFlag) {
            return false;
          }
        };
      };

      return new Promise((resolve, reject) => {
        getDefaultClient(opt).enter('deleteFiles', (client) => {
          items.forEach((item) => {
            item.key = item.path.toString();
          });
          const toDeleteObjects = items.filter((item) => item.itemType === 'file');
          let promises = [];
          if (toDeleteObjects && toDeleteObjects.length > 0) {
            promises.push(client.deleteObjects(region, bucket, toDeleteObjects.map((item) => item.key), deleteCallback(toDeleteObjects)));
            progress.total += toDeleteObjects.length;
            newProgressFn(progress);
          }

          const toDeleteFolders = items.filter((item) => item.itemType === 'folder');
          const semaphore = new Semaphore(3);
          promises = promises.concat(toDeleteFolders.map((toDeleteFolder) => {
            return doDeleteFolder(client, region, toDeleteFolder, progress, newProgressFn, semaphore, deleteCallback);
          }));

          return Promise.all(promises).then(() => { resolve(errorItems); }, reject);
        });
      });

      function doDeleteFolder(client, region, folderObject, progress, progressFn, semaphore, deleteCallback) {
        return new Promise((resolve, reject) => {
          semaphore.acquire().then((release) => {
            _doDeleteFolder(client, region, folderObject, progress, progressFn, undefined, deleteCallback)
              .then(resolve, reject)
              .finally(() => { release(); });
          });
        });

        function _doDeleteFolder(client, region, folderObject, progress, progressFn, marker, deleteCallback) {
          return new Promise((resolve, reject) => {
            if (stopDeleteFilesFlag) {
              reject(new Error('User Cancelled'));
              return;
            }
            client.listObjects(region, folderObject.bucket, folderObject.path.toString(), { nextContinuationToken: marker, maxKeys: 1000 }).then((listedObjects) => {
              if (stopDeleteFilesFlag) {
                reject(new Error('User Cancelled'));
                return;
              } else if (!listedObjects.objects || listedObjects.objects.length === 0) {
                resolve();
                return;
              }

              const objects = listedObjects.objects.map((object) => Object.assign({ itemType: object.key.endsWith('/') ? 'folder' : 'file' }, object));

              progress.total += objects.length;
              progressFn(progress);

              let promise = client.deleteObjects(region, folderObject.bucket, objects.map((object) => object.key), deleteCallback(objects));
              if (listedObjects.nextContinuationToken) {
                const promises = [promise];
                promises.push(_doDeleteFolder(client, region, folderObject, progress, progressFn, listedObjects.nextContinuationToken, deleteCallback));
                promise = Promise.all(promises);
              }
              promise.then(resolve).catch(reject);
            }).catch(reject);
          });
        }
      }
    }

    function stopDeleteFiles() {
      stopDeleteFilesFlag = true;
    }

    function parseKodoPath(s3Path) {
      const KODO_ADDR_PROTOCOL = 'kodo://';

      if (!s3Path || s3Path.indexOf(KODO_ADDR_PROTOCOL) == -1 || s3Path == KODO_ADDR_PROTOCOL) {
        return {};
      }

      const str = s3Path.substring(KODO_ADDR_PROTOCOL.length);
      const index = str.indexOf('/');
      let bucketName, key = '';
      if (index == -1) {
        bucketName = str;
      } else {
        bucketName = str.substring(0, index);
        key = str.substring(index + 1);
      }

      return { bucketName: bucketName, key: key };
    }

    function signatureUrl(region, bucket, key, domain, expires, opt) {
      const deadline = new Date();
      deadline.setSeconds(deadline.getSeconds() + expires || 60);

      return new Promise((resolve, reject) => {
        getDefaultClient(opt).enter('signatureUrl', (client) => client.getObjectURL(region, { bucket: bucket, key: key.toString() }, domain, deadline)).
          then(resolve).
          catch((err) => {
            handleError(err);
            reject(err);
          });
      });
    }

    function getRegions(opt) {
      return new Promise((resolve, reject) => {
        getDefaultClient(opt).enter('getRegions', (_, regionOptions) => getRegionService(opt).getAllRegions(regionOptions)).
          then((regions) => {
            if (regions && regions.length > 0) {
              const lang = $rootScope.langSettings.lang.replace('-', '_');
              regions.forEach((region) => {
                if (region.translatedLabels && region.translatedLabels[lang]) {
                  region.translatedLabel = region.translatedLabels[lang];
                } else {
                  region.translatedLabel = region.label;
                }
              });
            }
            resolve(regions);
          }).catch((err) => {
            handleError(err);
            reject(err);
          });
      });
    }

    function clientBackendMode(opt) {
      const adapterOption = getAdapterOption(opt);
      const usePublicCloud = AuthInfo.usePublicCloud();
      if (adapterOption.regions && adapterOption.regions.length > 0 && !adapterOption.preferKodoAdapter || adapterOption.preferS3Adapter || !usePublicCloud) {
        return S3_MODE;
      } else {
        return KODO_MODE;
      }
    }

    function getRegionService(opt) {
      const adapterOption = getAdapterOption(opt);
      const cacheKey = makeAdapterCacheKey(adapterOption.accessKey, adapterOption.secretKey, adapterOption.ucUrl);

      if (regionServicesCache[cacheKey]) {
        return regionServicesCache[cacheKey];
      } else {
        const regionService = new RegionService(adapterOption);
        regionServicesCache[cacheKey] = regionService;
        return regionService;
      }
    }

    function getDefaultClient(opt) {
      switch (clientBackendMode(opt)) {
        case S3_MODE:
          return getS3Client(opt);
        case KODO_MODE:
          return getKodoClient(opt);
      }
    }

    function getKodoClient(opt) {
      const adapterOption = getAdapterOption(opt);
      const cacheKey = makeAdapterCacheKey(adapterOption.accessKey, adapterOption.secretKey, adapterOption.ucUrl);

      if (kodoAdaptersCache[cacheKey]) {
        return kodoAdaptersCache[cacheKey];
      } else {
        const qiniuAdapter = getQiniuAdapter(adapterOption.accessKey, adapterOption.secretKey, adapterOption.ucUrl, adapterOption.regions);
        const kodoClient = qiniuAdapter.mode(KODO_MODE, {
          appName: adapterOption.appName,
          appVersion: adapterOption.appVersion,
          requestCallback: debugRequest(KODO_MODE),
          responseCallback: debugResponse(KODO_MODE),
        });
        kodoAdaptersCache[cacheKey] = kodoClient;
        return kodoClient;
      }
    }

    function getS3Client(opt) {
      const adapterOption = getAdapterOption(opt);
      const cacheKey = makeAdapterCacheKey(adapterOption.accessKey, adapterOption.secretKey, adapterOption.ucUrl);

      if (s3AdaptersCache[cacheKey]) {
        return s3AdaptersCache[cacheKey];
      } else {
        const qiniuAdapter = getQiniuAdapter(adapterOption.accessKey, adapterOption.secretKey, adapterOption.ucUrl, adapterOption.regions);
        const s3Client = qiniuAdapter.mode(S3_MODE, {
          appName: adapterOption.appName,
          appVersion: adapterOption.appVersion,
          requestCallback: debugRequest(S3_MODE),
          responseCallback: debugResponse(S3_MODE),
        });
        s3AdaptersCache[cacheKey] = s3Client;
        return s3Client;
      }
    }

    function clearAllCache() {
      Object.keys(s3AdaptersCache).forEach((key) => {
        s3AdaptersCache[key].clearCache();
        delete s3AdaptersCache[key];
      });
      Object.keys(kodoAdaptersCache).forEach((key) => {
        kodoAdaptersCache[key].clearCache();
        delete kodoAdaptersCache[key];
      });
      Object.keys(regionServicesCache).forEach((key) => {
        regionServicesCache[key].clearCache();
        delete regionServicesCache[key];
      });
    }

    function debugRequest(mode) {
      return (request) => {
        if (settingsSvs.isDebug.get() === 0) {
          return;
        }
        let url = undefined, method = undefined, headers = undefined, data = undefined;
        if (request) {
          url = request.url;
          method = request.method;
          headers = request.headers;
          data = request.data;
        }
        console.info('>>', mode, 'REQ_URL:', url, 'REQ_METHOD:', method, 'REQ_HEADERS:', headers, 'REQ_DATA:', data);
      };
    }

    function debugResponse(mode) {
      return (response) => {
        if (settingsSvs.isDebug.get() === 0) {
          return;
        }

        let requestUrl = undefined, requestMethod = undefined, requestHeaders = undefined, requestData = undefined,
          responseStatusCode = undefined, responseHeaders = undefined, responseInterval = undefined, responseData = undefined, responseError = undefined;
        if (response) {
          responseStatusCode = response.statusCode;
          responseHeaders = response.headers;
          responseInterval = response.interval;
          responseData = response.data;
          responseError = response.error;
          if (response.request) {
            requestUrl = response.request.url;
            requestMethod = response.request.method;
            requestHeaders = response.request.headers;
            requestData = response.request.data;
          }
        }
        console.info('<<', mode, 'REQ_URL:', requestUrl, 'REQ_METHOD:', requestMethod, 'REQ_HEADERS: ', requestHeaders, 'REQ_DATA: ', requestData,
          'RESP_STATUS:', responseStatusCode, 'RESP_HEADERS:', responseHeaders, 'RESP_INTERVAL:', responseInterval, 'ms RESP_DATA:', responseData, 'RESP_ERROR:', responseError);
      };
    }

    function getQiniuAdapter(accessKey, secretKey, ucUrl, regions) {
      if (!accessKey || !secretKey) {
        throw new Error('`accessKey` or `secretKey` is unavailable');
      }
      return new Qiniu(accessKey, secretKey, ucUrl, `Kodo-Browser/${Global.app.version}`, regions);
    }

    function getAdapterOption(opt) {
      const result = {
        appName: 'kodo-browser', appVersion: Global.app.version,
      };

      if (typeof opt === 'object' && opt.id && opt.secret) {
        result.accessKey = opt.id;
        result.secretKey = opt.secret;
        config = Config.load(opt.isPublicCloud);
      } else {
        result.accessKey = AuthInfo.get().id;
        result.secretKey = AuthInfo.get().secret;
        config = Config.load();
      }

      if (config.ucUrl) {
        result.ucUrl = config.ucUrl;
      }
      result.regions = config.regions || [];
      if (opt && opt.preferS3Adapter) {
        result.preferS3Adapter = opt.preferS3Adapter;
      }
      return result;
    }

    function makeAdapterCacheKey(accessKey, secretKey, ucUrl) {
      return `${accessKey}:${secretKey}:${ucUrl}`;
    }

    function handleError(err) {
      if (err.code === 'InvalidAccessKeyId') {
        $state.go('login');
      } else {
        if (!err.code) {
          if (err.message.indexOf('Failed to fetch') != -1) {
            err = {
              code: 'Error',
              message: 'Connection Error'
            };
          } else
            err = {
              code: 'Error',
              message: err.message
            };
        }

        console.error(err);
        if (err.code === 'Forbidden' || err.code === 'AccessDenied') {
          Toast.error(T('permission.denied'));
        } else {
          Toast.error(err.code + ': ' + err.message);
        }
      }
    }
  }
]);

angular.module("web").factory("safeApply", [
  function() {
    return function($scope, fn) {
      if (!$scope.$root) return;
      var phase = $scope.$root.$$phase;
      if (phase == "$apply" || phase == "$digest") {
        if (fn) {
          $scope.$eval(fn);
        }
      } else {
        if (fn) {
          $scope.$apply(fn);
        } else {
          $scope.$apply();
        }
      }
    };
  }
]);

angular.module("web").factory("settingsSvs", [
  function () {
    return {
      isDebug: {
        get: function () {
          return parseInt(localStorage.getItem("isDebug") || 0);
        },
        set: function (v) {
          return localStorage.setItem("isDebug", v);
        }
      },

      autoUpgrade: {
        get: function () {
          return parseInt(localStorage.getItem("autoUpgrade") || 0);
        },
        set: function (v) {
          return localStorage.setItem("autoUpgrade", v);
        }
      },

      resumeUpload: {
        get: function () {
          return parseInt(localStorage.getItem("resumeUpload") || 0);
        },
        set: function (v) {
          return localStorage.setItem("resumeUpload", v);
        }
      },

      maxUploadConcurrency: {
        get: function () {
          return parseInt(localStorage.getItem("maxUploadConcurrency") || 1);
        },
        set: function (v) {
          return localStorage.setItem("maxUploadConcurrency", v);
        }
      },

      multipartUploadSize: {
        get: function () {
          return parseInt(localStorage.getItem("multipartUploadSize") || 8);
        },
        set: function (v) {
          if (parseInt(v) % 4 == 0) {
            return localStorage.setItem("multipartUploadSize", v);
          }
        }
      },

      multipartUploadThreshold: {
        get: function () {
          return parseInt(localStorage.getItem("multipartUploadThreshold") || 100);
        },
        set: function (v) {
          return localStorage.setItem("multipartUploadThreshold", v);
        }
      },

      uploadSpeedLimitEnabled: {
        get: function () {
          return parseInt(localStorage.getItem("uploadSpeedLimitEnabled") || 0);
        },
        set: function (v) {
          return localStorage.setItem("uploadSpeedLimitEnabled", v);
        }
      },

      uploadSpeedLimitKBperSec: {
        get: function () {
          return parseInt(localStorage.getItem("uploadSpeedLimit") || 1024);
        },
        set: function (v) {
          return localStorage.setItem("uploadSpeedLimit", v);
        }
      },

      resumeDownload: {
        get: function () {
          return parseInt(localStorage.getItem("resumeDownload") || 0);
        },
        set: function (v) {
          return localStorage.setItem("resumeDownload", v);
        }
      },

      maxDownloadConcurrency: {
        get: function () {
          return parseInt(localStorage.getItem("maxDownloadConcurrency") || 1);
        },
        set: function (v) {
          return localStorage.setItem("maxDownloadConcurrency", v);
        }
      },

      multipartDownloadSize: {
        get: function () {
          return parseInt(localStorage.getItem("multipartDownloadSize") || 8);
        },
        set: function (v) {
          return localStorage.setItem("multipartDownloadSize", v);
        }
      },

      multipartDownloadThreshold: {
        get: function () {
          return parseInt(localStorage.getItem("multipartDownloadThreshold") || 100);
        },
        set: function (v) {
          return localStorage.setItem("multipartDownloadThreshold", v);
        }
      },

      downloadSpeedLimitEnabled: {
        get: function () {
          return parseInt(localStorage.getItem("downloadSpeedLimitEnabled") || 0);
        },
        set: function (v) {
          return localStorage.setItem("downloadSpeedLimitEnabled", v);
        }
      },

      downloadSpeedLimitKBperSec: {
        get: function () {
          return parseInt(localStorage.getItem("downloadSpeedLimit") || 1024);
        },
        set: function (v) {
          return localStorage.setItem("downloadSpeedLimit", v);
        }
      },

      externalPathEnabled: {
        get: function () {
          return parseInt(localStorage.getItem("externalPathEnabled") || 0);
        },
        set: function (v) {
          return localStorage.setItem("externalPathEnabled", v);
        }
      },

      stepByStepLoadingFiles: {
        get: function () {
          return parseInt(localStorage.getItem("stepByStepLoadingFiles") || 0);
        },
        set: function (v) {
          return localStorage.setItem("stepByStepLoadingFiles", v);
        }
      },

      filesLoadingSize: {
        get: function () {
          return parseInt(localStorage.getItem("filesLoadingSize") || 500);
        },
        set: function (v) {
          return localStorage.setItem("filesLoadingSize", v);
        }
      },

      historiesLength: {
        get: function () {
          return parseInt(localStorage.getItem("multipartDownloadThreshold") || 100);
        },
        set: function (v) {
          return localStorage.setItem("multipartDownloadThreshold", v);
        }
      }
    };
  }
]);

angular.module("web").factory("subUserAKSvs", [
  "indexDBSvs",
  function(indexDBSvs) {
    var DBNAME = "subUserAkList";
    var DBVERSION = 1;
    var STORENAME = "subUserAK";
    var INITSTORE = { subUserAK: { keyPath: "AccessKeyId" } };
    return {
      save: save,
      get: get,
      list: list,
      delete: del
    };

    function list() {
      return indexDBSvs.open(DBNAME, DBVERSION, INITSTORE).then(function(db) {
        return indexDBSvs.list(db, STORENAME);
      });
    }
    function save(data) {
      return indexDBSvs.open(DBNAME, DBVERSION, INITSTORE).then(function(db) {
        return indexDBSvs.update(db, STORENAME, data["AccessKeyId"], data);
      });
    }
    function get(AccessKeyId) {
      return indexDBSvs.open(DBNAME, DBVERSION, INITSTORE).then(function(db) {
        return indexDBSvs.get(db, STORENAME, AccessKeyId);
      });
    }
    function del(AccessKeyId) {
      return indexDBSvs.open(DBNAME, DBVERSION, INITSTORE).then(function(db) {
        return indexDBSvs.delete(db, STORENAME, AccessKeyId);
      });
    }
  }
]);

angular.module("web").factory("UploadMgr", [
  "$q",
  "$timeout",
  "$translate",
  "QiniuClient",
  "AuthInfo",
  "Config",
  "settingsSvs",
  function (
    $q,
    $timeout,
    $translate,
    QiniuClient,
    AuthInfo,
    Config,
    settingsSvs
  ) {
    const fs = require("fs"),
          http = require("http"),
          https = require("https"),
          path = require("path"),
          os = require("os"),
          qiniuPath = require("qiniu-path"),
          QiniuStore = require("./node/qiniu-store"),
          T = $translate.instant;

    var $scope;
    var concurrency = 0;
    var stopCreatingFlag = false;

    return {
      init: init,
      createUploadJobs: createUploadJobs,
      trySchedJob: trySchedJob,
      trySaveProg: trySaveProg,

      stopCreatingJobs: function () {
        stopCreatingFlag = true;
      }
    };

    function init(scope) {
      $scope = scope;
      $scope.lists.uploadJobList = [];

      angular.forEach(tryLoadProg(), (prog) => {
        const job = createJob(prog);
        if (job.status === "waiting" || job.status === "running") {
          job.stop();
        }
        addEvents(job);
      });
    }

    /**
      * @param  options   { region, from, to, progress, checkPoints, ...}
      * @param  options.from {name, path}
      * @param  options.to   {bucket, key}
      * @return job  { start(), stop(), status, progress }
                job.events: statuschange, progress
      */
    function createJob(options) {
      const bucket = options.to.bucket,
            region = options.region;

      console.info(
        "PUT",
        "::",
        region,
        "::",
        options.from.path + "/" + options.from.name,
        "=>",
        options.to.bucket + "/" + options.to.key
      );

      const config = Config.load();

      options.clientOptions = {
        accessKey: AuthInfo.get().id,
        secretKey: AuthInfo.get().secret,
        ucUrl: config.ucUrl,
        regions: config.regions || [],
      };
      options.region = region;
      options.resumeUpload = (settingsSvs.resumeUpload.get() == 1);
      options.multipartUploadSize = settingsSvs.multipartUploadSize.get();
      options.multipartUploadThreshold = settingsSvs.multipartUploadThreshold.get();
      options.uploadSpeedLimit = (settingsSvs.uploadSpeedLimitEnabled.get() == 1 && settingsSvs.uploadSpeedLimitKBperSec.get());
      options.isDebug = (settingsSvs.isDebug.get() == 1);

      const store = new QiniuStore();
      return store.createUploadJob(options);
    }

    /**
     * upload
     * @param filePaths []  {array<string>}  有可能是目录，需要遍历
     * @param bucketInfo {object} {bucket, region, key}
     * @param jobsAddingFn {Function} 快速加入列表回调方法， 返回jobs引用，但是该列表长度还在增长。
     * @param jobsAddedFn {Function} 加入列表完成回调方法， jobs列表已经稳定
     */
    function createUploadJobs(filePaths, bucketInfo, jobsAddingFn) {
      stopCreatingFlag = false;

      _kdig(filePaths, () => {
        if (jobsAddingFn) {
          jobsAddingFn();
        }
      });
      return;

      function _kdig(filePaths, fn) {
        let t = [];
        let c = 0;
        const len = filePaths.length;

        function _dig() {
          if (stopCreatingFlag) {
            return;
          }

          const n = filePaths[c];
          const dirPath = n.parentDirectoryPath();

          dig(filePaths[c].toString(), dirPath, (jobs) => {
            t = t.concat(jobs);
            c++;

            if (c >= len) {
              fn(t);
            } else {
              _dig();
            }
          });
        }
        _dig();
      }

      function loop(parentPath, dirPath, arr, callFn) {
        var t = [];
        var len = arr.length;
        var c = 0;
        if (len == 0) {
          callFn([]);
        } else {
          inDig();
        }

        //串行
        function inDig() {
          dig(path.join(parentPath.toString(), arr[c].toString()), dirPath, (jobs) => {
            t = t.concat(jobs);

            c++;
            if (c >= len) {
              callFn(t);
            } else {
              if (stopCreatingFlag) {
                return;
              }

              inDig();
            }
          });
        }
      }

      function dig(absPath, dirPath, callFn) {
        if (stopCreatingFlag) {
          return;
        }

        const fileName = path.basename(absPath);
        let filePath = path.relative(dirPath.toString(), absPath);

        if (bucketInfo.key) {
          if (bucketInfo.key.endsWith('/')) {
            filePath = bucketInfo.key + filePath;
          } else {
            filePath = bucketInfo.key + '/' + filePath;
          }
        }
        const fileStat = fs.statSync(absPath);

        if (fileStat.isDirectory()) {
          //创建目录
          let subDirPath = filePath + '/';
          if (path.sep == '\\') {
            subDirPath = subDirPath.replace(/\\/g, '/');
          }
          subDirPath = qiniuPath.fromQiniuPath(subDirPath);

          //递归遍历目录
          fs.readdir(absPath, (err, arr) => {
            if (err) {
              console.error(err.stack);
            } else if (arr.length > 0 || arr.length === 0 && $scope.emptyFolderUploading.enabled) {
              QiniuClient
                .createFolder(bucketInfo.regionId, bucketInfo.bucketName, subDirPath)
                .then(() => {
                  checkNeedRefreshFileList(bucketInfo.bucketName, subDirPath);
                  loop(absPath, dirPath, arr, (jobs) => { $timeout(() => { callFn(jobs); }, 1); });
                });
            } else {
              $timeout(() => { callFn([]); }, 1);
            }
          });
        } else {
          //文件

          //修复 window 下 \ 问题
          if (path.sep == '\\') {
            filePath = filePath.replace(/\\/g, '/');
          }

          const job = createJob({
                        region: bucketInfo.regionId,
                        from: {
                          name: fileName,
                          path: absPath
                        },
                        to: {
                          bucket: bucketInfo.bucketName,
                          key: filePath
                        },
                        overwrite: $scope.overwriteUploading.enabled,
                        backendMode: bucketInfo.qiniuBackendMode,
                      });
          addEvents(job);
          $timeout(() => { callFn([job]); }, 1);
        }
      }
    }

    function addEvents(job) {
      if (!job.uploadedParts) {
        job.uploadedParts = [];
      }

      $scope.lists.uploadJobList.push(job);

      trySchedJob();
      trySaveProg();

      $timeout(() => {
        $scope.calcTotalProg();
      });

      job.on('fileDuplicated', (data) => {
        concurrency--;
        $timeout(() => {
          $scope.calcTotalProg();
        });
      });
      job.on("partcomplete", (data) => {
        job.uploadedId = data.uploadId;
        job.uploadedParts[data.part.partNumber] = data.part;

        trySaveProg();

        $timeout($scope.calcTotalProg);
      });
      job.on("statuschange", (status) => {
        if (status == "stopped") {
          concurrency--;
          $timeout(trySchedJob);
        }

        trySaveProg();
        $timeout($scope.calcTotalProg);
      });
      job.on("speedchange", () => {
        $timeout($scope.calcTotalProg);
      });
      job.on("complete", () => {
        concurrency--;

        $timeout(() => {
          trySchedJob();
          $scope.calcTotalProg();
          checkNeedRefreshFileList(job.to.bucket, qiniuPath.fromLocalPath(job.to.key));
        });
      });
      job.on("error", (err) => {
        if (err) {
          console.error(`upload kodo://${job.to.bucket}/${job.to.key} error: ${err}`);
        }
        if (job.message) {
          switch (job.message.error) {
          case 'Forbidden':
            job.message.i18n = T('permission.denied');
          }
        }

        concurrency--;
        $timeout(() => {
          trySchedJob();
          $scope.calcTotalProg();
        });
      });
    }

    function trySchedJob() {
      var maxConcurrency = settingsSvs.maxUploadConcurrency.get();
      var isDebug = (settingsSvs.isDebug.get() == 1);

      concurrency = Math.max(0, concurrency);
      if (isDebug) {
        console.log(`[JOB] upload max: ${maxConcurrency}, cur: ${concurrency}, jobs: ${$scope.lists.uploadJobList.length}`);
      }

      if (concurrency < maxConcurrency) {
        var jobs = $scope.lists.uploadJobList;

        for (var i = 0; i < jobs.length && concurrency < maxConcurrency; i++) {
          var job = jobs[i];
          if (isDebug) {
            console.log(`[JOB] sched ${job.status} => ${JSON.stringify(job._config)}`);
          }

          if (job.status === 'waiting') {
            concurrency++;

            if (job.prog.resumable) {
              var progs = tryLoadProg();

              if (progs && progs[job.id]) {
                job.start(true, progs[job.id]);
              } else {
                job.start(true);
              }
            } else {
              job.start();
            }
          }
        }
      }
    }

    function trySaveProg() {
      var t = {};
      angular.forEach($scope.lists.uploadJobList, (job) => {
        if (job.status === 'finished') return;

        if (!job.uploadedParts) {
          job.uploadedParts = [];
        }

        if (fs.existsSync(job.from.path)) {
          const fileStat = fs.statSync(job.from.path);

          job.from.size = fileStat.size;
          job.from.mtime = fileStat.mtimeMs;

          t[job.id] = {
            region: job.region,
            to: job.to,
            from: job.from,
            prog: job.prog,
            status: job.status,
            message: job.message,
            uploadedId: job.uploadedId,
            uploadedParts: job.uploadedParts.map((part) => {
              return { PartNumber: part.partNumber, ETag: part.etag };
            }),
            overwrite: job.overwrite,
            backendMode: job.backendMode,
          };
        }
      });

      fs.writeFileSync(getProgFilePath(), JSON.stringify(t));
    }

    function tryLoadProg() {
      let progs = {};
      try {
        const data = fs.readFileSync(getProgFilePath());
        progs = JSON.parse(data);
      } catch (e) {}

      Object.entries(progs).forEach(([jobId, job]) => {
        if (!job.uploadedParts) {
          job.uploadedParts = [];
        }
        job.uploadedParts = job.uploadedParts.
          filter(part => part && part.PartNumber && part.ETag).
          map((part) => {
          return { partNumber: part.PartNumber, etag: part.ETag };
        });
        if (job.from && job.from.size && job.from.mtime) {
          if (fs.existsSync(job.from.path)) {
            const fileStat = fs.statSync(job.from.path);
            if (fileStat.size === job.from.size && job.from.mtime === fileStat.mtimeMs) {
              return;
            }
          }
        }
        job.uploadedParts = [];
        if (job.prog) {
          delete job.prog.loaded;
        }
      });

      return progs;
    }

    function getProgFilePath() {
      var folder = Global.config_path;
      if (!fs.existsSync(folder)) {
        fs.mkdirSync(folder);
      }

      const username = AuthInfo.get().id || "kodo-browser";
      return path.join(folder, "upprog_" + username + ".json");
    }

    function checkNeedRefreshFileList(bucket, key) {
      if ($scope.currentInfo.bucketName === bucket) {
        if ($scope.currentInfo.key === key.parentDirectoryPath().toString()) {
          $scope.$emit('refreshFilesList');
        }
      }
    }
  }
]);

angular.module("web").factory("utilSvs", [
  "$timeout",
  function($timeout) {
    return {
      leftTime: leftTime
    };

    function leftTime(ms) {
      if (isNaN(ms)) {
        return "";
      }
      if (ms <= 0) return 0;
      else if (ms < 1000) return ms + "ms";

      //return moment.duration(ms).humanize();
      var t = [];

      var d = Math.floor(ms / 24 / 3600 / 1000);
      if (d) {
        ms = ms - d * 3600 * 1000 * 24;
        t.push(d + "D");
      }
      var h = Math.floor(ms / 3600 / 1000);
      if (h) {
        ms = ms - h * 3600 * 1000;
        t.push(h + "h");
      }
      var m = Math.floor(ms / 60 / 1000);
      if (m) {
        ms = ms - m * 60 * 1000;
        t.push(m + "m");
      }
      var s = Math.floor(ms / 1000);
      if (s) {
        ms = ms - s * 1000;
        t.push(s + "s");
      }

      return t.join(" ");
    }
  }
]);

angular.module("web").controller("loginCtrl", [
  "$q",
  "$timeout",
  "$scope",
  "$rootScope",
  "$translate",
  "$uibModal",
  "Auth",
  "AuthInfo",
  "$location",
  "Const",
  "Config",
  "Dialog",
  "Toast",
  "AkHistory",
  "AuditLog",
  function (
    $q,
    $timeout,
    $scope,
    $rootScope,
    $translate,
    $modal,
    Auth,
    AuthInfo,
    $location,
    Const,
    Config,
    Dialog,
    Toast,
    AkHistory,
    AuditLog,
  ) {
    const T = $translate.instant;

    angular.extend($scope, {
      item: {},
      clouds: [{
        name: "auth.defaultCloud",
        value: "default"
      }, {
        name: "auth.customizedCloud",
        value: "customized"
      }],
      isPrivateCloudConfigured: isPrivateCloudConfigured(),
      selectedCloud: selectedCloud(),

      showGuestNav: 1,

      onSubmit: onSubmit,
      showCustomizedCloud: showCustomizedCloud,
      showAkHistories: showAkHistories,

      open: open,
    });

    function selectedCloud() {
      return AuthInfo.usePublicCloud() || !Config.exists() ? 'default' : 'customized';
    }

    function isPrivateCloudConfigured() {
      const df = $q.defer();

      $timeout(() => {
        if (Config.exists()) {
          try {
            const config = Config.load(false);
            df.resolve(config && config.ucUrl);
          } catch (e) {
            df.resolve(false);
          }
        } else {
          df.resolve(false);
        }
      });

      return df.promise;
    }

    function open(a) {
      openExternal(a);
    }

    function showAkHistories() {
      $modal.open({
        templateUrl: "main/auth/modals/ak-histories-modal.html",
        controller: "akHistoriesModalCtrl",
        size: 'lg',
        resolve: {
          choose: function() {
            return function(history) {
              if (history.isPublicCloud) {
                $scope.selectedCloud = 'default';
              } else {
                $scope.selectedCloud = 'customized';
              }
              $scope.item.id = history.accessKeyId;
              $scope.item.secret = history.accessKeySecret;
              $scope.item.description = history.description;
            }
          }
        }
      }).result.then(angular.noop, angular.noop);
    }

    function showCustomizedCloud() {
      $modal.open({
        templateUrl: "main/auth/modals/customize-cloud-modal.html",
        controller: "customizeCloudModalCtrl"
      }).result.then(angular.noop, () => { $scope.isPrivateCloudConfigured = isPrivateCloudConfigured(); });
    }

    function onSubmit(form1) {
      if (!form1.$valid) return;

      const data = angular.copy($scope.item),
            isPublicCloud = $scope.selectedCloud === 'default';

      if (data.id) {
        data.id = data.id.trim();
      }
      if (data.secret) {
        data.secret = data.secret.trim();
      }
      data.isPublicCloud = isPublicCloud;

      Toast.info(T("logining"), 1000);

      Auth.login(data).then(() => {
        if (isPublicCloud) {
          AuthInfo.switchToPublicCloud();
        } else {
          AuthInfo.switchToPrivateCloud();
        }

        if (data.remember) {
          AkHistory.add(isPublicCloud, data.id, data.secret, data.description);
        }
        AuditLog.log('login');
        Toast.success(T("login.successfully"), 1000);
        $location.url("/");
      }).catch((err) => {
        Toast.error(err.code + ":" + err.message, 5000);
        Dialog.alert(T('auth.login.error.title'), T('auth.login.error.description'), null, 1);
      });

      return false;
    }
  }
]);

angular.module("web").controller("filesCtrl", [
  "$rootScope",
  "$scope",
  "$filter",
  "$uibModal",
  "$timeout",
  "$translate",
  "$location",
  "safeApply",
  "Auth",
  "AuthInfo",
  "AuditLog",
  "QiniuClient",
  "settingsSvs",
  "ExternalPath",
  "fileSvs",
  "Toast",
  "Dialog",
  "Customize",
  "Domains",
  function (
    $rootScope,
    $scope,
    $filter,
    $modal,
    $timeout,
    $translate,
    $location,
    safeApply,
    Auth,
    AuthInfo,
    AuditLog,
    QiniuClient,
    settingsSvs,
    ExternalPath,
    fileSvs,
    Toast,
    Dialog,
    Customize,
    Domains,
  ) {
    const deepEqual = require('fast-deep-equal'),
          { Base64 } = require('js-base64'),
          T = $translate.instant,
          { S3_MODE } = require('kodo-s3-adapter-sdk'),
          path = require('path'),
          qiniuPath = require('qiniu-path');

    angular.extend($scope, {
      showTab: 1,

      // supported mode: localBuckets, localFiles, externalPaths, externalFiles
      ref: {
        mode: 'localFiles',
        isListView: true
      },

      currentListView: true,
      keepMoveOptions: null,

      transVisible: localStorage.getItem("transVisible") == "true",
      toggleTransVisible: function (visible) {
        localStorage.setItem("transVisible", visible);

        $timeout(() => {
          $scope.transVisible = visible;
        });
      },

      // Read OEM Customization config to disable features
      disabledFeatures: Customize.disable,

      // search
      sch: {
        bucketName: "",
        objectName: "",
        externalPathName: ""
      },
      selectedDomain: {
        bucketName: null,
        domain: null,
      },
      domains: [],
      showDomains: showDomains,
      refreshDomains: refreshDomains,
      searchObjectName: searchObjectName,
      searchBucketName: searchBucketName,
      searchExternalPathName: searchExternalPathName,

      // bucket selection
      bucket_sel: null,
      selectBucket: selectBucket,

      // file selection
      sel: {
        all: false, //boolean
        has: [], //[] item: s3Object={name,path,...}
        x: {} //{} {'i_'+$index, true|false}
      },
      selectFile: selectFile,
      selectFileAndStopEventPropagation: selectFileAndStopEventPropagation,
      openFileAndStopEventPropagation: openFileAndStopEventPropagation,
      toLoadMore: toLoadMore,

      stepByStepLoadingFiles: stepByStepLoadingFiles,

      // bucket ops
      showAddBucket: showAddBucket,
      showDeleteBucket: showDeleteBucket,

      // external links selection
      external_path_sel: null,
      selectExternalPath: selectExternalPath,

      // bucket ops
      showAddExternalPath: showAddExternalPath,
      showDeleteExternalPath: showDeleteExternalPath,

      // object ops
      showAddFolder: showAddFolder,
      showRename: showRename,
      showMove: showMove,
      showDeleteFiles: showDeleteFiles,
      showDeleteFilesSelected: showDeleteFilesSelected,
      showRestoreFiles: showRestoreFiles,
      showRestoreFilesSelected: showRestoreFilesSelected,
      showSetStorageClassOfFiles: showSetStorageClassOfFiles,
      showSetStorageClassOfFilesSelected: showSetStorageClassOfFilesSelected,

      // upload && download
      handlers: {
        uploadFilesHandler: null,
        downloadFilesHandler: null
      },
      handlerDrop: handlerDrop,
      showUploadDialog: showUploadDialog,
      showDownloadDialog: showDownloadDialog,
      showDownload: showDownload,

      // utils
      gotoAddress: gotoAddress,
      gotoExternalMode: gotoExternalMode,
      gotoLocalMode: gotoLocalMode,
      gotoExternalPath: gotoExternalPath,
      showDownloadLink: showDownloadLink,
      showDownloadLinkOfFilesSelected: showDownloadLinkOfFilesSelected,
      showPreview: showPreview,

      showPaste: showPaste,
      disablePaste: disablePaste,
      cancelPaste: cancelPaste
    });

    $scope.$watch("ref.isListView", (v) => {
      if ($scope.ref.mode === 'localBuckets') {
        initBucketSelect();
      } else if ($scope.ref.mode === 'externalPaths') {
        initExternalPathSelect();
      } else {
        initFilesSelect();
      }

      $timeout(() => {
        if ($scope.currentListView) {
          $scope.currentListView = v;

          return;
        }

        if (!v) {
          $scope.currentListView = v;

          return;
        }

        if ($scope.ref.mode === 'localBuckets') {
          showBucketsTable($scope.buckets);
        } else if ($scope.ref.mode === 'externalPaths') {
          showExternalPathsTable($scope.externalPaths);
        } else {
          showFilesTable($scope.objects);
        }
      });
    });

    /////////////////////////////////
    function gotoAddress(bucket, prefix) {
      const KODO_ADDR_PROTOCOL = 'kodo://';
      let kodoAddress = KODO_ADDR_PROTOCOL;

      if (bucket.startsWith(kodoAddress)) {
        bucket = bucket.substring(kodoAddress.length);
      }

      if (bucket) {
        kodoAddress += `${bucket}/${prefix || ""}`;
      }

      $rootScope.$broadcast("gotoKodoAddress", kodoAddress);
    }

    function gotoExternalMode() {
      $rootScope.$broadcast("gotoExternalMode");
    }

    function gotoLocalMode() {
      $rootScope.$broadcast("gotoLocalMode");
    }

    function getCurrentPath() {
      return `kodo://${$scope.currentInfo.bucketName}/${$scope.currentInfo.key}`;
    }

    function gotoExternalPath(bucketId, prefix) {
      let externalPath = "kodo://";

      if (bucketId) {
        externalPath += `${bucketId}/${prefix || ""}`;
      }

      $rootScope.$broadcast("gotoExternalPath", externalPath);
    }

    function getCurrentExternalPath() {
      return `kodo://${$scope.currentInfo.bucketName}/${$scope.currentInfo.key}`;
    }

    /////////////////////////////////

    function stepByStepLoadingFiles() {
      return settingsSvs.stepByStepLoadingFiles.get() == 1;
    }

    /////////////////////////////////

    function showDomains() {
      return AuthInfo.usePublicCloud() && $scope.ref.mode.startsWith('local');
    }

    function refreshDomains() {
      const info = $scope.currentInfo;
      Domains.list(info.regionId, info.bucketName, info.bucketGrantedPermission).
              then((domains) => {
                $scope.domains = domains;
                let found = false;
                if ($scope.selectedDomain.domain !== null) {
                  angular.forEach(domains, (domain) => {
                    if (domain.name() === $scope.selectedDomain.domain.name()) {
                      $scope.selectedDomain.domain = domain;
                      found = true;
                    }
                  })
                }
                if (!found) {
                  angular.forEach(domains, (domain) => {
                    if (domain.default()) {
                      $scope.selectedDomain.domain = domain;
                      found = true;
                    }
                  });
                }
                if (!found) {
                    $scope.selectedDomain.domain = domains[0];
                }
              }, (err) => {
                console.error(err);
                Toast.error(err);
              });
    }

    /////////////////////////////////
    var refreshTid;

    $scope.$on("refreshFilesList", (e) => {
      $timeout.cancel(refreshTid);

      const bucketName = $scope.currentInfo.bucketName,
            key = $scope.currentInfo.key;
      refreshTid = $timeout(() => {
        if (bucketName === $scope.currentInfo.bucketName &&
            key === $scope.currentInfo.key) {
          gotoAddress(bucketName, key);
        }
      }, 600);
    });

    var searchTid;

    function searchObjectName() {
      $timeout.cancel(searchTid);

      var info = angular.copy($scope.currentInfo);
      searchTid = $timeout(() => {
        if (info.bucketName === $scope.currentInfo.bucketName &&
            info.key === $scope.currentInfo.key) {
          info.key += $scope.sch.objectName;
          listFiles(info, true);
        }
      }, 600);
    }

    function searchBucketName() {
      $timeout.cancel(searchTid);
      searchTid = $timeout(listBuckets, 600);
    }

    function searchExternalPathName() {
      $timeout.cancel(searchTid);
      searchTid = $timeout(listExternalPaths, 600);
    }

    /////////////////////////////////
    $timeout(init, 100);

    function init() {
      var user = AuthInfo.get();
      if (!user.isAuthed) {
        Auth.logout().then(() => {
          $location.url("/login");
        });

        return;
      }

      $rootScope.currentUser = user;
      $scope.ref.mode = 'localBuckets';

      $timeout(() => {
        addEvents();
        $scope.$broadcast("filesViewReady");
      });
    }

    function addEvents() {
      const KODO_ADDR_PROTOCOL = 'kodo://',
            onKodoAddressChange = (addr) => {
        let fileName;

        const info = QiniuClient.parseKodoPath(addr);
        info.qiniuBackendMode = QiniuClient.clientBackendMode();
        $scope.currentInfo = info;
        if (!info.bucketName) {
          $scope.domains = [];
          $scope.selectedDomain.bucketName = null;
          $scope.selectedDomain.domain = null;
        }

        if (info.key) {
          const lastSlash = info.key.lastIndexOf("/");
          // if not endswith /
          if (lastSlash != info.key.length - 1) {
            fileName = info.key.substring(lastSlash + 1);
            info.key = info.key.substring(0, lastSlash + 1);
          }
        }

        if (info.bucketName) {
          // list objects
          const bucketInfo = $rootScope.bucketsMap[info.bucketName];
          if (bucketInfo) {
            if (!bucketInfo.regionId) {
              Toast.error("Forbidden");
              clearFilesList();
              return;
            }
            $scope.currentInfo.regionId = bucketInfo.regionId;
            $scope.ref.mode = 'localFiles';
            info.bucketName = bucketInfo.name;
            info.bucketId = bucketInfo.id;
            if (bucketInfo.grantedPermission) {
              info.bucketGrantedPermission = bucketInfo.grantedPermission;
            }
          } else {
            const regionId = ExternalPath.getRegionByBucketSync(info.bucketName);
            if (regionId) {
              $scope.currentInfo.regionId = regionId;
              $scope.ref.mode = 'externalFiles';
              info.bucketName = info.bucketName;
              info.qiniuBackendMode = S3_MODE;
              // TODO: Add bucket id here
            } else {
              Toast.error("Forbidden");
              clearFilesList();
              return;
            }
          }

          if (info.bucketName !== $scope.selectedDomain.bucketName) {
            $scope.domains = [Domains.s3(info.regionId, info.bucketName)];
            $scope.selectedDomain.bucketName = info.bucketName;
            $scope.selectedDomain.domain = $scope.domains[0];
          }

          $scope.currentBucketId = info.bucketId;
          $scope.currentBucketName = info.bucketName;

          // try to resolve bucket perm
          const user = $rootScope.currentUser;

          if (showDomains()) {
            refreshDomains();
          }
          // search
          if (fileName) {
            $scope.sch.objectName = fileName;

            searchObjectName();
          } else {
            $timeout(listFiles, 100);
          }
        } else {
          // list buckets
          delete $scope.currentBucketId;
          delete $scope.currentBucketName;

          if ($scope.ref.mode.startsWith('local')) {
            $scope.ref.mode = 'localBuckets';
            $timeout(listBuckets, 100);
          } else if ($scope.ref.mode.startsWith('external')) {
            $scope.ref.mode = 'externalPaths';
            $timeout(listExternalPaths, 100);
          } else {
            throw new Error('Unrecognized mode');
          }
        }
      };

      $scope.$on("kodoAddressChange", (evt, addr) => {
        evt.stopPropagation();

        console.log(`on:kodoAddressChange: ${addr}`);
        if ($rootScope.bucketsMap) {
          onKodoAddressChange(addr);
        } else {
          $scope.isLoading = true;

          QiniuClient.listAllBuckets(getQiniuClientOpt()).then((bucketList) => {
            $rootScope.bucketsMap = {};
            if (bucketList) {
              bucketList.forEach((bucket) => {
                $rootScope.bucketsMap[bucket.name] = bucket;
              });
            }
            onKodoAddressChange(addr);
          }).finally(() => {
            $scope.isLoading = false;
            safeApply($scope);
          });
        }
      });
    }

    function listBuckets(fn) {
      clearFilesList();
      $scope.isLoading = true;

      QiniuClient.listAllBuckets(getQiniuClientOpt()).then((bucketList) => {
        $rootScope.bucketsMap = {};
        if (bucketList) {
          bucketList.forEach((bucket) => {
            $rootScope.bucketsMap[bucket.name] = bucket;
          });
        }

        const buckets = [];
        for (const bucketName in $rootScope.bucketsMap) {
          if ($scope.sch.bucketName && bucketName.indexOf($scope.sch.bucketName) < 0) {
            continue;
          }
          buckets.push($rootScope.bucketsMap[bucketName]);
        }
        $scope.buckets = buckets;
        showBucketsTable(buckets);
        if (fn) fn(null);
      }).catch((err) => {
        if (fn) fn(err);
      }).finally(() => {
        $scope.isLoading = false;
        safeApply($scope);
      });
    }

    function listFiles(info, keepObjectSearchName) {
      clearFilesList();
      if (!keepObjectSearchName) {
        $scope.sch.objectName = '';
      }
      $scope.isLoading = true;

      info = info || angular.copy($scope.currentInfo);

      tryListFiles(info, null, (err, files) => {
        $scope.isLoading = false;

        if (info.bucketName !== $scope.currentInfo.bucketName ||
            info.key !== $scope.currentInfo.key + $scope.sch.objectName) {
          return;
        }

        if (err) {
          Toast.error(err.message);
          return;
        }

        showFilesTable(files);
      });
    }

    function tryListFiles(info, marker, fn) {
      if (!info || !info.bucketName) {
        return;
      }

      const filesLoadingSize = settingsSvs.filesLoadingSize.get();
      QiniuClient.listFiles(
        info.regionId, info.bucketName, info.key, marker || undefined,
        angular.extend(getQiniuClientOpt(), { maxKeys: filesLoadingSize, minKeys: filesLoadingSize }),
      ).then((result) => {
        $timeout(() => {
          if (info.bucketName !== $scope.currentInfo.bucketName ||
              info.key !== $scope.currentInfo.key + $scope.sch.objectName) {
            return;
          }

          const nextObjectsMarker = result.marker || null;
          $scope.nextObjectsMarker = nextObjectsMarker;
          $scope.nextObjectsMarkerInfo = info;

          $scope.objects = $scope.objects.concat(result.data);

          if ($scope.nextObjectsMarker) {
            if (!$scope.stepByStepLoadingFiles()) {
              $timeout(function() {
                tryLoadMore(info, nextObjectsMarker);
              }, 100);
            }
          }
        });

        if (fn) fn(null, result.data);
      }).catch((err) => {
        console.error(`list files: kodo://${info.bucketName}/${info.key}?marker=${marker}`, err);

        clearFilesList();

        if (fn) fn(err);
      });
    }

    var lastObjectsMarkerForLoadMore = null; // 最近一次点击 Load More 时的 nextObjectsMarker
    function toLoadMore() {
      if (lastObjectsMarkerForLoadMore !== $scope.nextObjectsMarker) {
        $timeout(() => {
          tryLoadMore($scope.nextObjectsMarkerInfo, $scope.nextObjectsMarker, {
            starting: () => {
              $scope.isLoading = true;
            },
            completed: () => {
              $scope.isLoading = false;
              lastObjectsMarkerForLoadMore = null;
            }
          });
          lastObjectsMarkerForLoadMore = $scope.nextObjectsMarker;
        }, 100);
      }
    }

    function tryLoadMore(info, nextObjectsMarker, callback) {
      callback = callback || {};

      if (info.bucketName !== $scope.currentInfo.bucketName ||
          info.key !== $scope.currentInfo.key + $scope.sch.objectName ||
          $scope.nextObjectsMarker !== nextObjectsMarker) {
        return;
      }
      console.log(`loading next kodo://${info.bucketName}/${info.key}?marker=${nextObjectsMarker}`);

      if (callback.starting) callback.starting();

      tryListFiles(info, nextObjectsMarker, (err, files) => {
        if (err) {
          Toast.error(err.message);
          return;
        }

        showFilesTable(files, true);
        if (callback.completed) callback.completed();
      });
    }

    function listExternalPaths(fn) {
      clearFilesList();

      $timeout(() => {
        $scope.isLoading = true;
      });

      ExternalPath.list().then((externalPaths) => {
        if ($scope.sch.externalPathName) {
          externalPaths = externalPaths.filter((ep) => ep.shortPath.indexOf($scope.sch.externalPathName) >= 0);
        }

        $timeout(() => {
          $scope.externalPaths = externalPaths;
          $scope.isLoading = false;
          showExternalPathsTable(externalPaths);
          if (fn) fn(null);
        });
      }).catch((err) => {
        console.error("list external paths error", err);
        $timeout(() => {
          $scope.isLoading = false;
          if (fn) fn(err);
        });
      });
    }

    function isFrozenOrNot(region, bucket, key, callbacks) {
      QiniuClient.getFrozenInfo(region, bucket, key, getQiniuClientOpt()).then((data) => {
        const callback = callbacks[data.status.toLowerCase()];
        if (callback) {
          callback();
        }
      }).catch((err) => {
        const callback = callbacks.error;
        if (callback) {
          callback(err);
        }
      });
    }

    /////////////////////////////////
    function showAddBucket() {
      $modal.open({
        templateUrl: "main/files/modals/add-bucket-modal.html",
        controller: "addBucketModalCtrl",
        resolve: {
          item: () => {
            return null;
          },
          callback: () => {
            return () => {
              Toast.success(T("bucket.add.success"));
              $timeout(listBuckets, 500);
            };
          },
          qiniuClientOpt: () => {
            return getQiniuClientOpt();
          },
          regions: () => {
            return QiniuClient.getRegions(getQiniuClientOpt());
          }
        }
      }).result.then(angular.noop, angular.noop);
    }

    function showAddExternalPath() {
      $modal.open({
        templateUrl: "main/files/modals/add-external-path-modal.html",
        controller: "addExternalPathModalCtrl",
        resolve: {
          item: () => {
            return null;
          },
          callback: () => {
            return () => {
              Toast.success(T("externalPath.add.success"));
              $timeout(listExternalPaths, 500);
            };
          },
          regions: () => {
            return QiniuClient.getRegions(getQiniuClientOpt());
          }
        }
      }).result.then(angular.noop, angular.noop);
    }

    function showDeleteBucket(item) {
      if (item.grantedPermission) {
        Toast.error(T('permission.denied'));
        return;
      }

      const title = T("bucket.delete.title"),
          message = T("bucket.delete.message", {
            name: item.name,
            region: item.regionId,
          });

      Dialog.confirm(title, message,
        (btn) => {
          if (btn) {
            QiniuClient.deleteBucket(item.regionId, item.name, getQiniuClientOpt()).then(() => {
              AuditLog.log('deleteBucket', {
                regionId: item.regionId,
                name: item.name,
              });
              Toast.success(T("bucket.delete.success")); //删除Bucket成功
              $timeout(listBuckets, 1000);
            });
          }
        },
        1
      );
    }

    function showDeleteExternalPath(item) {
      const title = T("externalPath.delete.title"),
          message = T("externalPath.delete.message", {
            region: item.regionId,
            path: item.fullPath
          });

      Dialog.confirm(
        title,
        message,
        (btn) => {
          if (btn) {
            ExternalPath.remove(item.fullPath, item.regionId).then(() => {
              Toast.success(T("externalPath.delete.success")); //删除外部路径成功

              AuditLog.log('deleteExternalPath', {
                regionId: item.regionId,
                fullPath: item.fullPath
              });
              $timeout(listExternalPaths, 1000);
            });
          }
        },
        1
      );
    }

    function showAddFolder() {
      if ($scope.currentInfo.bucketGrantedPermission === 'readonly') {
        Toast.error(T('permission.denied'));
        return;
      }

      $modal.open({
        templateUrl: "main/files/modals/add-folder-modal.html",
        controller: "addFolderModalCtrl",
        resolve: {
          currentInfo: () => {
            return angular.copy($scope.currentInfo);
          },
          qiniuClientOpt: () => {
            return getQiniuClientOpt();
          },
          callback: () => {
            return () => {
              Toast.success(T("folder.create.success"));

              $timeout(listFiles, 300);
            };
          }
        }
      }).result.then(angular.noop, angular.noop);
    }

    function showPreview(item, type) {
      if (!$scope.domains || $scope.domains.length === 0) {
        Toast.error(T('permission.denied'));
        return;
      }

      var fileType = fileSvs.getFileType(item);
      fileType.type = type || fileType.type;

      var templateUrl = "main/files/modals/preview/others-modal.html",
        controller = "othersModalCtrl",
        backdrop = true;

      if (fileType.type === "code") {
        templateUrl = "main/files/modals/preview/code-modal.html";
        controller = "codeModalCtrl";
        backdrop = "static";
      } else if (fileType.type === "picture") {
        templateUrl = "main/files/modals/preview/picture-modal.html";
        controller = "pictureModalCtrl";
      } else if (fileType.type === "video" || fileType.type === "audio") {
        templateUrl = "main/files/modals/preview/media-modal.html";
        controller = "mediaModalCtrl";
      }
      // else if(fileType.type=='doc'){
      //   templateUrl= 'main/files/modals/preview/doc-modal.html';
      //   controller= 'docModalCtrl';
      // }

      $modal.open({
        templateUrl: templateUrl,
        controller: controller,
        size: "lg",
        //backdrop: backdrop,
        resolve: {
          bucketInfo: () => {
            return angular.copy($scope.currentInfo);
          },
          objectInfo: () => {
            return item;
          },
          qiniuClientOpt: () => {
            return getQiniuClientOpt();
          },
          fileType: () => {
            return fileType;
          },
          selectedDomain: () => {
            return $scope.selectedDomain;
          },
          reload: () => {
            return () => {
              $timeout(listFiles, 300);
            };
          },
          showFn: () => {
            return {
              preview: showPreview,
              download: () => {
                showDownload(item);
              },
              move: (isCopy) => {
                showMove([item], isCopy);
              },
              remove: () => {
                showDeleteFiles([item]);
              },
              rename: () => {
                showRename(item);
              },
              downloadLink: () => {
                showDownloadLink(item);
              },
              updateStorageClass: () => {
                showUpdateStorageClass(item);
              }
            };
          }
        }
      }).result.then(angular.noop, angular.noop);
    }

    function showRename(item) {
      if ($scope.currentInfo.bucketGrantedPermission === 'readonly') {
        Toast.error(T('permission.denied'));
        return;
      }

      $modal.open({
        templateUrl: "main/files/modals/rename-modal.html",
        controller: "renameModalCtrl",
        backdrop: "static",
        resolve: {
          item: () => {
            return angular.copy(item);
          },
          moveTo: () => {
            return angular.copy($scope.currentInfo);
          },
          currentInfo: () => {
            return angular.copy($scope.currentInfo);
          },
          qiniuClientOpt: () => {
            return getQiniuClientOpt();
          },
          isCopy: () => {
            return false;
          },
          callback: () => {
            return () => {
              $timeout(listFiles, 100);
            };
          }
        }
      }).result.then(angular.noop, angular.noop);
    }

    function disablePaste() {
      return $scope.currentInfo.bucketGrantedPermission === 'readonly' ||
             $scope.keepMoveOptions.currentInfo.regionId !== $scope.currentInfo.regionId ||
             $scope.keepMoveOptions.currentInfo.qiniuBackendMode !== $scope.currentInfo.qiniuBackendMode;
    }

    function showPaste() {
      var keyword = $scope.keepMoveOptions.isCopy ? T('copy') : T('move');

      if ($scope.keepMoveOptions.items.length == 1 &&
          deepEqual($scope.currentInfo, $scope.keepMoveOptions.currentInfo)) {
        $modal.open({
          templateUrl: 'main/files/modals/rename-modal.html',
          controller: 'renameModalCtrl',
          backdrop: 'static',
          resolve: {
            item: () => {
              return angular.copy($scope.keepMoveOptions.items[0]);
            },
            moveTo: () => {
              return angular.copy($scope.currentInfo);
            },
            currentInfo: () => {
              return angular.copy($scope.keepMoveOptions.currentInfo);
            },
            isCopy: () => {
              return $scope.keepMoveOptions.isCopy;
            },
            qiniuClientOpt: () => {
              return getQiniuClientOpt();
            },
            callback: () => {
              return () => {
                $scope.keepMoveOptions = null;
                $timeout(listFiles, 100);
              };
            }
          }
        }).result.then(angular.noop, angular.noop);

        return;
      }

      var msg = T("paste.message1", {
        name: $scope.keepMoveOptions.items[0].name,
        action: keyword
      });

      Dialog.confirm(keyword, msg, (isMove) => {
        if (isMove) {
          $modal.open({
            templateUrl: "main/files/modals/move-modal.html",
            controller: "moveModalCtrl",
            backdrop: "static",
            resolve: {
              items: () => {
                return angular.copy($scope.keepMoveOptions.items);
              },
              moveTo: () => {
                return angular.copy($scope.currentInfo);
              },
              isCopy: () => {
                return $scope.keepMoveOptions.isCopy;
              },
              renamePath: () => {
                return "";
              },
              fromInfo: () => {
                return angular.copy($scope.keepMoveOptions.currentInfo);
              },
              qiniuClientOpt: () => {
                return getQiniuClientOpt();
              },
              callback: () => {
                return () => {
                  $scope.keepMoveOptions = null;
                  $timeout(listFiles, 100);
                };
              }
            }
          }).result.then(angular.noop, angular.noop);
        }
      });
    }

    function cancelPaste() {
      $timeout(() => {
        $scope.keepMoveOptions = null;
      });
    }

    function showMove(items, isCopy) {
      if ($scope.currentInfo.bucketGrantedPermission === 'readonly') {
        Toast.error(T('permission.denied'));
        return;
      }

      $scope.keepMoveOptions = {
        items: items,
        isCopy: isCopy,
        currentInfo: angular.copy($scope.currentInfo),
        originPath: getCurrentPath()
      };
    }

    function showDownloadLink(item) {
      if (!$scope.domains || $scope.domains.length === 0) {
        Toast.error(T('permission.denied'));
        return;
      }

      $modal.open({
        templateUrl: "main/files/modals/show-download-link-modal.html",
        controller: "showDownloadLinkModalCtrl",
        resolve: {
          item: () => {
            return angular.copy(item);
          },
          current: () => {
            return {
              info: angular.copy($scope.currentInfo),
              domain: angular.copy($scope.selectedDomain.domain),
            };
          },
          qiniuClientOpt: () => {
            return getQiniuClientOpt();
          },
          domains: () => {
            return angular.copy($scope.domains);
          },
          showDomains: () => {
            return showDomains();
          }
        }
      }).result.then(angular.noop, angular.noop);
    }

    function showDownloadLinkOfFilesSelected() {
      showDownloadLinks($scope.sel.has);
    }

    function showDownloadLinks(items) {
      if (!$scope.domains || $scope.domains.length === 0) {
        Toast.error(T('permission.denied'));
        return;
      }

      $modal.open({
        templateUrl: "main/files/modals/show-download-links-modal.html",
        controller: "showDownloadLinksModalCtrl",
        resolve: {
          items: () => {
            return angular.copy(items);
          },
          current: () => {
            return {
              info: angular.copy($scope.currentInfo),
              domain: angular.copy($scope.selectedDomain.domain),
            };
          },
          qiniuClientOpt: () => {
            return getQiniuClientOpt();
          },
          domains: () => {
            return angular.copy($scope.domains);
          },
          showDomains: () => {
            return showDomains();
          }
        }
      }).result.then(angular.noop, angular.noop);
    }

    function showUpdateStorageClass(item) {
      $modal.open({
        templateUrl: "main/files/modals/update-storage-class-modal.html",
        controller: "updateStorageClassModalCtrl",
        resolve: {
          item: () => {
            return angular.copy(item);
          },
          currentInfo: () => {
            return angular.copy($scope.currentInfo);
          },
          qiniuClientOpt: () => {
            return getQiniuClientOpt();
          },
          callback: () => {
            return () => {
              $timeout(listFiles, 100);
            };
          }
        }
      }).result.then(angular.noop, angular.noop);
    }

    function showRestore(item) {
      $modal.open({
        templateUrl: "main/files/modals/restore-modal.html",
        controller: "restoreModalCtrl",
        resolve: {
          item: () => {
            return angular.copy(item);
          },
          currentInfo: () => {
            return angular.copy($scope.currentInfo);
          },
          qiniuClientOpt: () => {
            return getQiniuClientOpt();
          },
        }
      }).result.then(angular.noop, angular.noop);
    }

    function showDownload(item) {
      if (!$scope.domains || $scope.domains.length === 0) {
        Toast.error(T('permission.denied'));
        return;
      }

      const bucketInfo = angular.copy($scope.currentInfo),
            fromInfo = angular.copy(item),
            domain = angular.copy($scope.selectedDomain.domain);

      fromInfo.region = bucketInfo.regionId;
      fromInfo.bucket = bucketInfo.bucketName;
      fromInfo.domain = domain;
      fromInfo.qiniuBackendMode = bucketInfo.qiniuBackendMode;
      Dialog.showDownloadDialog((folderPaths) => {
        if (!folderPaths || folderPaths.length == 0) {
          return;
        }
        if (!folderPaths[0].endsWith(path.sep)) {
          folderPaths[0] += path.sep;
        }
        const to = qiniuPath.fromLocalPath(folderPaths[0]);
        $scope.handlers.downloadFilesHandler([fromInfo], to);
      });
    }

    function showDeleteFilesSelected() {
      showDeleteFiles($scope.sel.has);
    }

    function showDeleteFiles(items) {
      if ($scope.currentInfo.bucketGrantedPermission === 'readonly') {
        Toast.error(T('permission.denied'));
        return;
      }

      $modal.open({
        templateUrl: "main/files/modals/delete-files-modal.html",
        controller: "deleteFilesModalCtrl",
        backdrop: "static",
        resolve: {
          items: () => {
            return items;
          },
          currentInfo: () => {
            return angular.copy($scope.currentInfo);
          },
          qiniuClientOpt: () => {
            return getQiniuClientOpt();
          },
          callback: () => {
            return () => {
              $timeout(listFiles, 300);
            };
          }
        }
      }).result.then(angular.noop, angular.noop);
    }

    function showRestoreFilesSelected() {
      showRestoreFiles($scope.sel.has);
    }

    function showRestoreFiles(items) {
      if ($scope.currentInfo.bucketGrantedPermission === 'readonly') {
        Toast.error(T('permission.denied'));
        return;
      }

      $modal.open({
        templateUrl: "main/files/modals/restore-files-modal.html",
        controller: "restoreFilesModalCtrl",
        backdrop: "static",
        resolve: {
          items: () => {
            return items;
          },
          currentInfo: () => {
            return angular.copy($scope.currentInfo);
          },
          qiniuClientOpt: () => {
            return getQiniuClientOpt();
          },
          callback: () => {
            return () => {
              $timeout(listFiles, 300);
            };
          }
        }
      }).result.then(angular.noop, angular.noop);
    }

    function showSetStorageClassOfFilesSelected() {
      showSetStorageClassOfFiles($scope.sel.has);
    }

    function showSetStorageClassOfFiles(items) {
      if ($scope.currentInfo.bucketGrantedPermission === 'readonly') {
        Toast.error(T('permission.denied'));
        return;
      }

      $modal.open({
        templateUrl: "main/files/modals/update-storage-classes-modal.html",
        controller: "updateStorageClassesModalCtrl",
        backdrop: "static",
        resolve: {
          items: () => {
            return items;
          },
          currentInfo: () => {
            return angular.copy($scope.currentInfo);
          },
          qiniuClientOpt: () => {
            return getQiniuClientOpt();
          },
          callback: () => {
            return () => {
              $timeout(listFiles, 300);
            };
          }
        }
      }).result.then(angular.noop, angular.noop);
    }

    ////////////////////////
    function selectBucket(item) {
      if ($scope.bucket_sel === item) {
        $timeout(() => {
          $scope.bucket_sel = null;
        });
      } else {
        $timeout(() => {
          $scope.bucket_sel = item;
        });
      }
    }

    function selectFile(item) {
      $timeout(() => {
        var idx = $scope.objects.indexOf(item);
        if (idx > -1) {
          $scope.sel.x[`i_${idx}`] = !$scope.sel.x[`i_${idx}`];

          var subidx = $scope.sel.has.indexOf(item);
          if (subidx > -1) {
            if (!$scope.sel.x[`i_${idx}`]) {
              $scope.sel.has.splice(subidx, 1);
            }
          } else {
            if ($scope.sel.x[`i_${idx}`]) {
              $scope.sel.has.push(item);
            }
          }

          $scope.sel.all = $scope.sel.has.length == $scope.objects.length;
        }
      });
    }

    function selectFileAndStopEventPropagation(item, event) {
      selectFile(item);
      event.stopPropagation();
    }

    function openFileAndStopEventPropagation(item, event) {
      if (item.itemType === 'folder') {
        gotoAddress($scope.currentBucketName, item.path)
      } else {
        if ($scope.domains && $scope.domains.length > 0) {
          showPreview(item);
        }
        event.stopPropagation();
      }
    }

    function selectExternalPath(item) {
      $timeout(() => {
        if ($scope.external_path_sel == item) {
          $scope.external_path_sel = null;
        } else {
          $scope.external_path_sel = item;
        }
      });
    }

    function initBucketSelect() {
      $timeout(() => {
        $scope.bucket_sel = null;
      });
    }

    function initFilesSelect() {
      $timeout(() => {
        $scope.sel.all = false;
        $scope.sel.has = [];
        $scope.sel.x = {};
      });
    }

    function initExternalPathSelect() {
      $timeout(() => {
        $scope.external_path_sel = null;
      });
    }

    function clearFilesList() {
      initFilesSelect();

      $timeout(() => {
        $scope.objects = [];
        $scope.nextObjectsMarker = null;
      });
    }

    ////////////////////////////////
    var uploadDialog, downloadDialog;

    function showUploadDialog() {
      if (uploadDialog) return;

      if ($scope.currentInfo.bucketGrantedPermission === 'readonly') {
        Toast.error(T('permission.denied'));
        return;
      }

      uploadDialog = true;
      $timeout(() => {
        uploadDialog = false;
      }, 600);

      Dialog.showUploadDialog((filePaths) => {
        if (!filePaths || filePaths.length == 0) {
          return;
        }

        $scope.handlers.uploadFilesHandler(filePaths.map(qiniuPath.fromLocalPath), angular.copy($scope.currentInfo));
      });
    }

    function showDownloadDialog() {
      if (downloadDialog) return;

      downloadDialog = true;
      $timeout(() => {
        downloadDialog = false;
      }, 600);

      Dialog.showDownloadDialog((folderPaths) => {
        if (!folderPaths || folderPaths.length == 0 || $scope.sel.has.length == 0) {
          return;
        }

        tryDownloadFiles(folderPaths[0]);
      });
    }

    function tryDownloadFiles(to) {
      to = to.replace(/(\/*$)/g, "");
      if (!to.endsWith(path.sep)) {
        to += path.sep;
      }
      const localPath = qiniuPath.fromLocalPath(to);

      const selectedFiles = angular.copy($scope.sel.has);
      angular.forEach(selectedFiles, (n) => {
        n.region = $scope.currentInfo.regionId;
        n.bucket = $scope.currentInfo.bucketName;
        n.domain = angular.copy($scope.selectedDomain.domain);
        n.qiniuBackendMode = $scope.currentInfo.qiniuBackendMode;
      });
      /**
       * @param fromS3Path {array}  item={region, bucket, path, name, size }
       * @param toLocalPath {string}
       */
      $scope.handlers.downloadFilesHandler(selectedFiles, localPath);
    }

    /**
     * 监听 drop
     * @param e
     * @returns {boolean}
     */
    function handlerDrop(e) {
      e.preventDefault();
      e.stopPropagation();

      const files = e.originalEvent.dataTransfer.files;
      const filePaths = [];
      if (files) {
        angular.forEach(files, (n) => {
          filePaths.push(qiniuPath.fromLocalPath(n.path));
        });
      }

      if (filePaths.length) {
        $scope.handlers.uploadFilesHandler(filePaths, angular.copy($scope.currentInfo));
      }

      return false;
    }

    function showBucketsTable(buckets) {
      initBucketSelect();

      QiniuClient.getRegions(getQiniuClientOpt()).then((regions) => {
        var $list = $('#bucket-list').bootstrapTable({
          columns: [{
            field: '_',
            title: '-',
            radio: true
          }, {
            field: 'name',
            title: T('bucket.name'),
            formatter: (val, row, idx, field) => {
              if (row.grantedPermission === 'readonly') {
                return `<i class="text-warning"><img src="icons/buckets/zhidu.png" style="display: inline-block; height: 15px;" /></i>&nbsp;<a href=""><span class="bucket-name bucket-readonly">${val}</span></a>`;
              } else if (row.grantedPermission === 'readwrite') {
                return `<i class="text-warning"><img src="icons/buckets/duxie.png" style="display: inline-block; height: 15px;" /></i>&nbsp;<a href=""><span class="bucket-name bucket-readwrite">${val}</span></a>`;
              } else {
                return `<i class="fa fa-database text-warning"></i>&nbsp;<a href=""><span class="bucket-name">${val}</span></a>`;
              }
            },
            events: {
              'click a': (evt, val, row, idx) => {
                gotoAddress(val);

                return false;
              }
            }
          }, {
            field: 'regionId',
            title: T('bucket.region'),
            formatter: (id) => {
              if (!id) {
                return T('region.get.error');
              }
              let regionLabel = undefined;
              const region = regions.find((region) => region.s3Id === id && region.translatedLabel);
              if (region) {
                regionLabel = region.translatedLabel;
              }
              return regionLabel || T('region.unknown');
            }
          }, {
            field: 'createDate',
            title: T('creationTime'),
            formatter: (val) => {
              return $filter('timeFormat')(val);
            }
          }],
          clickToSelect: true,
          onCheck: (row, $row) => {
            if (row === $scope.bucket_sel) {
              $row.parents('tr').removeClass('info');

              $list.bootstrapTable('uncheckBy', {
                field: 'name',
                values: [row.name],
              });

              $timeout(() => {
                $scope.bucket_sel = null;
              });
            } else {
              $list.find('tr').removeClass('info');
              $row.parents('tr').addClass('info');
              $timeout(() => {
                $scope.bucket_sel = row;
              });
            }

            return false;
          }
        });

        $list.bootstrapTable('load', buckets).bootstrapTable('uncheckAll');
        angular.forEach($('#bucket-list tbody tr .bucket-name.bucket-readonly'), (row) => {
          $(row).tooltip('destroy');
          $(row).tooltip({ delay: 0, title: T('privilege.readonly'), trigger: 'hover' });
        });
        angular.forEach($('#bucket-list tbody tr .bucket-name.bucket-readwrite'), (row) => {
          $(row).tooltip('destroy');
          $(row).tooltip({ delay: 0, title: T('privilege.readwrite'), trigger: 'hover' });
        });
      }).catch((err) => {
        console.error(err);
        Toast.error(err);
      })
    }

    function showExternalPathsTable(externalPaths) {
      initExternalPathSelect();

      QiniuClient.getRegions(getQiniuClientOpt()).then((regions) => {
        var $list = $('#external-path-list').bootstrapTable({
          columns: [{
            field: '_',
            title: '-',
            radio: true
          }, {
            field: 'fullPath',
            title: T('externalPath.path'),
            formatter: (val, row, idx, field) => {
              return `<i class="fa fa-map-signs text-warning"></i> <a href=""><span>${val}</span></a>`;
            },
            events: {
              'click a': (evt, val, row, idx) => {
                gotoAddress(val);

                return false;
              }
            }
          }, {
            field: 'regionId',
            title: T('region'),
            formatter: (id) => {
              if (id === null) {
                return T('region.get.error');
              }
              let regionLabel = undefined;
              const region = regions.find((region) => region.s3Id === id && region.translatedLabel);
              if (region) {
                regionLabel = region.translatedLabel;
              }
              return regionLabel || T('region.unknown');
            }
          }],
          clickToSelect: true,
          onCheck: (row, $row) => {
            if (row === $scope.external_path_sel) {
              $row.parents('tr').removeClass('info');

              $list.bootstrapTable('uncheckBy', {
                field: 'fullPath',
                values: [row.fullPath],
              });

              $timeout(() => {
                $scope.external_path_sel = null;
              });
            } else {
              $list.find('tr').removeClass('info');
              $row.parents('tr').addClass('info');

              $timeout(() => {
                $scope.external_path_sel = row;
              });
            }

            return false;
          }
        });

        $list.bootstrapTable('load', externalPaths).bootstrapTable('uncheckAll');
      }).catch((err) => {
        console.error(err);
        Toast.error(err);
      });
    }

    function showFilesTable(files, isAppend) {
      if (!isAppend) {
        initFilesSelect();
      }
      var $list = $('#file-list').bootstrapTable({
        columns: [{
          field: '_',
          title: '-',
          checkbox: true
        }, {
          field: 'name',
          title: T('name'),
          formatter: (val, row, idx, field) => {
            let htmlAttributes = '';
            if (row.storageClass) {
              const currentInfo = $scope.currentInfo;
              htmlAttributes = `data-storage-class="${row.storageClass.toLowerCase()}" data-key="${Base64.encode(row.path.toString())}" data-region="${currentInfo.regionId}" data-bucket="${currentInfo.bucketName}"`;
            }
            return `
              <div class="text-overflow file-item-name" style="cursor:pointer; ${row.itemType === 'folder' ? 'color:orange' : ''}" ${htmlAttributes}>
                <i class="fa fa-${$filter('fileIcon')(row)}"></i>
                <a href="" style="width: 700px; display: inline-block;"><span>${$filter('htmlEscape')(val)}</span></a>
              </div>
            `;
          },
          events: {
            'click a,i': (evt, val, row, idx) => {
              if (row.itemType === 'folder') {
                $timeout(() => {
                  $scope.total_folders = 0;
                });

                gotoAddress($scope.currentBucketName, row.path);
              }

              return false;
            },
            'dblclick a,i': (evt, val, row, idx) => {
              if (row.itemType === 'folder') {
                $timeout(() => {
                  $scope.total_folders = 0;
                });

                gotoAddress($scope.currentBucketName, row.path);
              } else if ($scope.domains && $scope.domains.length > 0) {
                showPreview(row);
              }

              return false;
            }
          }
        }, {
          field: 'size',
          title: `${T('type')} / ${T('size')}`,
          formatter: (val, row, idx, field) => {
            if (row.itemType === 'folder') {
              return `<span class="text-muted">${T('folder')}</span>`;
            }

            return $filter('sizeFormat')(val);
          }
        }, {
          field: 'storageClass',
          title: T('storageClassesType'),
          formatter: (val, row, idx, field) => {
            if (row.itemType === 'folder') {
              return `<span class="text-muted">${T('folder')}</span>`;
            } else if (row.storageClass) {
              return T(`storageClassesType.${row.storageClass.toLowerCase()}`);
            } else {
              return '-';
            }
          }
        }, {
          field: 'lastModified',
          title: T('lastModifyTime'),
          formatter: (val, row, idx, field) => {
            if (row.itemType === 'folder') {
              return '-';
            }

            return $filter('timeFormat')(val);
          }
        }, {
          field: 'actions',
          title: T('actions'),
          formatter: (val, row, idx, field) => {
            var acts = ['<div class="btn-group btn-group-xs">'];
            if (row.itemType !== 'folder' && row.storageClass && row.storageClass.toLowerCase() === 'glacier' && $scope.currentInfo.bucketGrantedPermission !== 'readonly') {
              acts.push(`<button type="button" class="btn unfreeze text-warning" data-toggle="tooltip" data-toggle-i18n="restore"><span class="fa fa-fire"></span></button>`);
            }
            if ($scope.domains && $scope.domains.length > 0) {
              if (row.itemType !== 'folder' || row.path.directoryBasename() && row.path.directoryBasename().length > 0) {
                acts.push(`<button type="button" class="btn download" data-toggle="tooltip" data-toggle-i18n="download"><span class="fa fa-download"></span></button>`);
              }
              if (row.itemType !== 'folder') {
                acts.push(`<button type="button" class="btn download-link" data-toggle="tooltip" data-toggle-i18n="getDownloadLink"><span class="fa fa-link"></span></button>`);
              }
            }
            if (row.itemType !== 'folder') {
              acts.push(`<button type="button" class="btn updateStorageClass text-warning" data-toggle="tooltip" data-toggle-i18n="updateStorageClass"><span class="iconfont icon-a-726-8-02"></span></button>`);
            }
            if ($scope.currentInfo.bucketGrantedPermission !== 'readonly') {
              acts.push(`<button type="button" class="btn remove text-danger" data-toggle="tooltip" data-toggle-i18n="delete"><span class="fa fa-trash"></span></button>`);
            }
            acts.push('</div>');
            return acts.join("");
          },
          events: {
            'click button.download': (evt, val, row, idx) => {
              showDownload(row);

              return false;
            },
            'click button.download-link': (evt, val, row, idx) => {
              showDownloadLink(row);

              return false;
            },
            'click button.remove': (evt, val, row, idx) => {
              showDeleteFiles([row]);

              $timeout(() => {
                $scope.total_folders = $list.find('i.fa-folder').length;
              });

              return false;
            },
            'click button.updateStorageClass': (evt, val, row, idx) => {
              const region = $scope.currentInfo.regionId,
                    bucket = $scope.currentInfo.bucketName,
                    key = row.path.toString();
              isFrozenOrNot(region, bucket, key, {
                'normal': () => {
                  showUpdateStorageClass(row);
                },
                'unfrozen': () => {
                  showUpdateStorageClass(row);
                },
                'unfreezing': () => {
                  Dialog.alert(T('updateStorageClass.title'), T('updateStorageClassModal.message.unfreezing'));
                },
                'frozen': () => {
                  Dialog.alert(T('updateStorageClass.title'), T('updateStorageClassModal.message.frozen'));
                },
                'error': (err) => {
                  Dialog.alert(T('updateStorageClass.title'), T('updateStorageClassModal.message.head_error'));
                },
              });
              return false;
            },
            'click button.unfreeze': (evt, val, row, idx) => {
              const region = $scope.currentInfo.regionId,
                    bucket = $scope.currentInfo.bucketName,
                    key = row.path.toString();
              isFrozenOrNot(region, bucket, key, {
                'frozen': () => {
                  showRestore(row);
                },
                'unfreezing': () => {
                  Dialog.alert(T('restore.title'), T('restore.message.unfreezing'));
                },
                'unfrozen': () => {
                  Dialog.alert(T('restore.title'), T('restore.message.unfrozen'));
                },
                'error': (err) => {
                  Dialog.alert(T('restore.title'), T('restore.message.head_error'));
                },
              });
              return false;
            }
          }
        }],
        rowStyle: (row, idx) => {
          const ONE_HOUR = 60 * 60 * 1000;
          if (row.lastModified && ((new Date()) - row.lastModified) <= 4*ONE_HOUR) {
            return {
              classes: 'warning'
            };
          }

          return {};
        },
        onCheck: (row, $row) => {
          $row.parents('tr').addClass('info');

          $timeout(() => {
            $scope.sel.x[`i_${$row.data('index')}`] = true;

            var idx = $scope.sel.has.indexOf(row);
            if (idx < 0) {
              $scope.sel.has.push(row);
            }

            $scope.sel.all = $scope.sel.has.length == $scope.objects.length;
          });
        },
        onUncheck: (row, $row) => {
          $row.parents('tr').removeClass('info');

          $timeout(() => {
            $scope.sel.x[`i_${$row.data('index')}`] = false;

            var idx = $scope.sel.has.indexOf(row);
            if (idx > -1) {
              $scope.sel.has.splice(idx, 1);
            }

            $scope.sel.all = $scope.sel.has.length == $scope.objects.length;
          });
        },
        onCheckAll: (rows) => {
          $list.find('tr').removeClass('info').addClass('info');

          initFilesSelect();

          $timeout(() => {
            $scope.sel.all = true;
            $scope.sel.has = rows;

            angular.forEach(rows, (row, idx) => {
              $scope.sel.x[`i_${idx}`] = true;
            });
          });
        },
        onUncheckAll: (rows) => {
          $list.find('tr').removeClass('info');

          initFilesSelect();
        }
      });

      if (isAppend) {
        $list.bootstrapTable('append', files);
      } else {
        $list.bootstrapTable('load', files).bootstrapTable('uncheckAll');
      }
      angular.forEach($('#file-list tbody tr [data-storage-class="glacier"]'), (row) => {
        let isMouseOver = true;
        const region = $(row).attr('data-region'),
              bucket = $(row).attr('data-bucket'),
              key = Base64.decode($(row).attr('data-key')),
              span = $(row).find('a span'),
              addTooltip = (status) => {
                span.tooltip('destroy');
                $timeout(() => {
                  if (isMouseOver) {
                    span.attr('data-toggle', 'tooltip');
                    span.attr('data-placement', 'right');
                    span.tooltip({ delay: 0, title: T(`restore.tooltip.${status}`), trigger: 'hover' });
                    span.tooltip('show');
                  }
                }, 100);
              };
        $(span).off('mouseover').on('mouseover', () => {
          isMouseOver = true;
          isFrozenOrNot(region, bucket, key, {
            frozen: () => { addTooltip('frozen'); },
            unfreezing: () => { addTooltip('unfreezing'); },
            unfrozen: () => { addTooltip('unfrozen'); },
          });
        }).off('mouseleave').on('mouseleave', () => {
          isMouseOver = false;
        });
      });
      angular.forEach($('#file-list tbody tr .btn-group button[type="button"][data-toggle="tooltip"][data-toggle-i18n]'), (button) => {
        let mouseOverEventSetup = false;
        $(button).tooltip('destroy');
        $(button).off('mouseover').on('mouseover', () => {
          if (!mouseOverEventSetup) {
            mouseOverEventSetup = true;
            $(button).attr('data-placement', 'top');
            const i18nKey = $(button).attr('data-toggle-i18n');
            $(button).tooltip({ container: 'body', title: T(i18nKey), trigger: 'hover' });
            $(button).tooltip('show');
          }
        });
      });

      $timeout(() => {
        $scope.total_folders = $list.find('i.fa-folder').length;
      });
    }

    function getQiniuClientOpt() {
      if ($scope.ref.mode.startsWith('external')) {
        return { preferS3Adapter: true };
      } else {
        return {};
      }
    }
  }
]);

"use strict";

angular.module("web").controller("aboutCtrl", [
  "$scope",
  "$state",
  "$uibModalInstance",
  "$interval",
  "autoUpgradeSvs",
  "safeApply",
  "pscope",
  function (
    $scope,
    $state,
    $modalInstance,
    $interval,
    autoUpgradeSvs,
    safeApply,
    pscope
  ) {
    angular.extend($scope, {
      app_logo: Global.app.logo,
      app_version: Global.app.version,
      custom_about_html: Global.about_html,

      open: open,
      tryToOpenUpgradePackage: tryToOpenUpgradePackage,
      startUpgrade: startUpgrade,
      pauseUpgrade: pauseUpgrade,
      cancel: cancel
    });

    $interval(function () {
      Object.assign($scope.info, pscope.upgradeInfo);
    }, 1000);

    init();

    const fs = require('fs');

    function init() {
      $scope.info = pscope.upgradeInfo;

      if (!$scope.info.isLastVersion) {
        var converter = new showdown.Converter();
        autoUpgradeSvs.getLastestReleaseNote($scope.info.lastVersion, function (text) {
          text = text + "";

          var html = converter.makeHtml(text);

          $scope.info.lastReleaseNote = html;
        });
      }
    }

    function open(a) {
      openExternal(a);
    }

    function tryToOpenUpgradePackage(localPath) {
      if (fs.existsSync(localPath)) {
        open(`file://${localPath}`);
      } else {
        startUpgrade();
      }
    }

    function startUpgrade() {
      autoUpgradeSvs.start();
    }

    function pauseUpgrade() {
      autoUpgradeSvs.stop();
    }

    function cancel() {
      $modalInstance.dismiss("close");
    }
  }
]);

"use strict";

angular.module("web").controller("bookmarksCtrl", [
  "$scope",
  "$rootScope",
  "$translate",
  "$state",
  "$uibModalInstance",
  "Bookmark",
  "Toast",
  function ($scope, $rootScope, $translate, $state, $modalInstance, Bookmark, Toast) {
    const T = $translate.instant;

    angular.extend($scope, {
      goTo: goTo,
      remove: remove,
      cancel: cancel
    });

    function goTo(bookmark) {
      $rootScope.$broadcast("gotoKodoAddress", bookmark.fullPath);
      cancel();
    }

    function refresh() {
      $scope.bookmarks = Bookmark.list().map((bookmark) => {
        bookmark.time = new Date(moment.unix(bookmark.timestamp));
        return bookmark;
      });
    }

    function remove(bookmark) {
      Bookmark.remove(bookmark.fullPath, bookmark.mode);
      Toast.warning(T("bookmarks.delete.success"));
      refresh();
    }

    function cancel() {
      $modalInstance.dismiss("close");
    }

    refresh();
  }
]);

"use strict";

angular.module("web").controller("settingsCtrl", [
  "$scope",
  "$state",
  "$timeout",
  "$uibModalInstance",
  "$translate",
  "callback",
  "settingsSvs",
  "Toast",
  "Dialog",
  function (
    $scope,
    $state,
    $timeout,
    $modalInstance,
    $translate,
    callback,
    settingsSvs,
    Toast,
    Dialog
  ) {
    var T = $translate.instant;

    angular.extend($scope, {
      set: {
        isDebug: settingsSvs.isDebug.get(),
        autoUpgrade: settingsSvs.autoUpgrade.get(),
        resumeUpload: settingsSvs.resumeUpload.get(),
        maxUploadConcurrency: settingsSvs.maxUploadConcurrency.get(),
        multipartUploadThreshold: settingsSvs.multipartUploadThreshold.get(),
        multipartUploadSize: settingsSvs.multipartUploadSize.get(),
        uploadSpeedLimitEnabled: settingsSvs.uploadSpeedLimitEnabled.get(),
        uploadSpeedLimitKBperSec: settingsSvs.uploadSpeedLimitKBperSec.get(),
        resumeDownload: settingsSvs.resumeDownload.get(),
        maxDownloadConcurrency: settingsSvs.maxDownloadConcurrency.get(),
        multipartDownloadThreshold: settingsSvs.multipartDownloadThreshold.get(),
        multipartDownloadSize: settingsSvs.multipartDownloadSize.get(),
        downloadSpeedLimitEnabled: settingsSvs.downloadSpeedLimitEnabled.get(),
        downloadSpeedLimitKBperSec: settingsSvs.downloadSpeedLimitKBperSec.get(),
        externalPathEnabled: settingsSvs.externalPathEnabled.get(),
        stepByStepLoadingFiles: settingsSvs.stepByStepLoadingFiles.get(),
        filesLoadingSize: settingsSvs.filesLoadingSize.get(),
      },
      setChange: setChange,
      cancel: cancel
    });

    var tid;

    function setChange(form1, key, ttl) {
      if (typeof $scope.set[key] === 'undefined') {
        return;
      }

      $timeout.cancel(tid);

      tid = $timeout(function () {
        settingsSvs[key].set($scope.set[key]);
        Toast.success(T("settings.success")); //已经保存设置
      }, ttl || 100);
    }

    function cancel() {
      if (callback) callback();

      $modalInstance.dismiss("close");
    }
  }
]);

angular.module("web").controller("topCtrl", [
  "$scope",
  "$rootScope",
  "$uibModal",
  "$location",
  "$translate",
  "$timeout",
  "Dialog",
  "Auth",
  "AuthInfo",
  "settingsSvs",
  "Toast",
  "Config",
  "autoUpgradeSvs",
  "AuditLog",
  function(
    $scope,
    $rootScope,
    $modal,
    $location,
    $translate,
    $timeout,
    Dialog,
    Auth,
    AuthInfo,
    settingsSvs,
    Toast,
    Config,
    autoUpgradeSvs,
    AuditLog
  ) {
    var fs = require("fs");
    var path = require("path");
    var T = $translate.instant;

    angular.extend($scope, {
      logout: logout,
      switchAccount: switchAccount,
      showBookmarks: showBookmarks,
      showAbout: showAbout,
      showReleaseNote: showReleaseNote,
      showBucketsOrFiles: showBucketsOrFiles,
      showExternalPaths: showExternalPaths,
      isExternalPathEnabled: isExternalPathEnabled,
      click10: click10
    });

    var ctime = 0;
    var tid;
    function click10() {
      ctime++;
      if (ctime > 10) {
        openDevTools();
      }
      $timeout.cancel(tid);
      tid = $timeout(function() {
        ctime = 0;
      }, 600);
    }

    $rootScope.app = {};
    angular.extend($rootScope.app, Global.app);

    $scope.authInfo = AuthInfo.get();
    $scope.authInfo.expirationStr = moment(
      new Date($scope.authInfo.expiration)
    ).format("YYYY-MM-DD HH:mm:ss");

    $scope.$watch("upgradeInfo.isLastVersion", function(v) {
      if (false === v) {
        if (1 == settingsSvs.autoUpgrade.get()) autoUpgradeSvs.start();
        else $scope.showAbout();
      }
    });
    $scope.$watch("upgradeInfo.upgradeJob.status", function(s) {
      if ("failed" == s || "finished" == s) {
        $scope.showAbout();
      }
    });

    $rootScope.showSettings = function(fn) {
      $modal.open({
        templateUrl: "main/modals/settings.html",
        controller: "settingsCtrl",
        resolve: {
          callback: function() {
            return fn;
          }
        }
      }).result.then(angular.noop, angular.noop);
    };

    function logout() {
      var title = T("logout");
      var message = T("logout.message");
      Dialog.confirm(
        title,
        message,
        function(b) {
          if (b) {
            const originalAccessKeyId = AuthInfo.get().id;
            Auth.logout().then(() => {
              AuditLog.log('logout', { from: originalAccessKeyId });
              $location.url("/login");
            }).catch((err) => {
              Toast.error(err.message, 5000);
              Dialog.alert(T('auth.logout.error.title'), T('auth.logout.error.description'), null, 1);
            });
          }
        },
        1
      );
    }

    function switchAccount() {
      $modal.open({
        templateUrl: "main/auth/modals/ak-histories-modal.html",
        controller: "akHistoriesModalCtrl",
        size: 'lg',
        resolve: {
          choose: function() {
            return function(history) {
              const originalAccessKeyId = AuthInfo.get().id;
              Auth.logout().then(
                function () {
                  const isPublicCloud = history.isPublicCloud;
                  Auth.login({
                    id: history.accessKeyId,
                    secret: history.accessKeySecret,
                    isPublicCloud: isPublicCloud
                  }).then(
                    function () {
                      if (isPublicCloud) {
                        AuthInfo.switchToPublicCloud();
                      } else {
                        AuthInfo.switchToPrivateCloud();
                      }
                      AuditLog.log('switchAccount', { from: originalAccessKeyId });
                      Toast.success(T("login.successfully"), 1000);
                      const { ipcRenderer } = require('electron');
                      ipcRenderer.send('asynchronous', { key: 'reloadWindow' });
                    },
                    function (err) {
                      Toast.error(err.message, 5000);
                      Dialog.alert(T('auth.switch.error.title'), T('auth.switch.error.description'), null, 1);
                    });
                },
                function (err) {
                  Toast.error(err.message, 5000);
                  Dialog.alert(T('auth.logout.error.title'), T('auth.logout.error.description'), null, 1);
                });
            }
          }
        }
      }).result.then(angular.noop, angular.noop);
    }

    function showReleaseNote() {
      var converter = new showdown.Converter();
      fs.readFile(
        path.join(__dirname, "release-notes", Global.app.version + ".md"),
        function(err, text) {
          if (err) {
            console.error(err);
            return;
          }
          text = text + "";
          var html = converter.makeHtml(text);
          var message = T("main.upgration"); //'主要更新'
          Dialog.alert(message, html, function() {}, { size: "lg" });
        }
      );
    }

    function showBookmarks() {
      $modal.open({
        templateUrl: "main/modals/bookmarks.html",
        controller: "bookmarksCtrl",
        size: "lg"
      }).result.then(angular.noop, angular.noop);
    }

    function showAbout() {
      $modal.open({
        templateUrl: "main/modals/about.html",
        controller: "aboutCtrl",
        size: "md",
        resolve: {
          pscope: function() {
            return $scope;
          }
        }
      }).result.then(angular.noop, angular.noop);
    }

    function isExternalPathEnabled() {
      return settingsSvs.externalPathEnabled.get() > 0
    }

    function showBucketsOrFiles() {
      if (isExternalPathsView()) {
        $scope.gotoLocalMode();
      }
    }

    function showExternalPaths() {
      if (isBucketsOrFilesView()) {
        $scope.gotoExternalMode();
      }
    }

    function isBucketsOrFilesView() {
      return $scope.ref.mode.startsWith('local');
    }

    function isExternalPathsView() {
      return $scope.ref.mode.startsWith('external');
    }
  }
]);

angular.module('web')
  .controller('akHistoriesModalCtrl', ['$scope', '$translate', '$uibModalInstance', 'choose', 'Dialog', 'Config', 'AkHistory',
    function ($scope, $translate, $modalInstance, choose, Dialog, Config, AkHistory) {
      const T = $translate.instant;
      angular.extend($scope, {
        histories: AkHistory.list(),
        chooseIt: chooseIt,
        removeIt: removeIt,
        cleanAll: cleanAll,
        cancel: cancel,
      });

      function chooseIt(history) {
        choose(history);
        cancel();
      }

      function removeIt(history) {
          const title = T("auth.removeAK.title");
          const message = T("auth.removeAK.message", { id: history.accessKeyId, description: history.description });
          Dialog.confirm(title, message, function (confirm) {
              if (confirm) {
                AkHistory.remove(history.accessKeyId)
                $scope.histories = AkHistory.list();
              }
            },
            1
          );
      }

      function cleanAll() {
          const title = T("auth.clearAKHistories.title");
          const message = T("auth.clearAKHistories.message");
          const successMessage = T("auth.clearAKHistories.successMessage");
          Dialog.confirm(title, message, function (confirm) {
              if (confirm) {
                AkHistory.clearAll();
                $scope.histories = AkHistory.list();
                Toast.success(successMessage);
              }
            },
            1
          );
      }

      function cancel() {
        $modalInstance.dismiss('close');
      }
    }
]);

angular.module('web')
  .controller('customizeCloudModalCtrl', ['$scope', '$translate', '$uibModalInstance', '$timeout', 'Config', 'QiniuClient',
    function ($scope, $translate, $modalInstance, $timeout, Config, QiniuClient) {
      const T = $translate.instant;

      let config = { ucUrl: '', regions: [{}] };
      if (Config.exists()) {
        try {
          config = Config.load(false);
        } catch (e) {
          // do nothing;
        }
      }
      if (config.regions && config.regions.length > 0) {
        config.regions.forEach((region) => {
          if (region.s3Urls && region.s3Urls.length > 0) {
            region.endpoint = region.s3Urls[0];
          }
        });
      } else {
        config.regions = [];
      }

      angular.extend($scope, {
        editRegions: editRegions,
        queryAvailable: false,
        ucUrl: config.ucUrl,
        regions: config.regions,
        addRegion: addRegion,
        removeRegion: removeRegion,
        onSubmit: onSubmit,
        cancel: cancel,
        onUcUrlUpdate: onUcUrlUpdate,
      });
      normalizeRegions();
      onUcUrlUpdate();

      function editRegions() {
        return $scope.regions && $scope.regions.length || !$scope.queryAvailable;
      }

      function onUcUrlUpdate() {
        if (!$scope.ucUrl) {
          $timeout(() => {
            $scope.queryAvailable = false;
            normalizeRegions();
          });
          return;
        }
        const ucUrl = $scope.ucUrl;
        QiniuClient.isQueryRegionAPIAvaiable($scope.ucUrl).then((result) => {
          if (ucUrl === $scope.ucUrl) {
            $timeout(() => {
              $scope.queryAvailable = result;
              normalizeRegions();
            });
          }
        }).catch((err) => {
          if (ucUrl === $scope.ucUrl) {
            $timeout(() => {
              $scope.queryAvailable = false;
              normalizeRegions();
            });
          }
        });
      }

      function normalizeRegions() {
        if (($scope.regions === null || $scope.regions.length === 0) && !$scope.queryAvailable) {
          $scope.regions = [{}];
        }
      }

      function addRegion() {
        if ($scope.regions === null) {
          $scope.regions = [];
        }
        $scope.regions.push({});
      }

      function removeRegion(index) {
        if ($scope.regions === null) {
          $scope.regions = [];
        }
        $scope.regions.splice(index, 1);
      }

      function onSubmit(form) {
        if (!form.$valid) return false;

        let ucUrl = $scope.ucUrl;
        let regions = null;
        if (editRegions()) {
          regions = angular.copy($scope.regions);
          regions.forEach((region) => {
            if (region.endpoint) {
              region.s3Urls = [region.endpoint];
            }
          });
        }
        Config.save(ucUrl, regions);
        cancel();
      }

      function cancel() {
        $modalInstance.dismiss('close');
      }
    }
  ]);

angular.module('web')
  .controller('addBucketModalCtrl', ['$scope', '$uibModalInstance', '$translate', 'callback', 'QiniuClient', 'qiniuClientOpt', 'Const', 'Toast', 'AuditLog', 'regions',
    function ($scope, $modalInstance, $translate, callback, QiniuClient, qiniuClientOpt, Const, Toast, AuditLog, regions) {
      const T = $translate.instant,
            bucketACL = angular.copy(Const.bucketACL);
      angular.extend($scope, {
        regions: regions.filter((region) => !region.cannotCreateBucket),
        bucketACL: [], //angular.copy(Const.bucketACL),
        cancel: cancel,
        onSubmit: onSubmit,
        item: {
          acl: bucketACL[0].acl,
          region: regions[0].s3Id,
        },
        reg: /^[a-z0-9][a-z0-9\-]{1,61}[a-z0-9]$/i,
        openURL: function (v) {
          openExternal(v)
        }
      });

      i18nBucketACL();

      function i18nBucketACL() {
        const acls = angular.copy(Const.bucketACL);
        angular.forEach(acls, function (n) {
          n.label = T('aclType.' + n.acl);
        });
        $scope.bucketACL = acls;
      }

      function cancel() {
        $modalInstance.dismiss('cancel');
      }

      function onSubmit(form) {
        if (!form.$valid) return;

        var item = angular.copy($scope.item);
        QiniuClient.createBucket(item.region, item.name, qiniuClientOpt).then((result) => {
          AuditLog.log('addBucket', {
            regionId: item.region,
            name: item.name,
            acl: item.acl,
          });
          callback();
          cancel();
        });
      }
    }
  ]);

angular.module('web')
  .controller('addExternalPathModalCtrl', ['$scope', '$uibModalInstance', '$translate', 'callback', 'ExternalPath', 'QiniuClient', 'Const', 'regions', 'AuditLog', 'Toast',
    function ($scope, $modalInstance, $translate, callback, ExternalPath, QiniuClient, Const, regions, AuditLog, Toast) {
      const T = $translate.instant;

      angular.extend($scope, {
        regions: regions,
        item: {
            regionId: regions[0].s3Id,
        },
        cancel: cancel,
        onSubmit: onSubmit,
      });

      function cancel() {
        $modalInstance.dismiss('cancel');
      }

      function onSubmit(form) {
        if (!form.$valid) return;

        const item = angular.copy($scope.item);

        const newExternalPath = ExternalPath.new(item.path, item.regionId);
        QiniuClient.listFiles(
          newExternalPath.regionId, newExternalPath.bucketId, newExternalPath.objectPrefix, undefined,
          { preferS3Adapter: true, maxKeys: 1, minKeys: 0 },
        ).then(() => {
          ExternalPath.create(item.path, item.regionId).then(() => {
            AuditLog.log('addExternalPath', {
              path: item.path,
              regionId: item.regionId
            });
            callback();
            cancel();
          }).catch((err) => {
            Toast.error(err);
            cancel();
          });
        }).catch((err) => {
          Toast.error(err);
          cancel();
        })
      }
    }
  ]);

angular.module('web')
  .controller('addFolderModalCtrl', ['$scope', '$uibModalInstance', 'currentInfo', 'qiniuClientOpt', 'callback', 'QiniuClient', 'AuditLog',
    function ($scope, $modalInstance, currentInfo, qiniuClientOpt, callback, QiniuClient, AuditLog) {

      angular.extend($scope, {
        currentInfo: currentInfo,
        item: {},
        cancel: cancel,
        onSubmit: onSubmit,
        reg: {
          folderName: /^[^\/]+$/
        }
      });

      function cancel() {
        $modalInstance.dismiss('close');
      }

      function onSubmit(form) {
        if (!form.$valid) return;

        const folderName = $scope.item.name;
        const fullPath = currentInfo.key + folderName + '/';
        QiniuClient.createFolder(currentInfo.regionId, currentInfo.bucketName, fullPath, qiniuClientOpt).then(function () {
          AuditLog.log('addFolder', {
            regionId: currentInfo.regionId,
            bucket: currentInfo.bucketName,
            path: fullPath
          });
          callback();
          cancel();
        });
      }
    }
  ]);

angular.module('web')
  .controller('deleteFilesModalCtrl', ['$scope', '$q', '$uibModalInstance', '$timeout', 'items', 'currentInfo', 'callback', 'QiniuClient', 'qiniuClientOpt', 'safeApply', 'AuditLog',
    function ($scope, $q, $modalInstance, $timeout, items, currentInfo, callback, QiniuClient, qiniuClientOpt, safeApply, AuditLog) {
      angular.extend($scope, {
        items: items,

        currentInfo:currentInfo,
        step : 1,
        start: start,
        stop: stop,
        close: close
      });

      function stop() {
        //$modalInstance.dismiss('cancel');
        $scope.isStop = true;
        QiniuClient.stopDeleteFiles();
      }
      function close(){
        $modalInstance.dismiss('cancel');
      }

      function start(){
        $scope.isStop = false;
        $scope.step = 2;

        AuditLog.log('deleteFiles', {
          regionId: currentInfo.regionId,
          bucket: currentInfo.bucketName,
          paths: items.map((item) => item.path),
        });

        QiniuClient.deleteFiles(currentInfo.regionId, currentInfo.bucketName, items, (prog) => {
          //进度
          $scope.progress = angular.copy(prog);
          safeApply($scope);
        }, qiniuClientOpt).then((terr) => {
          //结果
          $scope.step = 3;
          $scope.terr = terr;
          if (!terr || terr.length === 0) {
            AuditLog.log('deleteFilesDone');
          }
          callback();
        });
      }
    }])
;

angular.module('web')
  .controller('moveModalCtrl', ['$scope', '$uibModalInstance', '$translate', '$timeout', 'items', 'isCopy', 'renamePath', 'fromInfo', 'moveTo', 'qiniuClientOpt', 'callback', 'QiniuClient', 'Toast', 'safeApply', 'AuditLog',
    function ($scope, $modalInstance, $translate, $timeout, items, isCopy, renamePath, fromInfo, moveTo, qiniuClientOpt, callback, QiniuClient, Toast, safeApply, AuditLog) {
      const path = require("path"),
            T = $translate.instant;

      angular.extend($scope, {
        renamePath: renamePath,
        fromInfo: fromInfo,
        items: items,
        isCopy: isCopy,
        step: 2,

        cancel: cancel,
        start: start,
        stop: stop,

        moveTo: {
          region: moveTo.regionId,
          bucket: moveTo.bucketName,
          key: moveTo.key,
        },
      });

      start();

      function stop() {
        //$modalInstance.dismiss('cancel');
        $scope.isStop = true;
        QiniuClient.stopMoveOrCopyFiles();
      }

      function cancel() {
        $modalInstance.dismiss('cancel');
      }

      function start() {
        $scope.isStop = false;
        $scope.step = 2;
        safeApply($scope);

        var target = angular.copy($scope.moveTo);
        var items = angular.copy($scope.items).filter((item) => {
          if (fromInfo.bucketName !== target.bucket) {
            return true;
          }
          var entries = [target.key, item.name].filter((name) => name);
          var path = entries.map((name) => name.replace(/^\/*([^/].+[^/])\/*$/, '$1'));
          if (item.itemType === 'folder') {
            return item.path !== path + '/';
          }
          return item.path !== path;
        });

        if (items.length === 0) {
          cancel();
          callback();
          return;
        }

        AuditLog.log('moveOrCopyFilesStart', {
          regionId: fromInfo.regionId,
          from: items.map((item) => {
            return { bucket: item.bucket, path: item.path };
          }),
          to: {
            bucket: target.bucket,
            path: target.key
          },
          type: isCopy ? 'copy' : 'move'
        });

        //复制 or 移动
        QiniuClient.moveOrCopyFiles(fromInfo.regionId, items, target, (prog) => {
          //进度
          $scope.progress = angular.copy(prog);
          safeApply($scope);
        }, isCopy, renamePath, qiniuClientOpt).then((terr) => {
          //结果
          $scope.step = 3;
          $scope.terr = terr;
          AuditLog.log('moveOrCopyFilesDone');
          callback();
          safeApply($scope);
        });
      }
    }
  ]);

angular.module('web')
  .controller('renameModalCtrl', ['$scope', '$uibModalInstance', '$translate', '$uibModal', 'item', 'isCopy', 'currentInfo', 'moveTo', 'qiniuClientOpt', 'callback', 'QiniuClient', 'Dialog', 'Toast', 'AuditLog', 'safeApply',
    function ($scope, $modalInstance, $translate, $modal, item, isCopy, currentInfo, moveTo, qiniuClientOpt, callback, QiniuClient, Dialog, Toast, AuditLog, safeApply) {
      var T = $translate.instant;
      //console.log(item)
      angular.extend($scope, {
        currentInfo: currentInfo,
        moveTo: moveTo,
        item: item,
        isCopy: isCopy,
        keep: {
          name: item.name
        },
        cancel: cancel,
        onSubmit: onSubmit,
        reg: {
          folderName: /^[^\/]+$/
        },
        isLoading: false,
        error_message: null
      });

      function cancel() {
        $modalInstance.dismiss('close');
      }

      function onSubmit(form) {
      if (!form.$valid) return;

        const title = T('whetherCover.title'); //是否覆盖
        const msg1 = T('whetherCover.message1'); //已经有同名目录，是否覆盖?
        const msg2 = T('whetherCover.message2'); //已经有同名文件，是否覆盖?

        if ($scope.item.itemType === 'folder') {
          const newPath = `${moveTo.key == '' ? item.name : (moveTo.key.replace(/(\/$)/, '') + '/' + item.name)}/`;
          if (item.path == newPath) return;

          $scope.isLoading = true;
          QiniuClient.checkFolderExists(moveTo.regionId, moveTo.bucketName, newPath, qiniuClientOpt).then((has) => {
            if (has) {
              Dialog.confirm(title, msg1, function (b) {
                if (b) {
                  showMoveFolder(newPath);
                } else {
                  $scope.isLoading = false;
                  safeApply($scope);
                }
              });
            } else {
              showMoveFolder(newPath);
            }
          }).catch((err) => {
            $scope.isLoading = false;
            safeApply($scope);
          });
        } else {
          const newPath = moveTo.key == '' ? item.name : (moveTo.key.replace(/(\/$)/, '') + '/' + item.name);
          if (item.path == newPath) return;

          $scope.isLoading = true;
          QiniuClient.checkFileExists(moveTo.regionId, moveTo.bucketName, newPath, qiniuClientOpt).then((exists) => {
            if (exists) {
              Dialog.confirm(title, msg2, (b) => {
                if (b) {
                  renameFile(newPath);
                } else {
                  $scope.isLoading = false;
                  safeApply($scope);
                }
              });
            } else {
              renameFile(newPath);
            }
          });
        }
      }

      function renameFile(newPath) {
        var onMsg = T('rename.on'); //正在重命名...
        var successMsg = T('rename.success'); //重命名成功

        Toast.info(onMsg);
        QiniuClient.moveOrCopyFile(currentInfo.regionId, currentInfo.bucketName, item.path, newPath, isCopy, qiniuClientOpt).then(() => {
          Toast.success(successMsg);

          AuditLog.log('moveOrCopyFile', {
            regionId: currentInfo.regionId,
            bucket: currentInfo.bucketName,
            from: item.path,
            to: newPath,
            type: isCopy ? 'copy' : 'move',
            storageClass: item.StorageClass
          });

          callback();
          cancel();
        }).finally(() => {
          $scope.isLoading = false;
          safeApply($scope);
        });
      }

      function showMoveFolder(newPath) {
        var successMsg = T('rename.success'); //重命名成功
        $modal.open({
          templateUrl: 'main/files/modals/move-modal.html',
          controller: 'moveModalCtrl',
          backdrop: 'static',
          resolve: {
            items: function () {
              return angular.copy([item]);
            },
            moveTo: function () {
              return angular.copy(moveTo);
            },
            renamePath: function () {
              return newPath;
            },
            isCopy: function () {
              return isCopy;
            },
            fromInfo: function () {
              return angular.copy(currentInfo);
            },
            qiniuClientOpt: () => {
              return angular.copy(qiniuClientOpt);
            },
            callback: function () {
              return function () {
                Toast.success(successMsg);
                $scope.isLoading = false;
                safeApply($scope);
                callback();
                cancel();
              };
            }
          }
        }).result.then(angular.noop, angular.noop);
      }
    }
  ]);

angular.module('web')
  .controller('restoreFilesModalCtrl', ['$scope', '$q', '$uibModalInstance', '$timeout', 'items', 'currentInfo', 'callback', 'QiniuClient', 'qiniuClientOpt', 'safeApply', 'AuditLog',
    function ($scope, $q, $modalInstance, $timeout, items, currentInfo, callback, QiniuClient, qiniuClientOpt, safeApply, AuditLog) {
      angular.extend($scope, {
        items: items,
        currentInfo:currentInfo,
        info: {
          days: 1,
        },
        step : 1,
        stop: stop,
        close: close,
        onSubmit: onSubmit
      });

      function stop() {
        $scope.isStop = true;
        QiniuClient.stopRestoreFiles();
      }

      function close(){
        $modalInstance.dismiss('cancel');
      }

      function onSubmit(form1) {
      if(!form1.$valid) return;

        $scope.isStop = false;
        $scope.step = 2;
        const days = $scope.info.days;

        AuditLog.log('restoreFiles', {
          regionId: currentInfo.regionId,
          bucket: currentInfo.bucketName,
          paths: items.map((item) => item.path),
          days: days,
        });

        QiniuClient.restoreFiles(currentInfo.regionId, currentInfo.bucketName, items, days, (prog) => {
          //进度
          $scope.progress = angular.copy(prog);
          safeApply($scope);
        }, qiniuClientOpt).then((terr) => {
          //结果
          $scope.step = 3;
          $scope.terr = terr;
          if (!terr || terr.length === 0) {
            AuditLog.log('restoreFilesDone');
          }
          callback();
        });
      }
    }])
;

angular.module('web')
  .controller('restoreModalCtrl', ['$scope', '$uibModalInstance', '$translate', '$timeout', 'QiniuClient', 'item', 'currentInfo', 'qiniuClientOpt', 'Toast', 'safeApply',
    function($scope, $modalInstance, $translate, $timeout, QiniuClient, item, currentInfo, qiniuClientOpt, Toast, safeApply) {
      const T = $translate.instant;
      angular.extend($scope, {
        currentInfo: currentInfo,
        item: item,
        path: `kodo://${currentInfo.bucketName}/${currentInfo.key}${item.name}`,
        info: {
          days: 1,
          msg: null
        },
        cancel: cancel,
        onSubmit: onSubmit
      });

      $timeout(init);

      function init(){
        $scope.isLoading = true;
        QiniuClient.getFrozenInfo(currentInfo.regionId, currentInfo.bucketName, item.path, qiniuClientOpt).then((data) => {
          switch (data.status.toLowerCase()) {
          case 'frozen':
            $scope.info.type = 1;
            // $scope.info.msg = null;
            break;
          case 'unfreezing':
            $scope.info.type = 2;
            // $scope.info.msg = '正在恢复中，请耐心等待！';
            break;
          case 'unfrozen':
            $scope.info.type = 3;
            $scope.info.expiry_date = data['expiry-date'];
            // $scope.info.msg = '可读截止时间：'+ info['expiry-date']
            break;
          }
        }).finally(() => {
          $scope.isLoading = false;
          safeApply($scope);
        });
      }

      function cancel() {
        $modalInstance.dismiss('close');
      }

      function onSubmit(form1) {
        if(!form1.$valid) return;

        const days = $scope.info.days;

        Toast.info(T('restore.on')); //'提交中...'
        QiniuClient.restoreFile(currentInfo.regionId, currentInfo.bucketName, item.path, days, qiniuClientOpt).then(() => {
          Toast.success(T('restore.success')); //'恢复请求已经提交'
          cancel();
        });
      }
    }])
;

angular.module('web')
  .controller('showDownloadLinkModalCtrl', ['$scope', '$q', '$translate', '$uibModalInstance', 'safeApply', 'item', 'current', 'domains', 'showDomains', 'Toast', 'Domains', 'qiniuClientOpt',
    function($scope, $q, $translate, $modalInstance, safeApply, item, current, domains, showDomains, Toast, Domains, qiniuClientOpt) {
      const { clipboard } = require('electron'),
                        T = $translate.instant;

      initCurrentDomain(domains);

      angular.extend($scope, {
        item: item,
        current: current,
        domains: domains,
        showDomains: showDomains,
        info: {
          sec: 600,
          url: null,
        },
        cancel: cancel,
        onSubmit: onSubmit,
        copyDownloadLink: copyDownloadLink,
        refreshDomains: refreshDomains,
      });

      function cancel() {
        $modalInstance.dismiss('close');
      }

      function initCurrentDomain(domains) {
        let found = false;
        if (current.domain !== null) {
          domains.forEach((domain) => {
            if (current.domain.name() === domain.name()) {
              current.domain = domain;
              found = true;
            }
          });
        }
        if (!found) {
          domains.forEach((domain) => {
            if (domain.default()) {
              current.domain = domain;
              found = true;
            }
          });
        }
        if (!found) {
          current.domain = domains[0];
        }
      }

      function onSubmit(form1){
        if(!form1.$valid) return;

        $scope.current.domain.signatureUrl(item.path, $scope.info.sec, qiniuClientOpt).then((url) => {
          $scope.info.url = url.toString();
          safeApply($scope);
        });
      }

      function copyDownloadLink() {
        clipboard.writeText($scope.info.url);
        Toast.success(T("copy.successfully")); //'复制成功'
      }

      function refreshDomains() {
        const info = $scope.current.info;
        Domains.list(info.regionId, info.bucketName, info.bucketGrantedPermission).
                then((domains) => {
                  $scope.domains = domains;
                  initCurrentDomain(domains);
                  safeApply($scope);
                });
      }
    }
  ]);

angular.module('web')
  .controller('showDownloadLinksModalCtrl', ['$scope', '$timeout', '$translate', '$uibModalInstance', 'safeApply', 'QiniuClient', 'items', 'current', 'domains', 'showDomains', 'Dialog', 'Toast', 'Domains', 'qiniuClientOpt',
    function($scope, $timeout, $translate, $modalInstance, safeApply, QiniuClient, items, current, domains, showDomains, Dialog, Toast, Domains, qiniuClientOpt) {
      const T = $translate.instant,
            fs = require('fs'),
            path = require('path'),
            csvStringify = require('csv-stringify');

      initCurrentDomain(domains);

      angular.extend($scope, {
        items: items,
        current: current,
        domains: domains,
        showDomains: showDomains,
        info: { sec: 600 },
        cancel: cancel,
        onSubmit: onSubmit,
      });

      function cancel() {
        $modalInstance.dismiss('close');
      }

      function initCurrentDomain(domains) {
        let found = false;
        if (current.domain !== null) {
          domains.forEach((domain) => {
            if (current.domain.name() === domain.name()) {
              current.domain = domain;
              found = true;
            }
          });
        }
        if (!found) {
          domains.forEach((domain) => {
            if (domain.default()) {
              current.domain = domain;
              found = true;
            }
          });
        }
        if (!found) {
          current.domain = domains[0];
        }
      }

      function onSubmit(form1){
        if(!form1.$valid) return;
        const lifetime = $scope.info.sec;

        Dialog.showDownloadDialog((folderPaths) => {
          if (!folderPaths || folderPaths.length == 0) {
            return;
          }
          const targetDirectory = folderPaths[0].replace(/(\/*$)/g, '');
          cancel();

          const fileName = `kodo-browser-download-links-${moment().utc().format('YYYYMMDDHHmmSS')}`;
          let filePath = path.join(targetDirectory, `${fileName}.csv`);
          for (let i = 1; fs.existsSync(filePath); i++) {
            filePath = path.join(targetDirectory, `${fileName}.${i}.csv`);
          }
          const csvFile = fs.createWriteStream(filePath);
          const csvStringifier = csvStringify();

          csvStringifier.on('readable', function() {
            let row;
            while(row = csvStringifier.read()) {
              csvFile.write(row);
            }
          });
          csvStringifier.on('error', function(err) {
            Toast.error(err.message)
          });
          csvStringifier.on('finish', function() {
            csvFile.end();
            Toast.success(T('exportDownloadLinks.message', {path: filePath}), 5000);
          });
          csvStringifier.write(['BucketName', 'ObjectName', 'URL']);
          const promises = [];
          loopItems(current.info.regionId, current.info.bucketName, items,
            (item) => {
              promises.push($scope.current.domain.signatureUrl(item.path, lifetime, qiniuClientOpt).then((url) => {
                              csvStringifier.write([current.info.bucketName, item.path.toString(), url.toString()]);
                            }));
            }, () => {
              Promise.all(promises).then(() => { csvStringifier.end(); });
            });
        });
      }

      function refreshDomains() {
        const info = $scope.current.info;
        Domains.list(info.regionId, info.bucketName, info.bucketGrantedPermission).
                then((domains) => {
                  $scope.domains = domains;
                  initCurrentDomain(domains);
                  safeApply($scope);
                });
      }

      function loopItems(region, bucket, items, eachCallback, doneCallback) {
        let waitForDirs = 0;
        loopItemsInDirectory(items, eachCallback, doneCallback);

        function loopItemsInDirectory(items, eachCallback, doneCallback) {
          items.forEach((item) => {
            if (item.itemType === 'folder') {
              waitForDirs += 1;
              loadFilesFromDirectory(
                item,
                (items) => {
                  loopItemsInDirectory(items, eachCallback, doneCallback);
                },
                () => {
                  waitForDirs--;
                  if (waitForDirs == 0) {
                    doneCallback();
                  }
                })
            } else {
              eachCallback(item);
            }
          });
          if (waitForDirs == 0) {
            doneCallback();
          }
        }

        function loadFilesFromDirectory(item, handleItems, doneCallback, marker) {
          QiniuClient
            .listFiles(
              region, bucket, item.path, marker,
              angular.extend(qiniuClientOpt, { maxKeys: 1000, minKeys: 0 }),
            )
            .then((result) => {
                handleItems(result.data || []);
                if (result.marker) {
                  loadFilesFromDirectory(item, handleItems, doneCallback, result.marker);
                } else {
                  doneCallback();
                }
            });
        }
      }
    }
  ]);

angular.module('web')
  .controller('updateStorageClassModalCtrl', ['$scope', '$uibModalInstance', '$translate', '$timeout', 'QiniuClient', 'item', 'currentInfo', 'qiniuClientOpt', 'Toast', 'safeApply', 'callback',
    function($scope, $modalInstance, $translate, $timeout, QiniuClient, item, currentInfo, qiniuClientOpt, Toast, safeApply, callback) {
      const T = $translate.instant;
      angular.extend($scope, {
        currentInfo: currentInfo,
        item: item,
        path: `kodo://${currentInfo.bucketName}/${currentInfo.key}${item.name}`,
        info: {
          type: 1,
          name: T('storageClassesType.standard'),
        },
        cancel: cancel,
        onSubmit: onSubmit
      });

      $timeout(init);

      function init(){
        $scope.isLoading = true;
        QiniuClient.headFile(currentInfo.regionId, currentInfo.bucketName, item.path, qiniuClientOpt).then((info) => {
          switch (info.storageClass.toLowerCase()) {
          case 'standard':
            $scope.info.type = 1;
            $scope.info.updateTo = 'InfrequentAccess';
            break;
          case 'infrequentaccess':
            $scope.info.type = 2;
            $scope.info.updateTo = 'Standard';
            break;
          case 'glacier':
            $scope.info.type = 3;
            $scope.info.updateTo = 'Standard';
            break;
          }
          $scope.info.name = T(`storageClassesType.${info.storageClass.toLowerCase()}`);
        }).finally(() => {
          $scope.isLoading = false;
          safeApply($scope);
        });
      }

      function cancel() {
        $modalInstance.dismiss('close');
      }

      function onSubmit(form1) {
        if(!form1.$valid) return;

        QiniuClient.setStorageClass(currentInfo.regionId, currentInfo.bucketName, item.path, $scope.info.updateTo, qiniuClientOpt).then(() => {
          Toast.success(T('updateStorageClass.success')); //'修改存储类型成功'
          callback();
          cancel();
        });
      }
    }])
;

angular.module('web')
  .controller('updateStorageClassesModalCtrl', ['$scope', '$q', '$uibModalInstance', '$timeout', 'items', 'currentInfo', 'callback', 'QiniuClient', 'qiniuClientOpt', 'safeApply', 'AuditLog',
    function ($scope, $q, $modalInstance, $timeout, items, currentInfo, callback, QiniuClient, qiniuClientOpt, safeApply, AuditLog) {
      angular.extend($scope, {
        items: items,
        currentInfo:currentInfo,
        info: {
          updateTo: 'Standard',
        },
        step : 1,
        stop: stop,
        close: close,
        onSubmit: onSubmit
      });

      function stop() {
        $scope.isStop = true;
        QiniuClient.stopSetStorageClassOfFiles();
      }

      function close(){
        $modalInstance.dismiss('cancel');
      }

      function onSubmit(form1) {
      if(!form1.$valid) return;

        $scope.isStop = false;
        $scope.step = 2;
        const newStorageClass = $scope.info.updateTo;

        AuditLog.log('setStorageClassOfFiles', {
          regionId: currentInfo.regionId,
          bucket: currentInfo.bucketName,
          paths: items.map((item) => item.path),
          updateTo: newStorageClass,
        });

        QiniuClient.setStorageClassOfFiles(currentInfo.regionId, currentInfo.bucketName, items, newStorageClass, (prog) => {
          //进度
          $scope.progress = angular.copy(prog);
          safeApply($scope);
        }, qiniuClientOpt).then((terr) => {
          //结果
          $scope.step = 3;
          $scope.terr = terr;
          if (!terr || terr.length === 0) {
            AuditLog.log('setStorageClassOfFilesDone');
          }
          callback();
        });
      }
    }])
;

"use strict";

angular.module("web").controller("transferDownloadsCtrl", [
  "$scope",
  "$timeout",
  "$translate",
  "jobUtil",
  "DownloadMgr",
  "DelayDone",
  "Toast",
  "Const",
  "Dialog",
  function (
    $scope,
    $timeout,
    $translate,
    jobUtil,
    DownloadMgr,
    DelayDone,
    Toast,
    Const,
    Dialog
  ) {
    var T = $translate.instant;

    angular.extend($scope, {
      triggerOverwriting: triggerOverwriting,
      showRemoveItem: showRemoveItem,
      clearAllCompleted: clearAllCompleted,
      clearAll: clearAll,
      stopAll: stopAll,
      startAll: startAll,
      checkStartJob: checkStartJob,

      sch: {
        downname: null
      },
      schKeyFn: function (item) {
        return (
          item.to.name +
          " " +
          item.status +
          " " +
          jobUtil.getStatusLabel(item.status)
        );
      },
      limitToNum: 100,
      loadMoreDownloadItems: loadMoreItems
    });

    function loadMoreItems() {
      var len = $scope.lists.downloadJobList.length;
      if ($scope.limitToNum < len) {
        $scope.limitToNum += Math.min(100, len - $scope.limitToNum);
      }
    }

    function triggerOverwriting() {
      $scope.overwriteDownloading.enabled = !$scope.overwriteDownloading.enabled;
      localStorage.setItem(Const.OVERWRITE_DOWNLOADING, $scope.overwriteDownloading.enabled);
    }

    function checkStartJob(item) {
      item.wait();

      DownloadMgr.trySchedJob();
    }

    function showRemoveItem(item) {
      if (item.status == "finished") {
        doRemove(item);
      } else {
        var title = T("remove.from.list.title"); //'从列表中移除'
        var message = T("remove.from.list.message"); //'确定移除该下载任务?'
        Dialog.confirm(
          title,
          message,
          (btn) => {
            if (btn) {
              if (item.status == "running" ||
                item.status == "waiting" ||
                item.status == "verifying") {
                item.stop();
              }

              doRemove(item);
            }
          },
          1
        );
      }
    }

    function doRemove(item) {
      var jobs = $scope.lists.downloadJobList;
      for (var i = 0; i < jobs.length; i++) {
        if (item === jobs[i]) {
          jobs.splice(i, 1);
          break;
        }
      }

      $timeout(() => {
        DownloadMgr.trySaveProg();
        $scope.calcTotalProg();
      });
    }

    function clearAllCompleted() {
      var jobs = $scope.lists.downloadJobList;
      for (var i = 0; i < jobs.length; i++) {
        if ("finished" == jobs[i].status) {
          jobs.splice(i, 1);
          i--;
        }
      }

      $timeout(() => {
        $scope.calcTotalProg();
      });
    }

    function clearAll() {
      if (!$scope.lists.downloadJobList ||
        $scope.lists.downloadJobList.length == 0) {
        return;
      }

      var title = T("clear.all.title"); //清空所有
      var message = T("clear.all.download.message"); //确定清空所有下载任务?
      Dialog.confirm(
        title,
        message,
        (btn) => {
          if (btn) {
            var jobs = $scope.lists.downloadJobList;
            for (var i = 0; i < jobs.length; i++) {
              var job = jobs[i];
              if (job.status == "running" ||
                job.status == "waiting" ||
                job.status == "verifying") {
                job.stop();
              }

              jobs.splice(i, 1);
              i--;
            }

            $timeout(() => {
              DownloadMgr.trySaveProg();
              $scope.calcTotalProg();
            });
          }
        },
        1
      );
    }

    var stopFlag = false;

    function stopAll() {
      var jobs = $scope.lists.downloadJobList;
      if (jobs && jobs.length > 0) {
        stopFlag = true;

        DownloadMgr.stopCreatingJobs();

        Toast.info(T("pause.on")); //'正在暂停...'
        $scope.allActionBtnDisabled = true;

        angular.forEach(jobs, (job) => {
          if (job.prog.resumable && (
              job.status == "running" ||
              job.status == "waiting" ||
              job.status == "verifying"
            ))
            n.stop();
        });
        Toast.success(T("pause.success")); //'暂停成功'

        $timeout(() => {
          DownloadMgr.trySaveProg();
          $scope.allActionBtnDisabled = false;
        });
      }
    }

    function startAll() {
      stopFlag = false;

      //串行
      var jobs = $scope.lists.downloadJobList;
      if (jobs && jobs.length > 0) {
        $scope.allActionBtnDisabled = true;
        DelayDone.seriesRun(
          jobs,
          (job, fn) => {
            if (stopFlag) return;

            if (job && (job.status == "stopped" || job.status == "failed")) {
              job.wait();
            }

            fn();
          },
          () => {
            DownloadMgr.trySchedJob();
            $scope.allActionBtnDisabled = false;
          }
        );
      }
    }
  }
]);

angular.module("web").controller("transferFrameCtrl", [
  "$scope",
  "$translate",
  "UploadMgr",
  "DownloadMgr",
  "Toast",
  "Const",
  "AuditLog",
  function (
    $scope,
    $translate,
    UploadMgr,
    DownloadMgr,
    Toast,
    Const,
    AuditLog
  ) {
    const T = $translate.instant;

    angular.extend($scope, {
      transTab: 1,

      lists: {
        uploadJobList: [],
        downloadJobList: []
      },
      emptyFolderUploading: {
        enabled: localStorage.getItem(Const.EMPTY_FOLDER_UPLOADING) || true,
      },
      overwriteUploading: {
        enabled: localStorage.getItem(Const.OVERWRITE_UPLOADING) || false,
      },
      overwriteDownloading: {
        enabled: localStorage.getItem(Const.OVERWRITE_DOWNLOADING) || false,
      },

      totalStat: {
        running: 0,
        total: 0,
        upDone: 0,
        upStopped: 0,
        upFailed: 0,
        downDone: 0,
        downStopped: 0,
        downFailed: 0
      },

      calcTotalProg: calcTotalProg
    });

    // functions in parent scope
    $scope.handlers.uploadFilesHandler = uploadFilesHandler;
    $scope.handlers.downloadFilesHandler = downloadFilesHandler;

    UploadMgr.init($scope);
    DownloadMgr.init($scope);

    /**
     * upload
     * @param filePaths []  {array<string>}, iter for folder
     * @param bucketInfo {object} {bucket, region, key}
     */
    function uploadFilesHandler(filePaths, bucketInfo) {
      Toast.info(T("upload.addtolist.on"));
      UploadMgr.createUploadJobs(filePaths, bucketInfo, function (isCancelled) {
        Toast.info(T("upload.addtolist.success"));

        $scope.transTab = 1;
        $scope.toggleTransVisible(true);

        AuditLog.log('uploadFilesStart', {
          regionId: bucketInfo.region,
          bucket: bucketInfo.bucketName,
          to: bucketInfo.key,
          from: filePaths
        });
      });
    }

    /**
     * download
     * @param fromRemotePath {array}  item={region, bucket, path, name, domain, size=0, itemType='file'}, create folder if required
     * @param toLocalPath {string}
     */
    function downloadFilesHandler(fromRemotePath, toLocalPath) {
      Toast.info(T("download.addtolist.on"));
      DownloadMgr.createDownloadJobs(fromRemotePath, toLocalPath, function (isCancelled) {
        Toast.info(T("download.addtolist.success"));

        AuditLog.log('downloadFilesStart', {
          from: fromRemotePath.map((entry) => {
            return { regionId: entry.region, bucket: entry.bucketName, path: entry.path.toString() };
          }),
          to: toLocalPath
        });

        $scope.transTab = 2;
        $scope.toggleTransVisible(true);
      });
    }

    function calcTotalProg() {
      let c = 0, c2 = 0, cf = 0, cf2 = 0, cs = 0, cs2 = 0;

      angular.forEach($scope.lists.uploadJobList, function (n) {
        if (n.status === 'running') {
          c++;
        }
        if (n.status === 'waiting') {
          c++;
        }
        if (n.status === 'verifying') {
          c++;
        }
        if (n.status === 'failed') {
          cf++;
        }
        if (n.status === 'stopped') {
          c++;
          cs++;
        }
      });
      angular.forEach($scope.lists.downloadJobList, function (n) {
        if (n.status === 'running') {
          c2++;
        }
        if (n.status === 'waiting') {
          c2++;
        }
        if (n.status === 'failed') {
          cf2++;
        }
        if (n.status === 'stopped') {
          c2++;
          cs2++;
        }
      });

      $scope.totalStat.running = c + c2;
      $scope.totalStat.total = $scope.lists.uploadJobList.length + $scope.lists.downloadJobList.length;
      $scope.totalStat.upDone = $scope.lists.uploadJobList.length - c;
      $scope.totalStat.upStopped = cs;
      $scope.totalStat.upFailed = cf;
      $scope.totalStat.downDone = $scope.lists.downloadJobList.length - c2;
      $scope.totalStat.downStopped = cs2;
      $scope.totalStat.downFailed = cf2;
    }
  }
]);

"use strict";

angular.module("web").controller("transferUploadsCtrl", [
  "$scope",
  "$timeout",
  "$translate",
  "jobUtil",
  "DelayDone",
  "UploadMgr",
  "Toast",
  "Const",
  "Dialog",
  function (
    $scope,
    $timeout,
    $translate,
    jobUtil,
    DelayDone,
    UploadMgr,
    Toast,
    Const,
    Dialog
  ) {
    var T = $translate.instant;

    angular.extend($scope, {
      triggerEmptyFolder: triggerEmptyFolder,
      triggerOverwriting: triggerOverwriting,
      showRemoveItem: showRemoveItem,
      clearAllCompleted: clearAllCompleted,
      clearAll: clearAll,
      stopAll: stopAll,
      startAll: startAll,
      checkStartJob: checkStartJob,

      sch: {
        upname: null
      },
      schKeyFn: function (item) {
        return (
          item.from.name +
          " " +
          item.status +
          " " +
          jobUtil.getStatusLabel(item.status)
        );
      },
      limitToNum: 100,
      loadMoreUploadItems: loadMoreItems
    });

    function loadMoreItems() {
      var len = $scope.lists.uploadJobList.length;
      if ($scope.limitToNum < len) {
        $scope.limitToNum += Math.min(100, len - $scope.limitToNum);
      }
    }

    function triggerEmptyFolder() {
      $scope.emptyFolderUploading.enabled = !$scope.emptyFolderUploading.enabled;
      localStorage.setItem(Const.EMPTY_FOLDER_UPLOADING, $scope.emptyFolderUploading.enabled);
    }

    function triggerOverwriting() {
      $scope.overwriteUploading.enabled = !$scope.overwriteUploading.enabled;
      localStorage.setItem(Const.OVERWRITE_UPLOADING, $scope.overwriteUploading.enabled);
    }

    function checkStartJob(item, force) {
      if (force) {
        item.start(true);
      } else {
        item.wait();
      }

      UploadMgr.trySchedJob();
    }

    function showRemoveItem(item) {
      if (item.status == "finished") {
        doRemove(item);
      } else {
        var title = T("remove.from.list.title"); //'从列表中移除'
        var message = T("remove.from.list.message"); //'确定移除该上传任务?'
        Dialog.confirm(
          title,
          message,
          (btn) => {
            if (btn) {
              if (item.status == "running" ||
                item.status == "waiting" ||
                item.status == "verifying" ||
                item.status == "duplicated") {
                item.stop();
              }

              doRemove(item);
            }
          },
          1
        );
      }
    }

    function doRemove(item) {
      var jobs = $scope.lists.uploadJobList;
      for (var i = 0; i < jobs.length; i++) {
        if (item === jobs[i]) {
          jobs.splice(i, 1);
          break;
        }
      }

      $timeout(() => {
        UploadMgr.trySaveProg();
        $scope.calcTotalProg();
      });
    }

    function clearAllCompleted() {
      var jobs = $scope.lists.uploadJobList;
      for (var i = 0; i < jobs.length; i++) {
        if ("finished" == jobs[i].status) {
          jobs.splice(i, 1);
          i--;
        }
      }

      $timeout(() => {
        $scope.calcTotalProg();
      });
    }

    function clearAll() {
      if (!$scope.lists.uploadJobList ||
        $scope.lists.uploadJobList.length == 0) {
        return;
      }

      var title = T("clear.all.title"); //清空所有
      var message = T("clear.all.upload.message"); //确定清空所有上传任务?
      Dialog.confirm(
        title,
        message,
        (btn) => {
          if (btn) {
            var jobs = $scope.lists.uploadJobList;
            for (var i = 0; i < jobs.length; i++) {
              var job = jobs[i];
              if (job.status == "running" ||
                job.status == "waiting" ||
                job.status == "verifying" ||
                job.status == "duplicated") {
                job.stop();
              }

              jobs.splice(i, 1);
              i--;
            }

            $timeout(() => {
              UploadMgr.trySaveProg();
              $scope.calcTotalProg();
            });
          }
        },
        1
      );
    }

    var stopFlag = false;

    function stopAll() {
      var arr = $scope.lists.uploadJobList;
      if (arr && arr.length > 0) {
        stopFlag = true;

        UploadMgr.stopCreatingJobs();

        Toast.info(T("pause.on")); //'正在暂停...'
        $scope.allActionBtnDisabled = true;

        angular.forEach(arr, function (n) {
          if (item.resumable && (
              n.status == "running" ||
              n.status == "waiting" ||
              n.status == "verifying"
            ))
            n.stop();
        });
        Toast.info(T("pause.success"));

        $timeout(function () {
          UploadMgr.trySaveProg();
          $scope.allActionBtnDisabled = false;
        }, 100);
      }
    }

    function startAll() {
      var arr = $scope.lists.uploadJobList;
      stopFlag = false;
      //串行
      if (arr && arr.length > 0) {
        $scope.allActionBtnDisabled = true;
        DelayDone.seriesRun(
          arr,
          function (n, fn) {
            if (stopFlag) {
              return;
            }

            if (n && (n.status == "stopped" || n.status == "failed")) {
              n.wait();
            }

            UploadMgr.trySchedJob();

            fn();
          },
          function doneFn() {
            $scope.allActionBtnDisabled = false;
          }
        );
      }
    }
  }
]);

angular.module('web')
  .controller('addressBarCtrl', ['$scope', '$translate', 'Bookmark', 'AuthInfo', 'Toast', 'settingsSvs',
    function ($scope, $translate, Bookmark, AuthInfo, Toast, settingsSvs) {

      const KODO_ADDR_PROTOCOL = 'kodo://',
            T = $translate.instant;

      angular.extend($scope, {
        address: KODO_ADDR_PROTOCOL,
        goUp: goUp,
        go: go,
        goHome: goHome,
        saveDefaultAddress: saveDefaultAddress,
        getDefaultAddress: getDefaultAddress,

        marked: marked,
        toggleMark: toggleMark,

        //历史，前进，后退
        canGoAhead: false,
        canGoBack: false,
        goBack: goBack,
        goAhead: goAhead
      });


      function marked() {
        const addressAndMode = getCurrentAddressAndMode();
        return Bookmark.marked(addressAndMode.address, addressAndMode.mode);
      }

      function toggleMark() {
        const addressAndMode = getCurrentAddressAndMode();
        if (Bookmark.marked(addressAndMode.address, addressAndMode.mode)) {
          Bookmark.remove(addressAndMode.address, addressAndMode.mode);
          Toast.warn(T('bookmark.remove.success')); //'已删除书签'
        } else {
          Bookmark.add(addressAndMode.address, addressAndMode.mode);
          Toast.success(T('bookmark.add.success'));//'添加书签成功'
        }
      }

      /************ 历史记录前进后退 start **************/
      const His = new function () {
        const arr = [];
        let index = -1;
        this.add = function (url, mode) {
          if (index > -1 && url === arr[index].url && mode === arr[index].mode) {
            return;
          }
          if (index < arr.length - 1) {
            arr.splice(index + 1, arr.length - index);
          }
          arr.push({ url: url, mode: mode, time: new Date().getTime() });
          index++;

          const MAX = settingsSvs.historiesLength.get();
          if (arr.length > MAX) {
            arr.splice(MAX, arr.length - MAX);
            index = arr.length - 1;
          }

          this._change(index, arr);
        };
        this.clear = function () {
          arr = [];
          index = -1;
          this._change(index, arr);
        };
        this.list = function () {
          return JSON.parse(JSON.stringify(arr));
        };
        this.goBack = function () {
          if (arr.length == 0) return null;
          if (index > 0) {
            index--;
            this._change(index, arr);
          }
          return arr[index];
        };
        this.goAhead = function () {
          if (arr.length == 0) return null;
          if (index < arr.length - 1) {
            index++;
            this._change(index, arr);
          }
          return arr[index];
        };

        //监听事件
        this.onChange = function (fn) {
          this._change = fn;
        };
      };

      His.onChange(function (index, arr) {
        //console.log('histories changed:', index, arr)
        if (arr.length == 0) {
          $scope.canGoBack = false;
          $scope.canGoAhead = false;
        } else {
          $scope.canGoBack = index > 0;
          $scope.canGoAhead = index < arr.length - 1;
        }
      });

      function goBack() {
        var addr = His.goBack();
        //console.log('-->',addr);
        $scope.address = addr.url;
        $scope.ref.mode = addr.mode;
        $scope.$emit('kodoAddressChange', addr.url);
      }
      function goAhead() {
        var addr = His.goAhead();
        //console.log('-->',addr);
        $scope.address = addr.url;
        $scope.ref.mode = addr.mode;
        $scope.$emit('kodoAddressChange', addr.url);
      }
      /************ 历史记录前进后退 end **************/


      $scope.$on('filesViewReady', function () {

        goHome();

        $scope.$on('gotoLocalMode', function (e) {
          console.log('on:gotoLocalMode');
          $scope.address = KODO_ADDR_PROTOCOL;
          $scope.ref.mode = 'localBuckets';
          go();
        });

        $scope.$on('gotoExternalMode', function (e) {
          console.log('on:gotoExternalMode');
          $scope.address = KODO_ADDR_PROTOCOL;
          $scope.ref.mode = 'externalPaths';
          go();
        });

        $scope.$on('gotoKodoAddress', function (e, addr) {
          console.log('on:gotoKodoAddress', addr);
          $scope.address = addr;
          go();
        });
      });

      function goHome() {
        const addressAndMode = getDefaultAddress();
        $scope.address = addressAndMode.address;
        $scope.ref.mode = addressAndMode.mode;
        go();
      }

      //保存默认地址
      function saveDefaultAddress() {
        AuthInfo.saveToAuthInfo(getCurrentAddressAndMode());
        Toast.success(T('saveAsHome.success'), 1000); //'设置默认地址成功'
      }
      function getDefaultAddress() {
        const info = AuthInfo.get();
        if (info.address && info.mode) {
          return { address: info.address, mode: info.mode };
        } else {
          return { address: KODO_ADDR_PROTOCOL, mode: 'localBuckets' };
        }
      }

      //修正并获取 address
      function getCurrentAddressAndMode() {
        let addr = $scope.address;
        if (!addr) {
          $scope.address = KODO_ADDR_PROTOCOL;
        } else if (addr == KODO_ADDR_PROTOCOL) {
          // do nothing
        } else if (addr.indexOf(KODO_ADDR_PROTOCOL) !== 0) {
          addr = addr.replace(/(^\/*)|(\/*$)/g, '');
          $scope.address = addr ? (KODO_ADDR_PROTOCOL + addr + '/') : KODO_ADDR_PROTOCOL;
        }
        return { address: $scope.address, mode: $scope.ref.mode };
      }

      //浏览
      function go() {
        const addressAndMode = getCurrentAddressAndMode();
        His.add(addressAndMode.address, addressAndMode.mode); //历史记录
        $scope.$emit('kodoAddressChange', addressAndMode.address);
      }
      //向上
      function goUp() {
        const addressAndMode = getCurrentAddressAndMode();
        if (addressAndMode.address == KODO_ADDR_PROTOCOL) {
          return go();
        }

        addressAndMode.address = addressAndMode.address.substring(KODO_ADDR_PROTOCOL.length);
        addressAndMode.address = addressAndMode.address.replace(/(^\/?)|(\/?$)/g, '');

        const splits = addressAndMode.address.split('/');

        splits.pop();

        if (splits.length === 0) {
          addressAndMode.address = KODO_ADDR_PROTOCOL;
        } else {
          addressAndMode.address = KODO_ADDR_PROTOCOL + splits.join('/') + '/';
        }
        $scope.address = addressAndMode.address;
        go();
      }
    }]);

angular.module('web')
  .controller('listViewOptionsCtrl', ['$scope', function ($scope) {

    angular.extend($scope, {
      setListView: setListView,
    });

    $scope.ref.isListView = getListView();

    function getListView(){
      return localStorage.getItem('is-list-view') !== 'false';
    }

    function setListView(f) {
      $scope.ref.isListView = f;
      localStorage.setItem('is-list-view', f);
    }
  }]);

angular.module('web')
  .controller('codeModalCtrl', ['$scope', 'safeApply', '$uibModalInstance', '$translate', '$timeout', '$uibModal', 'bucketInfo', 'objectInfo', 'selectedDomain', 'qiniuClientOpt',
             'fileType', 'showFn', 'reload', 'Toast', 'DiffModal', 'QiniuClient',
    function ($scope, safeApply, $modalInstance, $translate, $timeout, $modal, bucketInfo, objectInfo, selectedDomain, qiniuClientOpt, fileType, showFn, reload, Toast, DiffModal, QiniuClient) {
      const T = $translate.instant,
            urllib = require('urllib');
      angular.extend($scope, {
        bucketInfo: bucketInfo,
        objectInfo: objectInfo,
        fileType: fileType,
        qiniuClientOpt: qiniuClientOpt,
        afterCheckSuccess: afterCheckSuccess,
        domain: selectedDomain.domain,

        previewBarVisible: false,
        showFn: showFn,

        cancel: cancel,
        getContent: getContent,
        saveContent: saveContent,
        //showDownload: showDownload,
        MAX_SIZE: 5 * 1024 * 1024
      });

      function afterCheckSuccess() {
        $scope.previewBarVisible = true;
        if (objectInfo.size < $scope.MAX_SIZE) {
          // 修复ubuntu下无法获取的bug
          $timeout(getContent, 100);
        }
      }

      function saveContent() {
        var originalContent = $scope.originalContent;
        var v = editor.getValue();
        $scope.content = v;

        if (originalContent != v) {
          DiffModal.show('Diff', originalContent, v, function (v) {
            Toast.info(T('saving')); //'正在保存...'

            selectedDomain.domain.saveContent(objectInfo.path, v, qiniuClientOpt).then(() => {
              Toast.success(T('save.successfully'));//'保存成功'
              cancel();
              reload();
            });
          });
        } else {
          Toast.info(T('content.isnot.modified')); //内容没有修改
        }
      }

      function getContent() {
        $scope.isLoading = true;
        selectedDomain.domain.getContent(objectInfo.path, qiniuClientOpt).then((data) => {
          const dataString = data.toString();
          $scope.originalContent = dataString;
          $scope.content = dataString;
          editor.setValue(dataString);
        }).finally(() => {
          $scope.isLoading = false;
          safeApply($scope);
        });
      }

      function cancel() {
        $modalInstance.dismiss('close');
      }

      $scope.codeOptions = {
        lineNumbers: true,
        lineWrapping: true,
        autoFocus: true,
        readOnly: false,
        mode: fileType.mode
      };

      var editor;
      $scope.codemirrorLoaded = function (_editor) {
        editor = _editor;
        // Editor part
        var _doc = _editor.getDoc();
        _editor.focus();

        // Options
        _editor.setSize('100%', 500);

        _editor.refresh();

        _doc.markClean();
      };

    }
  ]);

angular.module('web')
  .controller('docModalCtrl', ['$scope','$uibModalInstance','bucketInfo','objectInfo','fileType',
    function ($scope, $modalInstance, bucketInfo, objectInfo, fileType) {

      angular.extend($scope, {
        bucketInfo: bucketInfo,
        objectInfo: objectInfo,
        fileType: fileType,

        cancel: cancel
      });

      function cancel() {
        $modalInstance.dismiss('close');
      }
    }])
;

angular.module('web')
  .controller('mediaModalCtrl', ['$scope', '$uibModalInstance', '$timeout','$sce', '$uibModal', 'selectedDomain', 'showFn', 'bucketInfo', 'objectInfo', 'fileType', 'qiniuClientOpt',
    function ($scope, $modalInstance, $timeout, $sce, $modal, selectedDomain, showFn, bucketInfo, objectInfo, fileType, qiniuClientOpt) {

      angular.extend($scope, {
        bucketInfo: bucketInfo,
        objectInfo: objectInfo,
        fileType: fileType,
        qiniuClientOpt: qiniuClientOpt,
        afterCheckSuccess: afterCheckSuccess,

        previewBarVisible: false,
        showFn: showFn,
        cancel: cancel,

        MAX_SIZE: 5 * 1024 * 1024 //5MB
      });

      function afterCheckSuccess() {
        $scope.previewBarVisible = true;
        genURL();
      }

      function cancel() {
        $modalInstance.dismiss('close');
      }

      function genURL() {
        selectedDomain.domain.signatureUrl(objectInfo.path, qiniuClientOpt).then((url) => {
          $scope.src_origin = url.toString();
          $scope.src = $sce.trustAsResourceUrl(url.toString());

          $timeout(() => {
            const ele = $('#video-player');
            if(parseInt(ele.css('height')) > parseInt(ele.css('width'))){
               ele.css('height', $(document).height()-240);
               ele.css('width', 'auto');
            }
          }, 1000);
        });
      }
    }
  ]);

angular.module('web')
  .controller('othersModalCtrl', ['$scope', '$uibModalInstance', '$uibModal', 'bucketInfo', 'objectInfo', 'fileType', 'qiniuClientOpt', 'showFn', 'safeApply',
    function ($scope, $modalInstance, $modal, bucketInfo, objectInfo, fileType, qiniuClientOpt, showFn, safeApply) {

      angular.extend($scope, {
        bucketInfo: bucketInfo,
        objectInfo: objectInfo,
        fileType: fileType,
        qiniuClientOpt: qiniuClientOpt,
        afterCheckSuccess:afterCheckSuccess,

        previewBarVisible: false,
        showFn: showFn,
        cancel: cancel,

        showAs: showAs,
        //showDownload: showDownload,

        showAsCodeBtn: shouldShowAsCodeBtn()
      });

      function afterCheckSuccess() {
        $scope.previewBarVisible = true;
      }

      function shouldShowAsCodeBtn(){
        var name = objectInfo.name;

        if(name.endsWith('.tar.gz') || name.endsWith('.tar') || name.endsWith('.zip') || name.endsWith('.bz') || name.endsWith('.xz')
         || name.endsWith('.dmg') || name.endsWith('.pkg') || name.endsWith('.apk')
         || name.endsWith('.exe') || name.endsWith('.msi') || name.endsWith('.dll')|| name.endsWith('.chm')
         || name.endsWith('.iso') || name.endsWith('.img') || name.endsWith('.img')
         || name.endsWith('.pdf') || name.endsWith('.doc') || name.endsWith('.docx')) {
          return false;
        }
        return true;
      }

      function cancel() {
        $modalInstance.dismiss('close');
      }

      function showAs(type){
        showFn.preview(objectInfo, type);
        cancel();
      }
    }])
;

angular.module('web')
  .controller('pictureModalCtrl', ['$scope', '$uibModalInstance', '$timeout', '$uibModal', 'showFn', 'selectedDomain', 'bucketInfo', 'objectInfo', 'qiniuClientOpt', 'AuthInfo', 'fileType',
    function ($scope, $modalInstance, $timeout, $modal, showFn, selectedDomain, bucketInfo, objectInfo, qiniuClientOpt, AuthInfo, fileType) {

      angular.extend($scope, {
        bucketInfo: bucketInfo,
        objectInfo: objectInfo,
        fileType: fileType,
        qiniuClientOpt: qiniuClientOpt,
        afterCheckSuccess: afterCheckSuccess,

        previewBarVisible: false,
        showFn: showFn,
        cancel: cancel,

        MAX_SIZE: 5 * 1024 * 1024 //5MB
      });

      function afterCheckSuccess() {
        $scope.previewBarVisible = true;
        if (objectInfo.size < $scope.MAX_SIZE) {
          getContent();
        }
      }

      function cancel() {
        $modalInstance.dismiss('close');
      }

      function getContent() {
        selectedDomain.domain.signatureUrl(objectInfo.path, qiniuClientOpt).then((url) => {
          $timeout(() => {
            $scope.imgsrc = url.toString();
          });
        });
      }
    }
  ]);

angular.module('web')
  .directive('restoreChecker', [
    function () {
      return {
        restrict: 'EA',
        templateUrl: 'main/files/modals/preview/restore-checker.html',
        transclude: true,
        scope: {
          bucketInfo: '=',
          objectInfo: '=',
          fileType: '=',
          qiniuClientOpt: '=',
          afterCheckSuccess: '&',
        },
        controller: ['$scope', '$timeout','$uibModal','QiniuClient','safeApply', ctrlFn]
      }

      function ctrlFn($scope, $timeout, $modal, QiniuClient, safeApply){
        angular.extend($scope, {
          info: {
            msg: null
          },
          _Loading: false,
          showRestore: showRestore,
        });

        init();
        function init() {
          check(function() {
            if ($scope.afterCheckSuccess) {
              $scope.afterCheckSuccess();
            }
          });
        }
        function check(successCallback){
          $scope._Loading = true;

          QiniuClient.getFrozenInfo($scope.bucketInfo.regionId, $scope.bucketInfo.bucketName, $scope.objectInfo.path, $scope.qiniuClientOpt)
                  .then(function (data) {
            switch (data.status) {
            case 'Normal':
              $scope.info.type = 0;
              $scope.info.showContent = true;
              if (successCallback) {
                successCallback()
              }
              break;
            case 'Frozen':
              $scope.info.type = 1; //归档文件，需要恢复才能预览或下载
              $scope.info.showContent = false;
              break;
            case 'Unfreezing':
              $scope.info.type = 2; // 归档文件正在恢复中，请耐心等待...;
              $scope.info.showContent = false;
              break;
            case 'Unfrozen':
              $scope.info.type = 3; // '归档文件，已恢复'
              $scope.info.showContent = true;
              if (successCallback) {
                successCallback()
              }
              break;
            default:
              console.error("Unrecognized status from QiniuClient.getFrozenInfo(): ", data.status);
            }
          }).finally(() => {
            $scope._Loading = false;
            safeApply($scope);
          });
        }

        function showRestore(){
          $modal.open({
            templateUrl: 'main/files/modals/restore-modal.html',
            controller: 'restoreModalCtrl',
            resolve: {
              item: function(){
                return angular.copy($scope.objectInfo);
              },
              currentInfo: function(){
                return angular.copy($scope.bucketInfo);
              },
              qiniuClientOpt: () => {
                return angular.copy($scope.qiniuClientOpt);
              },
              callback: function(){
                return init;
              }
            }
          });
        }
      }
    }])
;
