/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.index.engine;

import java.io.Closeable;
import java.io.IOException;
import java.util.Comparator;
import java.util.List;
import java.util.concurrent.atomic.AtomicBoolean;
import org.apache.lucene.codecs.StoredFieldsReader;
import org.apache.lucene.document.LongPoint;
import org.apache.lucene.index.LeafReader;
import org.apache.lucene.index.LeafReaderContext;
import org.apache.lucene.index.NumericDocValues;
import org.apache.lucene.search.BooleanClause;
import org.apache.lucene.search.BooleanQuery;
import org.apache.lucene.search.FieldDoc;
import org.apache.lucene.search.IndexSearcher;
import org.apache.lucene.search.Query;
import org.apache.lucene.search.ScoreDoc;
import org.apache.lucene.search.Sort;
import org.apache.lucene.search.SortField;
import org.apache.lucene.search.TopDocs;
import org.apache.lucene.search.TopFieldCollector;
import org.apache.lucene.util.ArrayUtil;
import org.elasticsearch.common.bytes.BytesReference;
import org.elasticsearch.common.lucene.Lucene;
import org.elasticsearch.common.lucene.index.SequentialStoredFieldsLeafReader;
import org.elasticsearch.common.lucene.search.Queries;
import org.elasticsearch.core.internal.io.IOUtils;
import org.elasticsearch.index.engine.CombinedDocValues;
import org.elasticsearch.index.engine.Engine;
import org.elasticsearch.index.engine.MissingHistoryOperationsException;
import org.elasticsearch.index.fieldvisitor.FieldsVisitor;
import org.elasticsearch.index.translog.Translog;
import org.elasticsearch.transport.Transports;

final class LuceneChangesSnapshot
implements Translog.Snapshot {
    static final int DEFAULT_BATCH_SIZE = 1024;
    private final int searchBatchSize;
    private final long fromSeqNo;
    private final long toSeqNo;
    private long lastSeenSeqNo;
    private int skippedOperations;
    private final boolean requiredFullRange;
    private final boolean singleConsumer;
    private final IndexSearcher indexSearcher;
    private int docIndex = 0;
    private final boolean accessStats;
    private final int totalHits;
    private ScoreDoc[] scoreDocs;
    private final ParallelArray parallelArray;
    private final Closeable onClose;
    private int storedFieldsReaderOrd = -1;
    private StoredFieldsReader storedFieldsReader = null;
    private final Thread creationThread;

    LuceneChangesSnapshot(Engine.Searcher engineSearcher, int searchBatchSize, long fromSeqNo, long toSeqNo, boolean requiredFullRange, boolean singleConsumer, boolean accessStats) throws IOException {
        if (fromSeqNo < 0L || toSeqNo < 0L || fromSeqNo > toSeqNo) {
            throw new IllegalArgumentException("Invalid range; from_seqno [" + fromSeqNo + "], to_seqno [" + toSeqNo + "]");
        }
        if (searchBatchSize <= 0) {
            throw new IllegalArgumentException("Search_batch_size must be positive [" + searchBatchSize + "]");
        }
        AtomicBoolean closed = new AtomicBoolean();
        this.onClose = () -> {
            if (closed.compareAndSet(false, true)) {
                IOUtils.close((Closeable)engineSearcher);
            }
        };
        long requestingSize = toSeqNo - fromSeqNo == Long.MAX_VALUE ? Long.MAX_VALUE : toSeqNo - fromSeqNo + 1L;
        this.creationThread = Thread.currentThread();
        this.searchBatchSize = requestingSize < (long)searchBatchSize ? Math.toIntExact(requestingSize) : searchBatchSize;
        this.fromSeqNo = fromSeqNo;
        this.toSeqNo = toSeqNo;
        this.lastSeenSeqNo = fromSeqNo - 1L;
        this.requiredFullRange = requiredFullRange;
        this.singleConsumer = singleConsumer;
        this.indexSearcher = LuceneChangesSnapshot.newIndexSearcher(engineSearcher);
        this.indexSearcher.setQueryCache(null);
        this.accessStats = accessStats;
        this.parallelArray = new ParallelArray(this.searchBatchSize);
        TopDocs topDocs = this.searchOperations(null, accessStats);
        this.totalHits = Math.toIntExact(topDocs.totalHits.value);
        this.scoreDocs = topDocs.scoreDocs;
        this.fillParallelArray(this.scoreDocs, this.parallelArray);
    }

    @Override
    public void close() throws IOException {
        assert (this.assertAccessingThread());
        this.onClose.close();
    }

    @Override
    public int totalOperations() {
        assert (this.assertAccessingThread());
        if (!this.accessStats) {
            throw new IllegalStateException("Access stats of a snapshot created with [access_stats] is false");
        }
        return this.totalHits;
    }

    @Override
    public int skippedOperations() {
        assert (this.assertAccessingThread());
        return this.skippedOperations;
    }

    @Override
    public Translog.Operation next() throws IOException {
        assert (this.assertAccessingThread());
        Translog.Operation op = null;
        int idx = this.nextDocIndex();
        while (idx != -1 && (op = this.readDocAsOp(idx)) == null) {
            idx = this.nextDocIndex();
        }
        if (this.requiredFullRange) {
            this.rangeCheck(op);
        }
        if (op != null) {
            this.lastSeenSeqNo = op.seqNo();
        }
        return op;
    }

    private boolean assertAccessingThread() {
        assert (!this.singleConsumer || this.creationThread == Thread.currentThread()) : "created by [" + this.creationThread + "] != current thread [" + Thread.currentThread() + "]";
        assert (Transports.assertNotTransportThread("reading changes snapshot may involve slow IO"));
        return true;
    }

    private void rangeCheck(Translog.Operation op) {
        if (op == null) {
            if (this.lastSeenSeqNo < this.toSeqNo) {
                throw new MissingHistoryOperationsException("Not all operations between from_seqno [" + this.fromSeqNo + "] and to_seqno [" + this.toSeqNo + "] found; prematurely terminated last_seen_seqno [" + this.lastSeenSeqNo + "]");
            }
        } else {
            long expectedSeqNo = this.lastSeenSeqNo + 1L;
            if (op.seqNo() != expectedSeqNo) {
                throw new MissingHistoryOperationsException("Not all operations between from_seqno [" + this.fromSeqNo + "] and to_seqno [" + this.toSeqNo + "] found; expected seqno [" + expectedSeqNo + "]; found [" + op + "]");
            }
        }
    }

    private int nextDocIndex() throws IOException {
        if (this.docIndex == this.scoreDocs.length && this.docIndex > 0) {
            ScoreDoc prev = this.scoreDocs[this.scoreDocs.length - 1];
            this.scoreDocs = this.searchOperations((FieldDoc)((FieldDoc)prev), (boolean)false).scoreDocs;
            this.fillParallelArray(this.scoreDocs, this.parallelArray);
            this.docIndex = 0;
        }
        if (this.docIndex < this.scoreDocs.length) {
            int idx = this.docIndex++;
            return idx;
        }
        return -1;
    }

    private void fillParallelArray(ScoreDoc[] scoreDocs, ParallelArray parallelArray) throws IOException {
        if (scoreDocs.length > 0) {
            for (int i2 = 0; i2 < scoreDocs.length; ++i2) {
                scoreDocs[i2].shardIndex = i2;
            }
            boolean bl = parallelArray.useSequentialStoredFieldsReader = this.singleConsumer && scoreDocs.length >= 10 && LuceneChangesSnapshot.hasSequentialAccess(scoreDocs);
            if (!parallelArray.useSequentialStoredFieldsReader) {
                this.storedFieldsReaderOrd = -1;
                this.storedFieldsReader = null;
            }
            if (!parallelArray.useSequentialStoredFieldsReader) {
                ArrayUtil.introSort(scoreDocs, Comparator.comparingInt(i -> i.doc));
            }
            int docBase = -1;
            int maxDoc = 0;
            List<LeafReaderContext> leaves = this.indexSearcher.getIndexReader().leaves();
            int readerIndex = 0;
            CombinedDocValues combinedDocValues = null;
            LeafReaderContext leaf = null;
            for (ScoreDoc scoreDoc : scoreDocs) {
                if (scoreDoc.doc >= docBase + maxDoc) {
                    while (scoreDoc.doc >= (docBase = leaf.docBase) + (maxDoc = (leaf = leaves.get(readerIndex++)).reader().maxDoc())) {
                    }
                    combinedDocValues = new CombinedDocValues(leaf.reader());
                }
                int segmentDocID = scoreDoc.doc - docBase;
                int index = scoreDoc.shardIndex;
                parallelArray.leafReaderContexts[index] = leaf;
                parallelArray.seqNo[index] = combinedDocValues.docSeqNo(segmentDocID);
                parallelArray.primaryTerm[index] = combinedDocValues.docPrimaryTerm(segmentDocID);
                parallelArray.version[index] = combinedDocValues.docVersion(segmentDocID);
                parallelArray.isTombStone[index] = combinedDocValues.isTombstone(segmentDocID);
                parallelArray.hasRecoverySource[index] = combinedDocValues.hasRecoverySource(segmentDocID);
            }
            if (!parallelArray.useSequentialStoredFieldsReader) {
                ArrayUtil.introSort(scoreDocs, Comparator.comparingInt(i -> i.shardIndex));
            }
        }
    }

    private static boolean hasSequentialAccess(ScoreDoc[] scoreDocs) {
        for (int i = 0; i < scoreDocs.length - 1; ++i) {
            if (scoreDocs[i].doc + 1 == scoreDocs[i + 1].doc) continue;
            return false;
        }
        return true;
    }

    private static IndexSearcher newIndexSearcher(Engine.Searcher engineSearcher) throws IOException {
        return new IndexSearcher(Lucene.wrapAllDocsLive(engineSearcher.getDirectoryReader()));
    }

    private static Query rangeQuery(long fromSeqNo, long toSeqNo) {
        return new BooleanQuery.Builder().add(LongPoint.newRangeQuery("_seq_no", fromSeqNo, toSeqNo), BooleanClause.Occur.MUST).add(Queries.newNonNestedFilter(), BooleanClause.Occur.MUST).build();
    }

    static int countOperations(Engine.Searcher engineSearcher, long fromSeqNo, long toSeqNo) throws IOException {
        if (fromSeqNo < 0L || toSeqNo < 0L || fromSeqNo > toSeqNo) {
            throw new IllegalArgumentException("Invalid range; from_seqno [" + fromSeqNo + "], to_seqno [" + toSeqNo + "]");
        }
        return LuceneChangesSnapshot.newIndexSearcher(engineSearcher).count(LuceneChangesSnapshot.rangeQuery(fromSeqNo, toSeqNo));
    }

    private TopDocs searchOperations(FieldDoc after, boolean accurateTotalHits) throws IOException {
        Query rangeQuery = LuceneChangesSnapshot.rangeQuery(Math.max(this.fromSeqNo, this.lastSeenSeqNo), this.toSeqNo);
        assert (!accurateTotalHits || after == null) : "accurate total hits is required by the first batch only";
        SortField sortBySeqNo = new SortField("_seq_no", SortField.Type.LONG);
        TopFieldCollector collector = TopFieldCollector.create(new Sort(sortBySeqNo), this.searchBatchSize, after, accurateTotalHits ? Integer.MAX_VALUE : 0);
        this.indexSearcher.search(rangeQuery, collector);
        return collector.topDocs();
    }

    private Translog.Operation readDocAsOp(int docIndex) throws IOException {
        Translog.Operation op;
        LeafReaderContext leaf = this.parallelArray.leafReaderContexts[docIndex];
        int segmentDocID = this.scoreDocs[docIndex].doc - leaf.docBase;
        long primaryTerm = this.parallelArray.primaryTerm[docIndex];
        assert (primaryTerm > 0L) : "nested child document must be excluded";
        long seqNo = this.parallelArray.seqNo[docIndex];
        if (seqNo == this.lastSeenSeqNo) {
            ++this.skippedOperations;
            return null;
        }
        long version = this.parallelArray.version[docIndex];
        String sourceField = this.parallelArray.hasRecoverySource[docIndex] ? "_recovery_source" : "_source";
        FieldsVisitor fields = new FieldsVisitor(true, sourceField);
        if (this.parallelArray.useSequentialStoredFieldsReader && this.storedFieldsReaderOrd != leaf.ord) {
            if (leaf.reader() instanceof SequentialStoredFieldsLeafReader) {
                this.storedFieldsReader = ((SequentialStoredFieldsLeafReader)leaf.reader()).getSequentialStoredFieldsReader();
                this.storedFieldsReaderOrd = leaf.ord;
            } else {
                this.storedFieldsReader = null;
                this.storedFieldsReaderOrd = -1;
            }
        }
        if (this.storedFieldsReader != null) {
            assert (this.singleConsumer) : "Sequential access optimization must not be enabled for multiple consumers";
            assert (this.parallelArray.useSequentialStoredFieldsReader);
            assert (this.storedFieldsReaderOrd == leaf.ord) : this.storedFieldsReaderOrd + " != " + leaf.ord;
            this.storedFieldsReader.visitDocument(segmentDocID, fields);
        } else {
            leaf.reader().document(segmentDocID, fields);
        }
        boolean isTombstone = this.parallelArray.isTombStone[docIndex];
        if (isTombstone && fields.id() == null) {
            op = new Translog.NoOp(seqNo, primaryTerm, fields.source().utf8ToString());
            assert (version == 1L) : "Noop tombstone should have version 1L; actual version [" + version + "]";
            assert (this.assertDocSoftDeleted(leaf.reader(), segmentDocID)) : "Noop but soft_deletes field is not set [" + op + "]";
        } else {
            String id = fields.id();
            if (isTombstone) {
                op = new Translog.Delete(id, seqNo, primaryTerm, version);
                assert (this.assertDocSoftDeleted(leaf.reader(), segmentDocID)) : "Delete op but soft_deletes field is not set [" + op + "]";
            } else {
                BytesReference source = fields.source();
                if (source == null) {
                    if (this.requiredFullRange) {
                        throw new MissingHistoryOperationsException("source not found for seqno=" + seqNo + " from_seqno=" + this.fromSeqNo + " to_seqno=" + this.toSeqNo);
                    }
                    ++this.skippedOperations;
                    return null;
                }
                long autoGeneratedIdTimestamp = -1L;
                op = new Translog.Index(id, seqNo, primaryTerm, version, source.toBytesRef().bytes, fields.routing(), -1L);
            }
        }
        assert (this.fromSeqNo <= op.seqNo() && op.seqNo() <= this.toSeqNo && this.lastSeenSeqNo < op.seqNo()) : "Unexpected operation; last_seen_seqno [" + this.lastSeenSeqNo + "], from_seqno [" + this.fromSeqNo + "], to_seqno [" + this.toSeqNo + "], op [" + op + "]";
        return op;
    }

    private boolean assertDocSoftDeleted(LeafReader leafReader, int segmentDocId) throws IOException {
        NumericDocValues ndv = leafReader.getNumericDocValues("__soft_deletes");
        if (ndv == null || !ndv.advanceExact(segmentDocId)) {
            throw new IllegalStateException("DocValues for field [__soft_deletes] is not found");
        }
        return ndv.longValue() == 1L;
    }

    boolean useSequentialStoredFieldsReader() {
        return this.storedFieldsReader != null;
    }

    private static final class ParallelArray {
        final LeafReaderContext[] leafReaderContexts;
        final long[] version;
        final long[] seqNo;
        final long[] primaryTerm;
        final boolean[] isTombStone;
        final boolean[] hasRecoverySource;
        boolean useSequentialStoredFieldsReader = false;

        ParallelArray(int size) {
            this.version = new long[size];
            this.seqNo = new long[size];
            this.primaryTerm = new long[size];
            this.isTombStone = new boolean[size];
            this.hasRecoverySource = new boolean[size];
            this.leafReaderContexts = new LeafReaderContext[size];
        }
    }
}

