/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.indices;

import java.util.List;
import java.util.Map;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.stream.Collectors;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.apache.logging.log4j.message.ParameterizedMessage;
import org.elasticsearch.ElasticsearchException;
import org.elasticsearch.ElasticsearchParseException;
import org.elasticsearch.Version;
import org.elasticsearch.action.ActionListener;
import org.elasticsearch.action.admin.indices.mapping.put.PutMappingRequest;
import org.elasticsearch.action.support.GroupedActionListener;
import org.elasticsearch.action.support.master.AcknowledgedResponse;
import org.elasticsearch.client.Client;
import org.elasticsearch.client.OriginSettingClient;
import org.elasticsearch.cluster.ClusterChangedEvent;
import org.elasticsearch.cluster.ClusterState;
import org.elasticsearch.cluster.ClusterStateListener;
import org.elasticsearch.cluster.health.ClusterHealthStatus;
import org.elasticsearch.cluster.health.ClusterIndexHealth;
import org.elasticsearch.cluster.metadata.IndexMetadata;
import org.elasticsearch.cluster.metadata.MappingMetadata;
import org.elasticsearch.cluster.metadata.Metadata;
import org.elasticsearch.cluster.routing.IndexRoutingTable;
import org.elasticsearch.gateway.GatewayService;
import org.elasticsearch.indices.SystemIndexDescriptor;
import org.elasticsearch.indices.SystemIndices;
import org.elasticsearch.xcontent.XContentType;

public class SystemIndexManager
implements ClusterStateListener {
    private static final Logger logger = LogManager.getLogger(SystemIndexManager.class);
    private final SystemIndices systemIndices;
    private final Client client;
    private final AtomicBoolean isUpgradeInProgress;

    public SystemIndexManager(SystemIndices systemIndices, Client client) {
        this.systemIndices = systemIndices;
        this.client = client;
        this.isUpgradeInProgress = new AtomicBoolean(false);
    }

    @Override
    public void clusterChanged(ClusterChangedEvent event) {
        ClusterState state = event.state();
        if (state.blocks().hasGlobalBlock(GatewayService.STATE_NOT_RECOVERED_BLOCK)) {
            logger.debug("Waiting until state has been recovered");
            return;
        }
        if (!state.nodes().isLocalNodeElectedMaster()) {
            return;
        }
        if (state.nodes().getMaxNodeVersion().after(state.nodes().getSmallestNonClientNodeVersion())) {
            logger.debug("Skipping system indices up-to-date check as cluster has mixed versions");
            return;
        }
        if (this.isUpgradeInProgress.compareAndSet(false, true)) {
            List<SystemIndexDescriptor> descriptors = this.getEligibleDescriptors(state.getMetadata()).stream().filter(descriptor -> this.getUpgradeStatus(state, (SystemIndexDescriptor)descriptor) == UpgradeStatus.NEEDS_MAPPINGS_UPDATE).collect(Collectors.toList());
            if (!descriptors.isEmpty()) {
                GroupedActionListener listener = new GroupedActionListener(ActionListener.wrap(() -> this.isUpgradeInProgress.set(false)), descriptors.size());
                descriptors.forEach(descriptor -> this.upgradeIndexMetadata((SystemIndexDescriptor)descriptor, listener));
            } else {
                this.isUpgradeInProgress.set(false);
            }
        } else {
            logger.trace("Update already in progress");
        }
    }

    List<SystemIndexDescriptor> getEligibleDescriptors(Metadata metadata) {
        return this.systemIndices.getSystemIndexDescriptors().stream().filter(SystemIndexDescriptor::isAutomaticallyManaged).filter(d -> metadata.hasIndexAbstraction(d.getPrimaryIndex())).collect(Collectors.toList());
    }

    UpgradeStatus getUpgradeStatus(ClusterState clusterState, SystemIndexDescriptor descriptor) {
        State indexState = this.calculateIndexState(clusterState, descriptor);
        String indexDescription = "[" + descriptor.getPrimaryIndex() + "] (alias [" + descriptor.getAliasName() + "])";
        if (indexState == null) {
            logger.debug("Index {} does not exist yet", (Object)indexDescription);
            return UpgradeStatus.UP_TO_DATE;
        }
        if (indexState.indexState == IndexMetadata.State.CLOSE) {
            logger.debug("Index {} is closed. This is likely to prevent some features from functioning correctly", (Object)indexDescription);
            return UpgradeStatus.CLOSED;
        }
        if (indexState.indexHealth == ClusterHealthStatus.RED) {
            logger.debug("Index {} health status is RED, any pending mapping upgrades will wait until this changes", (Object)indexDescription);
            return UpgradeStatus.UNHEALTHY;
        }
        if (!indexState.isIndexUpToDate) {
            logger.debug("Index {} is not on the current version. Features relying on the index will not be available until the index is upgraded", (Object)indexDescription);
            return UpgradeStatus.NEEDS_UPGRADE;
        }
        if (indexState.mappingUpToDate) {
            logger.trace("Index {} is up-to-date, no action required", (Object)indexDescription);
            return UpgradeStatus.UP_TO_DATE;
        }
        logger.info("Index {} mappings are not up-to-date and will be updated", (Object)indexDescription);
        return UpgradeStatus.NEEDS_MAPPINGS_UPDATE;
    }

    private void upgradeIndexMetadata(SystemIndexDescriptor descriptor, final ActionListener<AcknowledgedResponse> listener) {
        final String indexName = descriptor.getPrimaryIndex();
        PutMappingRequest request = new PutMappingRequest(indexName).source(descriptor.getMappings(), XContentType.JSON);
        OriginSettingClient originSettingClient = new OriginSettingClient(this.client, descriptor.getOrigin());
        originSettingClient.admin().indices().putMapping(request, new ActionListener<AcknowledgedResponse>(){

            @Override
            public void onResponse(AcknowledgedResponse response) {
                if (!response.isAcknowledged()) {
                    String message = "Put mapping request for [" + indexName + "] was not acknowledged";
                    logger.error(message);
                    listener.onFailure(new ElasticsearchException(message, new Object[0]));
                } else {
                    listener.onResponse(response);
                }
            }

            @Override
            public void onFailure(Exception e) {
                logger.error("Put mapping request for [" + indexName + "] failed", (Throwable)e);
                listener.onFailure(e);
            }
        });
    }

    State calculateIndexState(ClusterState state, SystemIndexDescriptor descriptor) {
        ClusterHealthStatus indexHealth;
        IndexMetadata indexMetadata = state.metadata().index(descriptor.getPrimaryIndex());
        if (indexMetadata == null) {
            return null;
        }
        boolean isIndexUpToDate = IndexMetadata.INDEX_FORMAT_SETTING.get(indexMetadata.getSettings()).intValue() == descriptor.getIndexFormat();
        boolean isMappingIsUpToDate = this.checkIndexMappingUpToDate(descriptor, indexMetadata);
        String concreteIndexName = indexMetadata.getIndex().getName();
        IndexMetadata.State indexState = indexMetadata.getState();
        if (indexState == IndexMetadata.State.CLOSE) {
            indexHealth = null;
            logger.warn("Index [{}] (alias [{}]) is closed. This is likely to prevent some features from functioning correctly", (Object)concreteIndexName, (Object)descriptor.getAliasName());
        } else {
            IndexRoutingTable routingTable = state.getRoutingTable().index(indexMetadata.getIndex());
            indexHealth = new ClusterIndexHealth(indexMetadata, routingTable).getStatus();
        }
        return new State(indexState, indexHealth, isIndexUpToDate, isMappingIsUpToDate);
    }

    private boolean checkIndexMappingUpToDate(SystemIndexDescriptor descriptor, IndexMetadata indexMetadata) {
        MappingMetadata mappingMetadata = indexMetadata.mapping();
        if (mappingMetadata == null) {
            return false;
        }
        return Version.CURRENT.onOrBefore(this.readMappingVersion(descriptor, mappingMetadata));
    }

    private Version readMappingVersion(SystemIndexDescriptor descriptor, MappingMetadata mappingMetadata) {
        String indexName = descriptor.getPrimaryIndex();
        try {
            String versionString;
            Map meta = (Map)mappingMetadata.sourceAsMap().get("_meta");
            if (meta == null) {
                logger.warn("Missing _meta field in mapping [{}] of index [{}]", (Object)mappingMetadata.type(), (Object)indexName);
                throw new IllegalStateException("Cannot read version string in index " + indexName);
            }
            Object rawVersion = meta.get(descriptor.getVersionMetaKey());
            if (rawVersion instanceof Integer) {
                return Version.V_EMPTY;
            }
            String string = versionString = rawVersion != null ? rawVersion.toString() : null;
            if (versionString == null) {
                logger.warn("No value found in mappings for [_meta.{}]", (Object)descriptor.getVersionMetaKey());
                return Version.V_EMPTY;
            }
            return Version.fromString(versionString);
        }
        catch (ElasticsearchParseException e) {
            logger.error(new ParameterizedMessage("Cannot parse the mapping for index [{}]", (Object)indexName), (Throwable)e);
            throw new ElasticsearchException("Cannot parse the mapping for index [{}]", (Throwable)e, indexName);
        }
        catch (IllegalArgumentException e) {
            logger.error(new ParameterizedMessage("Cannot parse the mapping for index [{}]", (Object)indexName), (Throwable)e);
            throw e;
        }
    }

    static class State {
        final IndexMetadata.State indexState;
        final ClusterHealthStatus indexHealth;
        final boolean isIndexUpToDate;
        final boolean mappingUpToDate;

        State(IndexMetadata.State indexState, ClusterHealthStatus indexHealth, boolean isIndexUpToDate, boolean mappingUpToDate) {
            this.indexState = indexState;
            this.indexHealth = indexHealth;
            this.isIndexUpToDate = isIndexUpToDate;
            this.mappingUpToDate = mappingUpToDate;
        }
    }

    static enum UpgradeStatus {
        CLOSED,
        UNHEALTHY,
        NEEDS_UPGRADE,
        UP_TO_DATE,
        NEEDS_MAPPINGS_UPDATE;

    }
}

