/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.cluster.routing.allocation;

import com.carrotsearch.hppc.ObjectLookupContainer;
import com.carrotsearch.hppc.cursors.ObjectObjectCursor;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicLong;
import java.util.function.LongSupplier;
import java.util.function.Supplier;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import java.util.stream.StreamSupport;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.apache.logging.log4j.message.Message;
import org.apache.logging.log4j.message.ParameterizedMessage;
import org.elasticsearch.action.ActionListener;
import org.elasticsearch.action.support.GroupedActionListener;
import org.elasticsearch.client.Client;
import org.elasticsearch.cluster.ClusterInfo;
import org.elasticsearch.cluster.ClusterState;
import org.elasticsearch.cluster.DiskUsage;
import org.elasticsearch.cluster.block.ClusterBlockLevel;
import org.elasticsearch.cluster.metadata.IndexMetadata;
import org.elasticsearch.cluster.metadata.SingleNodeShutdownMetadata;
import org.elasticsearch.cluster.node.DiscoveryNode;
import org.elasticsearch.cluster.routing.RerouteService;
import org.elasticsearch.cluster.routing.RoutingNode;
import org.elasticsearch.cluster.routing.RoutingNodes;
import org.elasticsearch.cluster.routing.ShardRouting;
import org.elasticsearch.cluster.routing.allocation.DiskThresholdSettings;
import org.elasticsearch.cluster.routing.allocation.decider.DiskThresholdDecider;
import org.elasticsearch.common.Priority;
import org.elasticsearch.common.Strings;
import org.elasticsearch.common.collect.ImmutableOpenMap;
import org.elasticsearch.common.settings.ClusterSettings;
import org.elasticsearch.common.settings.Settings;
import org.elasticsearch.common.unit.ByteSizeValue;
import org.elasticsearch.common.util.set.Sets;
import org.elasticsearch.index.Index;

public class DiskThresholdMonitor {
    private static final Logger logger = LogManager.getLogger(DiskThresholdMonitor.class);
    private static final Settings READ_ONLY_ALLOW_DELETE_SETTINGS = Settings.builder().put(IndexMetadata.SETTING_READ_ONLY_ALLOW_DELETE, Boolean.TRUE.toString()).build();
    private static final Settings NOT_READ_ONLY_ALLOW_DELETE_SETTINGS = Settings.builder().putNull(IndexMetadata.SETTING_READ_ONLY_ALLOW_DELETE).build();
    private final DiskThresholdSettings diskThresholdSettings;
    private final Client client;
    private final Supplier<ClusterState> clusterStateSupplier;
    private final LongSupplier currentTimeMillisSupplier;
    private final RerouteService rerouteService;
    private final AtomicLong lastRunTimeMillis = new AtomicLong(Long.MIN_VALUE);
    private final AtomicBoolean checkInProgress = new AtomicBoolean();
    private final Set<String> nodesOverLowThreshold = Sets.newConcurrentHashSet();
    private final Set<String> nodesOverHighThreshold = Sets.newConcurrentHashSet();
    private final Set<String> nodesOverHighThresholdAndRelocating = Sets.newConcurrentHashSet();

    public DiskThresholdMonitor(Settings settings, Supplier<ClusterState> clusterStateSupplier, ClusterSettings clusterSettings, Client client, LongSupplier currentTimeMillisSupplier, RerouteService rerouteService) {
        this.clusterStateSupplier = clusterStateSupplier;
        this.currentTimeMillisSupplier = currentTimeMillisSupplier;
        this.rerouteService = rerouteService;
        this.diskThresholdSettings = new DiskThresholdSettings(settings, clusterSettings);
        this.client = client;
    }

    private void checkFinished() {
        boolean checkFinished = this.checkInProgress.compareAndSet(true, false);
        assert (checkFinished);
        logger.trace("checkFinished");
    }

    public void onNewInfo(ClusterInfo info) {
        if (!this.checkInProgress.compareAndSet(false, true)) {
            logger.info("skipping monitor as a check is already in progress");
            return;
        }
        ImmutableOpenMap<String, DiskUsage> usages = info.getNodeLeastAvailableDiskUsages();
        if (usages == null) {
            logger.trace("skipping monitor as no disk usage information is available");
            this.checkFinished();
            return;
        }
        logger.trace("processing new cluster info");
        boolean reroute = false;
        String explanation = "";
        long currentTimeMillis = this.currentTimeMillisSupplier.getAsLong();
        ObjectLookupContainer<String> nodes = usages.keys();
        DiskThresholdMonitor.cleanUpRemovedNodes(nodes, this.nodesOverLowThreshold);
        DiskThresholdMonitor.cleanUpRemovedNodes(nodes, this.nodesOverHighThreshold);
        DiskThresholdMonitor.cleanUpRemovedNodes(nodes, this.nodesOverHighThresholdAndRelocating);
        ClusterState state = this.clusterStateSupplier.get();
        HashSet<String> indicesToMarkReadOnly = new HashSet<String>();
        RoutingNodes routingNodes = state.getRoutingNodes();
        HashSet<String> indicesNotToAutoRelease = new HashSet<String>();
        this.markNodesMissingUsageIneligibleForRelease(routingNodes, usages, indicesNotToAutoRelease);
        ArrayList<DiskUsage> usagesOverHighThreshold = new ArrayList<DiskUsage>();
        for (ObjectObjectCursor<String, DiskUsage> objectObjectCursor : usages) {
            String indexName;
            String node = (String)objectObjectCursor.key;
            DiskUsage usage = (DiskUsage)objectObjectCursor.value;
            RoutingNode routingNode = routingNodes.node(node);
            if (this.isDedicatedFrozenNode(routingNode)) {
                ByteSizeValue total = ByteSizeValue.ofBytes(usage.getTotalBytes());
                long frozenFloodStageThreshold = this.diskThresholdSettings.getFreeBytesThresholdFrozenFloodStage(total).getBytes();
                if (usage.getFreeBytes() >= frozenFloodStageThreshold) continue;
                logger.warn("flood stage disk watermark [{}] exceeded on {}", (Object)this.diskThresholdSettings.describeFrozenFloodStageThreshold(total), (Object)usage);
                continue;
            }
            if (usage.getFreeBytes() < this.diskThresholdSettings.getFreeBytesThresholdFloodStage().getBytes() || usage.getFreeDiskAsPercentage() < this.diskThresholdSettings.getFreeDiskThresholdFloodStage()) {
                this.nodesOverLowThreshold.add(node);
                this.nodesOverHighThreshold.add(node);
                this.nodesOverHighThresholdAndRelocating.remove(node);
                if (routingNode != null) {
                    for (ShardRouting routing : routingNode) {
                        indexName = routing.index().getName();
                        indicesToMarkReadOnly.add(indexName);
                        indicesNotToAutoRelease.add(indexName);
                    }
                }
                logger.warn("flood stage disk watermark [{}] exceeded on {}, all indices on this node will be marked read-only", (Object)this.diskThresholdSettings.describeFloodStageThreshold(), (Object)usage);
                continue;
            }
            if ((usage.getFreeBytes() < this.diskThresholdSettings.getFreeBytesThresholdHigh().getBytes() || usage.getFreeDiskAsPercentage() < this.diskThresholdSettings.getFreeDiskThresholdHigh()) && routingNode != null) {
                for (ShardRouting routing : routingNode) {
                    indexName = routing.index().getName();
                    indicesNotToAutoRelease.add(indexName);
                }
            }
            long reservedSpace = info.getReservedSpace(usage.getNodeId(), usage.getPath()).getTotal();
            DiskUsage usageWithReservedSpace = new DiskUsage(usage.getNodeId(), usage.getNodeName(), usage.getPath(), usage.getTotalBytes(), Math.max(0L, usage.getFreeBytes() - reservedSpace));
            if (usageWithReservedSpace.getFreeBytes() < this.diskThresholdSettings.getFreeBytesThresholdHigh().getBytes() || usageWithReservedSpace.getFreeDiskAsPercentage() < this.diskThresholdSettings.getFreeDiskThresholdHigh()) {
                this.nodesOverLowThreshold.add(node);
                this.nodesOverHighThreshold.add(node);
                if (this.lastRunTimeMillis.get() <= currentTimeMillis - this.diskThresholdSettings.getRerouteInterval().millis()) {
                    reroute = true;
                    explanation = "high disk watermark exceeded on one or more nodes";
                    usagesOverHighThreshold.add(usage);
                    continue;
                }
                logger.debug("high disk watermark exceeded on {} but an automatic reroute has occurred in the last [{}], skipping reroute", (Object)node, (Object)this.diskThresholdSettings.getRerouteInterval());
                continue;
            }
            if (usageWithReservedSpace.getFreeBytes() < this.diskThresholdSettings.getFreeBytesThresholdLow().getBytes() || usageWithReservedSpace.getFreeDiskAsPercentage() < this.diskThresholdSettings.getFreeDiskThresholdLow()) {
                this.nodesOverHighThresholdAndRelocating.remove(node);
                boolean wasUnderLowThreshold = this.nodesOverLowThreshold.add(node);
                boolean wasOverHighThreshold = this.nodesOverHighThreshold.remove(node);
                assert (!(wasUnderLowThreshold && wasOverHighThreshold));
                if (wasUnderLowThreshold) {
                    logger.info("low disk watermark [{}] exceeded on {}, replicas will not be assigned to this node", (Object)this.diskThresholdSettings.describeLowThreshold(), (Object)usage);
                    continue;
                }
                if (!wasOverHighThreshold) continue;
                logger.info("high disk watermark [{}] no longer exceeded on {}, but low disk watermark [{}] is still exceeded", (Object)this.diskThresholdSettings.describeHighThreshold(), (Object)usage, (Object)this.diskThresholdSettings.describeLowThreshold());
                continue;
            }
            this.nodesOverHighThresholdAndRelocating.remove(node);
            if (!this.nodesOverLowThreshold.contains(node)) continue;
            if (this.lastRunTimeMillis.get() <= currentTimeMillis - this.diskThresholdSettings.getRerouteInterval().millis()) {
                reroute = true;
                explanation = "one or more nodes has gone under the high or low watermark";
                this.nodesOverLowThreshold.remove(node);
                this.nodesOverHighThreshold.remove(node);
                logger.info("low disk watermark [{}] no longer exceeded on {}", (Object)this.diskThresholdSettings.describeLowThreshold(), (Object)usage);
                continue;
            }
            logger.debug("{} has gone below a disk threshold, but an automatic reroute has occurred in the last [{}], skipping reroute", (Object)node, (Object)this.diskThresholdSettings.getRerouteInterval());
        }
        GroupedActionListener<Void> listener = new GroupedActionListener<Void>(ActionListener.wrap(this::checkFinished), 3);
        if (reroute) {
            logger.debug("rerouting shards: [{}]", (Object)explanation);
            this.rerouteService.reroute("disk threshold monitor", Priority.HIGH, ActionListener.wrap(reroutedClusterState -> {
                for (DiskUsage diskUsage : usagesOverHighThreshold) {
                    DiskUsage usageIncludingRelocations;
                    long relocatingShardsSize;
                    RoutingNode routingNode = reroutedClusterState.getRoutingNodes().node(diskUsage.getNodeId());
                    if (routingNode != null) {
                        relocatingShardsSize = this.sizeOfRelocatingShards(routingNode, diskUsage, info, (ClusterState)reroutedClusterState);
                        usageIncludingRelocations = new DiskUsage(diskUsage.getNodeId(), diskUsage.getNodeName(), diskUsage.getPath(), diskUsage.getTotalBytes(), diskUsage.getFreeBytes() - relocatingShardsSize);
                    } else {
                        usageIncludingRelocations = diskUsage;
                        relocatingShardsSize = 0L;
                    }
                    if (usageIncludingRelocations.getFreeBytes() < this.diskThresholdSettings.getFreeBytesThresholdHigh().getBytes() || usageIncludingRelocations.getFreeDiskAsPercentage() < this.diskThresholdSettings.getFreeDiskThresholdHigh()) {
                        this.nodesOverHighThresholdAndRelocating.remove(diskUsage.getNodeId());
                        logger.warn("high disk watermark [{}] exceeded on {}, shards will be relocated away from this node; currently relocating away shards totalling [{}] bytes; the node is expected to continue to exceed the high disk watermark when these relocations are complete", (Object)this.diskThresholdSettings.describeHighThreshold(), (Object)diskUsage, (Object)(-relocatingShardsSize));
                        continue;
                    }
                    if (this.nodesOverHighThresholdAndRelocating.add(diskUsage.getNodeId())) {
                        logger.info("high disk watermark [{}] exceeded on {}, shards will be relocated away from this node; currently relocating away shards totalling [{}] bytes; the node is expected to be below the high disk watermark when these relocations are complete", (Object)this.diskThresholdSettings.describeHighThreshold(), (Object)diskUsage, (Object)(-relocatingShardsSize));
                        continue;
                    }
                    logger.debug("high disk watermark [{}] exceeded on {}, shards will be relocated away from this node; currently relocating away shards totalling [{}] bytes", (Object)this.diskThresholdSettings.describeHighThreshold(), (Object)diskUsage, (Object)(-relocatingShardsSize));
                }
                this.setLastRunTimeMillis();
                listener.onResponse(null);
            }, e -> {
                logger.debug("reroute failed", (Throwable)e);
                this.setLastRunTimeMillis();
                listener.onFailure((Exception)e);
            }));
        } else {
            logger.trace("no reroute required");
            listener.onResponse(null);
        }
        Map<String, String> map = StreamSupport.stream(state.getRoutingNodes().spliterator(), false).collect(Collectors.toMap(rn -> rn.node().getName(), RoutingNode::nodeId, (s1, s2) -> s2));
        Set nodesIdsPartOfReplacement = state.metadata().nodeShutdowns().values().stream().filter(meta -> meta.getType() == SingleNodeShutdownMetadata.Type.REPLACE).flatMap(meta -> Stream.of(meta.getNodeId(), (String)nodeNameToId.get(meta.getTargetNodeName()))).collect(Collectors.toSet());
        Set indicesOnReplaceSourceOrTarget = nodesIdsPartOfReplacement.stream().flatMap(nodeId -> state.getRoutingNodes().node((String)nodeId).copyShards().stream().map(ShardRouting::index).map(Index::getName)).collect(Collectors.toSet());
        Set<String> indicesToAutoRelease = state.routingTable().indicesRouting().keySet().stream().filter(index -> !indicesNotToAutoRelease.contains(index)).filter(index -> state.getBlocks().hasIndexBlock((String)index, IndexMetadata.INDEX_READ_ONLY_ALLOW_DELETE_BLOCK)).filter(index -> !indicesOnReplaceSourceOrTarget.contains(index)).collect(Collectors.toSet());
        if (!indicesToAutoRelease.isEmpty()) {
            logger.info("releasing read-only block on indices " + indicesToAutoRelease + " since they are now allocated to nodes with sufficient disk space");
            this.updateIndicesReadOnly(indicesToAutoRelease, listener, false);
        } else {
            logger.trace("no auto-release required");
            listener.onResponse(null);
        }
        indicesToMarkReadOnly.removeIf(index -> state.getBlocks().indexBlocked(ClusterBlockLevel.WRITE, (String)index));
        logger.trace("marking indices as read-only: [{}]", indicesToMarkReadOnly);
        if (!indicesToMarkReadOnly.isEmpty()) {
            this.updateIndicesReadOnly(indicesToMarkReadOnly, listener, true);
        } else {
            listener.onResponse(null);
        }
    }

    long sizeOfRelocatingShards(RoutingNode routingNode, DiskUsage diskUsage, ClusterInfo info, ClusterState reroutedClusterState) {
        return DiskThresholdDecider.sizeOfRelocatingShards(routingNode, true, diskUsage.getPath(), info, reroutedClusterState.metadata(), reroutedClusterState.routingTable());
    }

    private void markNodesMissingUsageIneligibleForRelease(RoutingNodes routingNodes, ImmutableOpenMap<String, DiskUsage> usages, Set<String> indicesToMarkIneligibleForAutoRelease) {
        for (RoutingNode routingNode : routingNodes) {
            if (usages.containsKey(routingNode.nodeId())) continue;
            for (ShardRouting routing : routingNode) {
                String indexName = routing.index().getName();
                indicesToMarkIneligibleForAutoRelease.add(indexName);
            }
        }
    }

    private void setLastRunTimeMillis() {
        this.lastRunTimeMillis.getAndUpdate(l -> Math.max(l, this.currentTimeMillisSupplier.getAsLong()));
    }

    protected void updateIndicesReadOnly(Set<String> indicesToUpdate, ActionListener<Void> listener, boolean readOnly) {
        ActionListener wrappedListener = ActionListener.wrap(r -> {
            this.setLastRunTimeMillis();
            listener.onResponse((Void)r);
        }, e -> {
            logger.debug((Message)new ParameterizedMessage("setting indices [{}] read-only failed", (Object)readOnly), (Throwable)e);
            this.setLastRunTimeMillis();
            listener.onFailure((Exception)e);
        });
        Settings readOnlySettings = readOnly ? READ_ONLY_ALLOW_DELETE_SETTINGS : NOT_READ_ONLY_ALLOW_DELETE_SETTINGS;
        this.client.admin().indices().prepareUpdateSettings(indicesToUpdate.toArray(Strings.EMPTY_ARRAY)).setSettings(readOnlySettings).execute(wrappedListener.map(r -> null));
    }

    private static void cleanUpRemovedNodes(ObjectLookupContainer<String> nodesToKeep, Set<String> nodesToCleanUp) {
        for (String node : nodesToCleanUp) {
            if (nodesToKeep.contains((Object)node)) continue;
            nodesToCleanUp.remove(node);
        }
    }

    private boolean isDedicatedFrozenNode(RoutingNode routingNode) {
        if (routingNode == null) {
            return false;
        }
        DiscoveryNode node = routingNode.node();
        return node.isDedicatedFrozenNode();
    }
}

