/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.common.settings;

import java.nio.charset.StandardCharsets;
import java.security.NoSuchAlgorithmException;
import java.security.spec.InvalidKeySpecException;
import java.util.Arrays;
import java.util.Base64;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.function.Consumer;
import javax.crypto.SecretKey;
import javax.crypto.SecretKeyFactory;
import javax.crypto.spec.PBEKeySpec;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.elasticsearch.cluster.ClusterState;
import org.elasticsearch.cluster.ClusterStateUpdateTask;
import org.elasticsearch.cluster.LocalNodeMasterListener;
import org.elasticsearch.cluster.metadata.Metadata;
import org.elasticsearch.cluster.service.ClusterService;
import org.elasticsearch.common.Priority;
import org.elasticsearch.common.UUIDs;
import org.elasticsearch.common.hash.MessageDigests;
import org.elasticsearch.common.settings.SecureSetting;
import org.elasticsearch.common.settings.Setting;
import org.elasticsearch.common.settings.Settings;

public final class ConsistentSettingsService {
    private static final Logger logger = LogManager.getLogger(ConsistentSettingsService.class);
    private final Settings settings;
    private final ClusterService clusterService;
    private final Collection<Setting<?>> secureSettingsCollection;
    private final SecretKeyFactory pbkdf2KeyFactory;

    public ConsistentSettingsService(Settings settings, ClusterService clusterService, Collection<Setting<?>> secureSettingsCollection) {
        this.settings = settings;
        this.clusterService = clusterService;
        this.secureSettingsCollection = secureSettingsCollection;
        try {
            this.pbkdf2KeyFactory = SecretKeyFactory.getInstance("PBKDF2WithHmacSHA512");
        }
        catch (NoSuchAlgorithmException e) {
            throw new RuntimeException("The \"PBKDF2WithHmacSHA512\" algorithm is required for consistent secure settings' hashes", e);
        }
    }

    public LocalNodeMasterListener newHashPublisher() {
        Map<String, String> computedHashesOfConsistentSettings = this.computeHashesOfConsistentSecureSettings();
        return new HashesPublisher(computedHashesOfConsistentSettings, this.clusterService);
    }

    public boolean areAllConsistent() {
        ClusterState state = this.clusterService.state();
        Map<String, String> publishedHashesOfConsistentSettings = state.metadata().hashesOfConsistentSettings();
        HashSet<String> publishedSettingKeysToVerify = new HashSet<String>();
        publishedSettingKeysToVerify.addAll(publishedHashesOfConsistentSettings.keySet());
        AtomicBoolean allConsistent = new AtomicBoolean(true);
        this.forEachConcreteSecureSettingDo(concreteSecureSetting -> {
            String publishedSaltAndHash = (String)publishedHashesOfConsistentSettings.get(concreteSecureSetting.getKey());
            byte[] localHash = concreteSecureSetting.getSecretDigest(this.settings);
            if (publishedSaltAndHash == null && localHash == null) {
                logger.debug("no published hash for the consistent secure setting [{}] but it also does NOT exist on the local node", (Object)concreteSecureSetting.getKey());
            } else if (publishedSaltAndHash == null && localHash != null) {
                logger.warn("no published hash for the consistent secure setting [{}] but it exists on the local node", (Object)concreteSecureSetting.getKey());
                if (state.nodes().isLocalNodeElectedMaster()) {
                    throw new IllegalStateException("Master node cannot validate consistent setting. No published hash for [" + concreteSecureSetting.getKey() + "] but setting exists.");
                }
                allConsistent.set(false);
            } else if (publishedSaltAndHash != null && localHash == null) {
                logger.warn("the consistent secure setting [{}] does not exist on the local node but there is a published hash for it", (Object)concreteSecureSetting.getKey());
                allConsistent.set(false);
            } else {
                assert (publishedSaltAndHash != null);
                assert (localHash != null);
                String[] parts = publishedSaltAndHash.split(":");
                if (parts == null || parts.length != 2) {
                    throw new IllegalArgumentException("published hash [" + publishedSaltAndHash + " ] for secure setting [" + concreteSecureSetting.getKey() + "] is invalid");
                }
                String publishedSalt = parts[0];
                String publishedHash = parts[1];
                byte[] computedSaltedHashBytes = this.computeSaltedPBKDF2Hash(localHash, publishedSalt.getBytes(StandardCharsets.UTF_8));
                String computedSaltedHash = new String(Base64.getEncoder().encode(computedSaltedHashBytes), StandardCharsets.UTF_8);
                if (!publishedHash.equals(computedSaltedHash)) {
                    logger.warn("the published hash [{}] of the consistent secure setting [{}] differs from the locally computed one [{}]", (Object)publishedHash, (Object)concreteSecureSetting.getKey(), (Object)computedSaltedHash);
                    if (state.nodes().isLocalNodeElectedMaster()) {
                        throw new IllegalStateException("Master node cannot validate consistent setting. The published hash [" + publishedHash + "] of the consistent secure setting [" + concreteSecureSetting.getKey() + "] differs from the locally computed one [" + computedSaltedHash + "].");
                    }
                    allConsistent.set(false);
                }
            }
            publishedSettingKeysToVerify.remove(concreteSecureSetting.getKey());
        });
        for (String publishedSettingKey : publishedSettingKeysToVerify) {
            for (Setting<?> setting : this.secureSettingsCollection) {
                if (!setting.match(publishedSettingKey)) continue;
                logger.warn("the consistent secure setting [{}] does not exist on the local node but there is a published hash for it", (Object)publishedSettingKey);
                allConsistent.set(false);
            }
        }
        return allConsistent.get();
    }

    private void forEachConcreteSecureSettingDo(Consumer<SecureSetting<?>> secureSettingConsumer) {
        for (Setting<?> setting : this.secureSettingsCollection) {
            assert (setting.isConsistent()) : "[" + setting.getKey() + "] is not a consistent setting";
            if (setting instanceof Setting.AffixSetting) {
                ((Setting.AffixSetting)setting).getAllConcreteSettings(this.settings).forEach(concreteSetting -> {
                    assert (concreteSetting instanceof SecureSetting) : "[" + concreteSetting.getKey() + "] is not a secure setting";
                    secureSettingConsumer.accept((SecureSetting)concreteSetting);
                });
                continue;
            }
            if (setting instanceof SecureSetting) {
                secureSettingConsumer.accept((SecureSetting)setting);
                continue;
            }
            assert (false) : "Unrecognized consistent secure setting [" + setting.getKey() + "]";
        }
    }

    private Map<String, String> computeHashesOfConsistentSecureSettings() {
        HashMap<String, String> hashesBySettingKey = new HashMap<String, String>();
        this.forEachConcreteSecureSettingDo(concreteSecureSetting -> {
            byte[] localHash = concreteSecureSetting.getSecretDigest(this.settings);
            if (localHash != null) {
                String salt = UUIDs.randomBase64UUID();
                byte[] publicHash = this.computeSaltedPBKDF2Hash(localHash, salt.getBytes(StandardCharsets.UTF_8));
                String encodedPublicHash = new String(Base64.getEncoder().encode(publicHash), StandardCharsets.UTF_8);
                hashesBySettingKey.put(concreteSecureSetting.getKey(), salt + ":" + encodedPublicHash);
            }
        });
        return hashesBySettingKey;
    }

    private byte[] computeSaltedPBKDF2Hash(byte[] bytes, byte[] salt) {
        int iterations = 5000;
        int keyLength = 512;
        char[] value = null;
        try {
            value = MessageDigests.toHexCharArray(bytes);
            PBEKeySpec spec = new PBEKeySpec(value, salt, 5000, 512);
            SecretKey key = this.pbkdf2KeyFactory.generateSecret(spec);
            byte[] byArray = key.getEncoded();
            return byArray;
        }
        catch (InvalidKeySpecException e) {
            throw new RuntimeException("Unexpected exception when computing PBKDF2 hash", e);
        }
        finally {
            if (value != null) {
                Arrays.fill(value, '0');
            }
        }
    }

    static final class HashesPublisher
    implements LocalNodeMasterListener {
        final Map<String, String> computedHashesOfConsistentSettings;
        final ClusterService clusterService;

        HashesPublisher(Map<String, String> computedHashesOfConsistentSettings, ClusterService clusterService) {
            this.computedHashesOfConsistentSettings = Map.copyOf(computedHashesOfConsistentSettings);
            this.clusterService = clusterService;
        }

        @Override
        public void onMaster() {
            this.clusterService.submitStateUpdateTask("publish-secure-settings-hashes", new ClusterStateUpdateTask(Priority.URGENT){

                @Override
                public ClusterState execute(ClusterState currentState) {
                    Map<String, String> publishedHashesOfConsistentSettings = currentState.metadata().hashesOfConsistentSettings();
                    if (computedHashesOfConsistentSettings.equals(publishedHashesOfConsistentSettings)) {
                        logger.debug("Nothing to publish. What is already published matches this node's view.");
                        return currentState;
                    }
                    return ClusterState.builder(currentState).metadata(Metadata.builder(currentState.metadata()).hashesOfConsistentSettings(computedHashesOfConsistentSettings)).build();
                }

                @Override
                public void onFailure(String source, Exception e) {
                    logger.error("unable to publish secure settings hashes", (Throwable)e);
                }
            });
        }

        @Override
        public void offMaster() {
            logger.trace("I am no longer master, nothing to do");
        }
    }
}

