"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.Kodo = exports.USER_AGENT = void 0;
const async_lock_1 = __importDefault(require("async-lock"));
const os_1 = __importDefault(require("os"));
const package_json_1 = __importDefault(require("./package.json"));
const form_data_1 = __importDefault(require("form-data"));
const buffer_crc32_1 = __importDefault(require("buffer-crc32"));
const js_md5_1 = __importDefault(require("js-md5"));
const semaphore_promise_1 = require("semaphore-promise");
const region_service_1 = require("./region_service");
const url_1 = require("url");
const urllib_1 = require("urllib");
const js_base64_1 = require("js-base64");
const kodo_auth_1 = require("./kodo-auth");
const kodo_http_client_1 = require("./kodo-http-client");
exports.USER_AGENT = `Qiniu-Kodo-S3-Adapter-NodeJS-SDK/${package_json_1.default.version} (${os_1.default.type()}; ${os_1.default.platform()}; ${os_1.default.arch()}; )/kodo`;
class Kodo {
    constructor(adapterOption) {
        this.adapterOption = adapterOption;
        this.bucketDomainsCache = {};
        this.bucketDomainsCacheLock = new async_lock_1.default();
        let userAgent = exports.USER_AGENT;
        if (adapterOption.appendedUserAgent) {
            userAgent += `/${adapterOption.appendedUserAgent}`;
        }
        this.client = new kodo_http_client_1.KodoHttpClient({
            accessKey: adapterOption.accessKey,
            secretKey: adapterOption.secretKey,
            ucUrl: adapterOption.ucUrl,
            regions: adapterOption.regions,
            appendedUserAgent: adapterOption.appendedUserAgent,
            userAgent: userAgent,
            timeout: [30000, 300000],
            retry: 10,
            retryDelay: 500,
            requestCallback: adapterOption.requestCallback,
            responseCallback: adapterOption.responseCallback,
        });
        this.regionService = new region_service_1.RegionService(adapterOption);
    }
    createBucket(s3RegionId, bucket) {
        return new Promise((resolve, reject) => {
            this.regionService.fromS3IdToKodoRegionId(s3RegionId).then((kodoRegionId) => {
                this.client.call({
                    method: 'POST',
                    serviceName: kodo_http_client_1.ServiceName.Uc,
                    path: `mkbucketv3/${bucket}/region/${kodoRegionId}`,
                }).then(() => {
                    resolve();
                }).catch(reject);
            }).catch(reject);
        });
    }
    deleteBucket(_region, bucket) {
        return new Promise((resolve, reject) => {
            this.client.call({
                method: 'POST',
                serviceName: kodo_http_client_1.ServiceName.Uc,
                bucketName: bucket,
                path: `drop/${bucket}`,
            }).then(() => {
                resolve();
            }).catch(reject);
        });
    }
    getBucketLocation(bucket) {
        return new Promise((resolve, reject) => {
            this.client.call({
                method: 'GET',
                serviceName: kodo_http_client_1.ServiceName.Uc,
                bucketName: bucket,
                path: `bucket/${bucket}`,
                dataType: 'json',
            }).then((response) => {
                const kodoRegionId = response.data.region;
                this.regionService.fromKodoRegionIdToS3Id(kodoRegionId)
                    .then(resolve).catch(reject);
            }).catch(reject);
        });
    }
    listBuckets() {
        return new Promise((resolve, reject) => {
            const bucketsQuery = new url_1.URLSearchParams();
            bucketsQuery.set('shared', 'rd');
            this.client.call({
                method: 'GET',
                serviceName: kodo_http_client_1.ServiceName.Uc,
                path: 'v2/buckets',
                dataType: 'json',
                query: bucketsQuery,
            }).then((response) => {
                const regionsPromises = response.data.map((info) => {
                    return new Promise((resolve) => {
                        this.regionService.fromKodoRegionIdToS3Id(info.region)
                            .then(resolve).catch(() => { resolve(undefined); });
                    });
                });
                Promise.all(regionsPromises).then((regionsInfo) => {
                    const bucketInfos = response.data.map((info, index) => {
                        let grantedPermission = undefined;
                        switch (info.perm) {
                            case 1:
                                grantedPermission = 'readonly';
                                break;
                            case 2:
                                grantedPermission = 'readwrite';
                                break;
                        }
                        return {
                            id: info.id, name: info.tbl,
                            createDate: new Date(info.ctime * 1000),
                            regionId: regionsInfo[index], grantedPermission: grantedPermission,
                        };
                    });
                    resolve(bucketInfos);
                }).catch(reject);
            }).catch(reject);
        });
    }
    listDomains(s3RegionId, bucket) {
        return new Promise((resolve, reject) => {
            const domainsQuery = new url_1.URLSearchParams();
            domainsQuery.set('sourceTypes', 'qiniuBucket');
            domainsQuery.set('sourceQiniuBucket', bucket);
            domainsQuery.set('operatingState', 'success');
            domainsQuery.set('limit', '50');
            const getBucketInfoQuery = new url_1.URLSearchParams();
            getBucketInfoQuery.set('bucket', bucket);
            const bucketDefaultDomainQuery = new url_1.URLSearchParams();
            bucketDefaultDomainQuery.set('bucket', bucket);
            const promises = [
                this.client.call({
                    method: 'GET',
                    serviceName: kodo_http_client_1.ServiceName.Qcdn,
                    path: 'domain',
                    query: domainsQuery,
                    dataType: 'json',
                    s3RegionId: s3RegionId,
                }),
                this.client.call({
                    method: 'POST',
                    serviceName: kodo_http_client_1.ServiceName.Uc,
                    path: 'v2/bucketInfo',
                    query: getBucketInfoQuery,
                    dataType: 'json',
                    s3RegionId: s3RegionId,
                }),
                this.client.call({
                    method: 'GET',
                    serviceName: kodo_http_client_1.ServiceName.Portal,
                    path: 'api/kodov2/domain/default/get',
                    query: bucketDefaultDomainQuery,
                    dataType: 'json',
                    s3RegionId: s3RegionId,
                }),
            ];
            Promise.all(promises).then(([domainResponse, bucketResponse, defaultDomainQuery]) => {
                if (bucketResponse.data.perm && bucketResponse.data.perm > 0) {
                    const result = defaultDomainQuery.data;
                    const domains = [];
                    if (result.domain && result.protocol) {
                        domains.push({
                            name: result.domain, protocol: result.protocol,
                            type: 'normal', private: bucketResponse.data.private != 0,
                        });
                    }
                    resolve(domains);
                }
                else {
                    const domains = domainResponse.data.domains.filter((domain) => {
                        switch (domain.type) {
                            case 'normal':
                            case 'pan':
                            case 'test':
                                return true;
                            default:
                                return false;
                        }
                    }).map((domain) => {
                        return {
                            name: domain.name, protocol: domain.protocol, type: domain.type,
                            private: bucketResponse.data.private != 0,
                        };
                    });
                    resolve(domains);
                }
            }).catch(reject);
        });
    }
    _listDomains(s3RegionId, bucket) {
        return new Promise((resolve, reject) => {
            if (this.bucketDomainsCache[bucket]) {
                resolve(this.bucketDomainsCache[bucket]);
                return;
            }
            this.bucketDomainsCacheLock.acquire(bucket, () => {
                if (this.bucketDomainsCache[bucket]) {
                    return Promise.resolve(this.bucketDomainsCache[bucket]);
                }
                return this.listDomains(s3RegionId, bucket);
            }).then((domains) => {
                this.bucketDomainsCache[bucket] = domains;
                resolve(domains);
            }).catch(reject);
        });
    }
    listBucketIdNames() {
        return new Promise((resolve, reject) => {
            this.client.call({
                method: 'GET',
                serviceName: kodo_http_client_1.ServiceName.Uc,
                path: 'v2/buckets',
                dataType: 'json',
            }).then((response) => {
                const bucketInfos = response.data.map((info) => {
                    return { id: info.id, name: info.tbl };
                });
                resolve(bucketInfos);
            }).catch(reject);
        });
    }
    isExists(s3RegionId, object) {
        return new Promise((resolve, reject) => {
            this.getObjectInfo(s3RegionId, object).then(() => {
                resolve(true);
            }).catch((error) => {
                if (error.message === 'no such file or directory') {
                    resolve(false);
                }
                else {
                    reject(error);
                }
            });
        });
    }
    deleteObject(s3RegionId, object) {
        return new Promise((resolve, reject) => {
            this.client.call({
                method: 'POST',
                serviceName: kodo_http_client_1.ServiceName.Rs,
                path: `delete/${encodeObject(object)}`,
                dataType: 'json',
                s3RegionId: s3RegionId,
                contentType: 'application/x-www-form-urlencoded',
            }).then(() => { resolve(); }).catch(reject);
        });
    }
    putObject(s3RegionId, object, data, originalFileName, header, option) {
        return new Promise((resolve, reject) => {
            const token = kodo_auth_1.makeUploadToken(this.adapterOption.accessKey, this.adapterOption.secretKey, kodo_auth_1.newUploadPolicy(object.bucket, object.key));
            const form = new form_data_1.default();
            form.append('key', object.key);
            form.append('token', token);
            if (header === null || header === void 0 ? void 0 : header.metadata) {
                for (const [metaKey, metaValue] of Object.entries(header.metadata)) {
                    form.append(`x-qn-meta-${metaKey}`, metaValue);
                }
            }
            form.append('crc32', buffer_crc32_1.default.unsigned(data));
            const fileOption = {
                filename: originalFileName,
            };
            if (header === null || header === void 0 ? void 0 : header.contentType) {
                fileOption.contentType = header.contentType;
            }
            form.append('file', data, fileOption);
            this.client.call({
                method: 'POST',
                serviceName: kodo_http_client_1.ServiceName.Up,
                dataType: 'json',
                s3RegionId: s3RegionId,
                contentType: form.getHeaders()['content-type'],
                form: form,
                uploadProgress: option === null || option === void 0 ? void 0 : option.progressCallback,
                uploadThrottle: option === null || option === void 0 ? void 0 : option.throttle,
            }).then(() => { resolve(); }).catch(reject);
        });
    }
    getObject(s3RegionId, object, domain) {
        return new Promise((resolve, reject) => {
            this.getObjectURL(s3RegionId, object, domain).then((url) => {
                let requestInfo = undefined;
                const beginTime = new Date().getTime();
                Kodo.httpClient.request(url.toString(), {
                    method: 'GET',
                    timeout: [30000, 300000],
                    retry: 10,
                    retryDelay: 500,
                    followRedirect: true,
                    beforeRequest: (info) => {
                        requestInfo = {
                            url: url.toString(),
                            method: info.method,
                            headers: info.headers,
                        };
                        if (this.adapterOption.requestCallback) {
                            this.adapterOption.requestCallback(requestInfo);
                        }
                    },
                }).then((response) => {
                    const responseInfo = {
                        request: requestInfo,
                        statusCode: response.status,
                        headers: response.headers,
                        interval: new Date().getTime() - beginTime,
                    };
                    try {
                        if (response.status === 200) {
                            resolve({ data: response.data, header: getObjectHeader(response) });
                        }
                        else {
                            const error = new Error(response.res.statusMessage);
                            responseInfo.error = error;
                            reject(error);
                        }
                    }
                    finally {
                        if (this.adapterOption.responseCallback) {
                            this.adapterOption.responseCallback(responseInfo);
                        }
                    }
                }).catch((err) => {
                    const responseInfo = {
                        request: requestInfo,
                        interval: new Date().getTime() - beginTime,
                        error: err,
                    };
                    if (this.adapterOption.responseCallback) {
                        this.adapterOption.responseCallback(responseInfo);
                    }
                    reject(err);
                });
            }).catch(reject);
        });
    }
    getObjectStream(s3RegionId, object, domain, option) {
        return new Promise((resolve, reject) => {
            this.getObjectURL(s3RegionId, object, domain).then((url) => {
                var _a, _b;
                let requestInfo = undefined;
                const beginTime = new Date().getTime();
                const headers = {};
                if ((option === null || option === void 0 ? void 0 : option.rangeStart) || (option === null || option === void 0 ? void 0 : option.rangeEnd)) {
                    headers['Range'] = `bytes=${(_a = option === null || option === void 0 ? void 0 : option.rangeStart) !== null && _a !== void 0 ? _a : ''}-${(_b = option === null || option === void 0 ? void 0 : option.rangeEnd) !== null && _b !== void 0 ? _b : ''}`;
                }
                Kodo.httpClient.request(url.toString(), {
                    method: 'GET',
                    timeout: [30000, 300000],
                    followRedirect: true,
                    headers: headers,
                    streaming: true,
                    beforeRequest: (info) => {
                        requestInfo = {
                            url: url.toString(),
                            method: info.method,
                            headers: info.headers,
                        };
                        if (this.adapterOption.requestCallback) {
                            this.adapterOption.requestCallback(requestInfo);
                        }
                    },
                }).then((response) => {
                    const responseInfo = {
                        request: requestInfo,
                        statusCode: response.status,
                        headers: response.headers,
                        interval: new Date().getTime() - beginTime,
                    };
                    try {
                        if (response.status === 200 || response.status === 206) {
                            resolve(response.res);
                        }
                        else {
                            const error = new Error(response.res.statusMessage);
                            responseInfo.error = error;
                            reject(error);
                        }
                    }
                    finally {
                        if (this.adapterOption.responseCallback) {
                            this.adapterOption.responseCallback(responseInfo);
                        }
                    }
                }).catch((err) => {
                    const responseInfo = {
                        request: requestInfo,
                        interval: new Date().getTime() - beginTime,
                        error: err,
                    };
                    if (this.adapterOption.responseCallback) {
                        this.adapterOption.responseCallback(responseInfo);
                    }
                    reject(err);
                });
            }).catch(reject);
        });
    }
    getObjectURL(s3RegionId, object, domain, deadline) {
        return new Promise((resolve, reject) => {
            const domainPromise = new Promise((resolve, reject) => {
                if (domain) {
                    resolve(domain);
                    return;
                }
                this._listDomains(s3RegionId, object.bucket).then((domains) => {
                    if (domains.length === 0) {
                        reject(new Error('no domain found'));
                        return;
                    }
                    const domainTypeScope = (domain) => {
                        switch (domain.type) {
                            case 'normal': return 1;
                            case 'pan': return 2;
                            case 'test': return 3;
                        }
                    };
                    domains = domains.sort((domain1, domain2) => domainTypeScope(domain1) - domainTypeScope(domain2));
                    resolve(domains[0]);
                }).catch(reject);
            });
            domainPromise.then((domain) => {
                let url = new url_1.URL(`${domain.protocol}://${domain.name}`);
                url.pathname = object.key;
                if (domain.private) {
                    url = kodo_auth_1.signPrivateURL(this.adapterOption.accessKey, this.adapterOption.secretKey, url, deadline);
                }
                resolve(url);
            }).catch(reject);
        });
    }
    getObjectInfo(s3RegionId, object) {
        return new Promise((resolve, reject) => {
            this.client.call({
                method: 'GET',
                serviceName: kodo_http_client_1.ServiceName.Rs,
                path: `stat/${encodeObject(object)}`,
                dataType: 'json',
                s3RegionId: s3RegionId,
                contentType: 'application/x-www-form-urlencoded',
            }).then((response) => {
                resolve({
                    bucket: object.bucket, key: response.data.key, size: response.data.fsize,
                    lastModified: new Date(response.data.putTime / 10000), storageClass: toStorageClass(response.data.type),
                });
            }).catch(reject);
        });
    }
    getObjectHeader(s3RegionId, object, domain) {
        return new Promise((resolve, reject) => {
            this.getObjectURL(s3RegionId, object, domain).then((url) => {
                let requestInfo = undefined;
                const beginTime = new Date().getTime();
                Kodo.httpClient.request(url.toString(), {
                    method: 'HEAD',
                    timeout: [30000, 300000],
                    retry: 10,
                    retryDelay: 500,
                    followRedirect: true,
                    beforeRequest: (info) => {
                        requestInfo = {
                            url: url.toString(),
                            method: info.method,
                            headers: info.headers,
                        };
                        if (this.adapterOption.requestCallback) {
                            this.adapterOption.requestCallback(requestInfo);
                        }
                    },
                }).then((response) => {
                    const responseInfo = {
                        request: requestInfo,
                        statusCode: response.status,
                        headers: response.headers,
                        interval: new Date().getTime() - beginTime,
                    };
                    try {
                        if (response.status === 200) {
                            resolve(getObjectHeader(response));
                        }
                        else {
                            const error = new Error(response.res.statusMessage);
                            responseInfo.error = error;
                            reject(error);
                        }
                    }
                    finally {
                        if (this.adapterOption.responseCallback) {
                            this.adapterOption.responseCallback(responseInfo);
                        }
                    }
                }).catch((err) => {
                    const responseInfo = {
                        request: requestInfo,
                        interval: new Date().getTime() - beginTime,
                        error: err,
                    };
                    if (this.adapterOption.responseCallback) {
                        this.adapterOption.responseCallback(responseInfo);
                    }
                    reject(err);
                });
            }).catch(reject);
        });
    }
    moveObject(s3RegionId, transferObject) {
        return new Promise((resolve, reject) => {
            this.client.call({
                method: 'POST',
                serviceName: kodo_http_client_1.ServiceName.Rs,
                path: `move/${encodeObject(transferObject.from)}/${encodeObject(transferObject.to)}/force/true`,
                dataType: 'json',
                s3RegionId: s3RegionId,
                contentType: 'application/x-www-form-urlencoded',
            }).then(() => { resolve(); }).catch(reject);
        });
    }
    copyObject(s3RegionId, transferObject) {
        return new Promise((resolve, reject) => {
            this.client.call({
                method: 'POST',
                serviceName: kodo_http_client_1.ServiceName.Rs,
                path: `copy/${encodeObject(transferObject.from)}/${encodeObject(transferObject.to)}/force/true`,
                dataType: 'json',
                s3RegionId: s3RegionId,
                contentType: 'application/x-www-form-urlencoded',
            }).then(() => { resolve(); }).catch(reject);
        });
    }
    moveObjects(s3RegionId, transferObjects, callback) {
        return this.moveOrCopyObjects('move', 100, s3RegionId, transferObjects, callback);
    }
    copyObjects(s3RegionId, transferObjects, callback) {
        return this.moveOrCopyObjects('copy', 100, s3RegionId, transferObjects, callback);
    }
    moveOrCopyObjects(op, batchCount, s3RegionId, transferObjects, callback) {
        const semaphore = new semaphore_promise_1.Semaphore(5);
        const transferObjectsBatches = [];
        while (transferObjects.length >= batchCount) {
            const batch = transferObjects.splice(0, batchCount);
            transferObjectsBatches.push(batch);
        }
        if (transferObjects.length > 0) {
            transferObjectsBatches.push(transferObjects);
        }
        let counter = 0;
        const promises = transferObjectsBatches.map((batch) => {
            const firstIndexInCurrentBatch = counter;
            counter += batch.length;
            return new Promise((resolve, reject) => {
                const params = new url_1.URLSearchParams();
                for (const transferObject of batch) {
                    params.append('op', `/${op}/${encodeObject(transferObject.from)}/${encodeObject(transferObject.to)}/force/true`);
                }
                semaphore.acquire().then((release) => {
                    this.client.call({
                        method: 'POST',
                        serviceName: kodo_http_client_1.ServiceName.Rs,
                        path: 'batch',
                        dataType: 'json',
                        s3RegionId: s3RegionId,
                        contentType: 'application/x-www-form-urlencoded',
                        data: params.toString(),
                    }).then((response) => {
                        let aborted = false;
                        const results = response.data.map((item, index) => {
                            var _a, _b;
                            const currentIndex = firstIndexInCurrentBatch + index;
                            const result = { bucket: batch[index].from.bucket, key: batch[index].from.key };
                            if ((_a = item === null || item === void 0 ? void 0 : item.data) === null || _a === void 0 ? void 0 : _a.error) {
                                const error = new Error((_b = item === null || item === void 0 ? void 0 : item.data) === null || _b === void 0 ? void 0 : _b.error);
                                if (callback && callback(currentIndex, error) === false) {
                                    aborted = true;
                                }
                                result.error = error;
                            }
                            else if (callback && callback(currentIndex) === false) {
                                aborted = true;
                            }
                            return result;
                        });
                        if (aborted) {
                            reject(new Error('aborted'));
                        }
                        else {
                            resolve(results);
                        }
                    }).catch((error) => {
                        let aborted = false;
                        const results = batch.map((transferObject, index) => {
                            const currentIndex = firstIndexInCurrentBatch + index;
                            if (callback && callback(currentIndex, error) === false) {
                                aborted = true;
                            }
                            return { bucket: transferObject.from.bucket, key: transferObject.from.key, error: error };
                        });
                        if (aborted) {
                            reject(new Error('aborted'));
                        }
                        else {
                            resolve(results);
                        }
                    }).finally(() => {
                        release();
                    });
                });
            });
        });
        return new Promise((resolve, reject) => {
            Promise.all(promises).then((batches) => {
                let results = [];
                for (const batch of batches) {
                    results = results.concat(batch);
                }
                resolve(results);
            }).catch(reject);
        });
    }
    deleteObjects(s3RegionId, bucket, keys, callback) {
        const semaphore = new semaphore_promise_1.Semaphore(5);
        const keysBatches = [];
        const batchCount = 1000;
        while (keys.length >= batchCount) {
            const batch = keys.splice(0, batchCount);
            keysBatches.push(batch);
        }
        if (keys.length > 0) {
            keysBatches.push(keys);
        }
        let counter = 0;
        const promises = keysBatches.map((batch) => {
            const firstIndexInCurrentBatch = counter;
            counter += batch.length;
            return new Promise((resolve, reject) => {
                const params = new url_1.URLSearchParams();
                for (const key of batch) {
                    params.append('op', `/delete/${encodeObject({ bucket: bucket, key: key })}`);
                }
                semaphore.acquire().then((release) => {
                    this.client.call({
                        method: 'POST',
                        serviceName: kodo_http_client_1.ServiceName.Rs,
                        path: 'batch',
                        dataType: 'json',
                        s3RegionId: s3RegionId,
                        contentType: 'application/x-www-form-urlencoded',
                        data: params.toString(),
                    }).then((response) => {
                        let aborted = false;
                        const results = response.data.map((item, index) => {
                            var _a, _b;
                            const currentIndex = firstIndexInCurrentBatch + index;
                            const result = { bucket: bucket, key: batch[index] };
                            if ((_a = item === null || item === void 0 ? void 0 : item.data) === null || _a === void 0 ? void 0 : _a.error) {
                                const error = new Error((_b = item === null || item === void 0 ? void 0 : item.data) === null || _b === void 0 ? void 0 : _b.error);
                                if (callback && callback(currentIndex, error) === false) {
                                    aborted = true;
                                }
                                result.error = error;
                            }
                            else if (callback && callback(currentIndex) === false) {
                                aborted = true;
                            }
                            return result;
                        });
                        if (aborted) {
                            reject(new Error('aborted'));
                        }
                        else {
                            resolve(results);
                        }
                    }).catch((error) => {
                        let aborted = false;
                        const results = batch.map((key, index) => {
                            const currentIndex = firstIndexInCurrentBatch + index;
                            if (callback && callback(currentIndex, error) === false) {
                                aborted = true;
                            }
                            return { bucket: bucket, key: key, error: error };
                        });
                        if (aborted) {
                            reject(new Error('aborted'));
                        }
                        else {
                            resolve(results);
                        }
                    }).finally(() => {
                        release();
                    });
                });
            });
        });
        return new Promise((resolve, reject) => {
            Promise.all(promises).then((batches) => {
                let results = [];
                for (const batch of batches) {
                    results = results.concat(batch);
                }
                resolve(results);
            }).catch(reject);
        });
    }
    freeze(s3RegionId, object) {
        return new Promise((resolve, reject) => {
            this.client.call({
                method: 'POST',
                serviceName: kodo_http_client_1.ServiceName.Rs,
                path: `chtype/${encodeObject(object)}/type/2`,
                dataType: 'json',
                s3RegionId: s3RegionId,
                contentType: 'application/x-www-form-urlencoded',
            }).then(() => { resolve(); }).catch(reject);
        });
    }
    getFrozenInfo(s3RegionId, object) {
        return new Promise((resolve, reject) => {
            this.client.call({
                method: 'POST',
                serviceName: kodo_http_client_1.ServiceName.Rs,
                path: `stat/${encodeObject(object)}`,
                dataType: 'json',
                s3RegionId: s3RegionId,
                contentType: 'application/x-www-form-urlencoded',
            }).then((response) => {
                if (response.data.type === 2) {
                    if (response.data.restoreStatus) {
                        if (response.data.restoreStatus === 1) {
                            resolve({ status: 'Unfreezing' });
                        }
                        else {
                            resolve({ status: 'Unfrozen' });
                        }
                    }
                    else {
                        resolve({ status: 'Frozen' });
                    }
                }
                else {
                    resolve({ status: 'Normal' });
                }
            }).catch(reject);
        });
    }
    unfreeze(s3RegionId, object, days) {
        return new Promise((resolve, reject) => {
            this.client.call({
                method: 'POST',
                serviceName: kodo_http_client_1.ServiceName.Rs,
                path: `restoreAr/${encodeObject(object)}/freezeAfterDays/${days}`,
                dataType: 'json',
                s3RegionId: s3RegionId,
                contentType: 'application/x-www-form-urlencoded',
            }).then(() => { resolve(); }).catch(reject);
        });
    }
    listObjects(s3RegionId, bucket, prefix, option) {
        return new Promise((resolve, reject) => {
            const results = { objects: [] };
            this._listObjects(s3RegionId, bucket, prefix, resolve, reject, results, option);
        });
    }
    _listObjects(s3RegionId, bucket, prefix, resolve, reject, results, option) {
        const query = new url_1.URLSearchParams();
        query.set('bucket', bucket);
        query.set('prefix', prefix);
        if (option === null || option === void 0 ? void 0 : option.nextContinuationToken) {
            query.set('marker', option.nextContinuationToken);
        }
        if (option === null || option === void 0 ? void 0 : option.maxKeys) {
            query.set('limit', option.maxKeys.toString());
        }
        if (option === null || option === void 0 ? void 0 : option.delimiter) {
            query.set('delimiter', option.delimiter);
        }
        const newOption = {
            delimiter: option === null || option === void 0 ? void 0 : option.delimiter,
        };
        this.client.call({
            method: 'POST',
            serviceName: kodo_http_client_1.ServiceName.Rsf,
            path: 'v2/list',
            s3RegionId: s3RegionId,
            query: query,
            contentType: 'application/x-www-form-urlencoded',
        }).then((response) => {
            let marker = undefined;
            delete results.nextContinuationToken;
            const lines = response.data.toString().split(/\s*\n+\s*/);
            lines.forEach((line) => {
                if (line === '') {
                    return;
                }
                const data = JSON.parse(line);
                if (data.item) {
                    results.objects.push({
                        bucket: bucket, key: data.item.key, size: data.item.fsize,
                        lastModified: new Date(data.item.putTime / 10000), storageClass: toStorageClass(data.item.type),
                    });
                }
                else if (data.dir) {
                    if (results.commonPrefixes === undefined) {
                        results.commonPrefixes = [];
                    }
                    results.commonPrefixes.push({
                        bucket: bucket, key: data.dir,
                    });
                }
                marker = data.marker;
            });
            if (marker) {
                newOption.nextContinuationToken = marker;
                results.nextContinuationToken = marker;
                if (option === null || option === void 0 ? void 0 : option.minKeys) {
                    let resultsSize = results.objects.length;
                    if (results.commonPrefixes) {
                        resultsSize += results.commonPrefixes.length;
                    }
                    if (resultsSize < option.minKeys) {
                        newOption.minKeys = option.minKeys;
                        newOption.maxKeys = option.minKeys - resultsSize;
                        this._listObjects(s3RegionId, bucket, prefix, resolve, reject, results, newOption);
                        return;
                    }
                }
            }
            resolve(results);
        }).catch(reject);
    }
    createMultipartUpload(s3RegionId, object, _originalFileName, _header) {
        return new Promise((resolve, reject) => {
            const token = kodo_auth_1.makeUploadToken(this.adapterOption.accessKey, this.adapterOption.secretKey, kodo_auth_1.newUploadPolicy(object.bucket, object.key));
            const path = `/buckets/${object.bucket}/objects/${urlSafeBase64(object.key)}/uploads`;
            this.client.call({
                method: 'POST',
                serviceName: kodo_http_client_1.ServiceName.Up,
                path: path,
                dataType: 'json',
                s3RegionId: s3RegionId,
                contentType: 'application/x-www-form-urlencoded',
                headers: { 'authorization': `UpToken ${token}` },
            }).then((response) => {
                resolve({ uploadId: response.data.uploadId });
            }).catch(reject);
        });
    }
    uploadPart(s3RegionId, object, uploadId, partNumber, data, option) {
        return new Promise((resolve, reject) => {
            const token = kodo_auth_1.makeUploadToken(this.adapterOption.accessKey, this.adapterOption.secretKey, kodo_auth_1.newUploadPolicy(object.bucket, object.key));
            const path = `/buckets/${object.bucket}/objects/${urlSafeBase64(object.key)}/uploads/${uploadId}/${partNumber}`;
            this.client.call({
                method: 'PUT',
                serviceName: kodo_http_client_1.ServiceName.Up,
                path: path,
                data: data,
                dataType: 'json',
                s3RegionId: s3RegionId,
                contentType: 'application/octet-stream',
                headers: {
                    'authorization': `UpToken ${token}`,
                    'content-md5': js_md5_1.default.hex(data),
                },
                uploadProgress: option === null || option === void 0 ? void 0 : option.progressCallback,
                uploadThrottle: option === null || option === void 0 ? void 0 : option.throttle,
            }).then((response) => {
                resolve({ etag: response.data.etag });
            }).catch(reject);
        });
    }
    completeMultipartUpload(s3RegionId, object, uploadId, parts, originalFileName, header) {
        return new Promise((resolve, reject) => {
            const token = kodo_auth_1.makeUploadToken(this.adapterOption.accessKey, this.adapterOption.secretKey, kodo_auth_1.newUploadPolicy(object.bucket, object.key));
            const path = `/buckets/${object.bucket}/objects/${urlSafeBase64(object.key)}/uploads/${uploadId}`;
            const metadata = {};
            if (header === null || header === void 0 ? void 0 : header.metadata) {
                for (const [metaKey, metaValue] of Object.entries(header.metadata)) {
                    metadata[`x-qn-meta-${metaKey}`] = metaValue;
                }
            }
            const data = { fname: originalFileName, parts: parts, metadata: metadata };
            if (header === null || header === void 0 ? void 0 : header.contentType) {
                data.mimeType = header.contentType;
            }
            this.client.call({
                method: 'POST',
                serviceName: kodo_http_client_1.ServiceName.Up,
                path: path,
                data: JSON.stringify(data),
                dataType: 'json',
                s3RegionId: s3RegionId,
                headers: { 'authorization': `UpToken ${token}` },
            }).then(() => { resolve(); }).catch(reject);
        });
    }
    clearCache() {
        Object.keys(this.bucketDomainsCache).forEach((key) => { delete this.bucketDomainsCache[key]; });
        this.client.clearCache();
        this.regionService.clearCache();
    }
}
exports.Kodo = Kodo;
Kodo.httpClient = new urllib_1.HttpClient2();
function toStorageClass(type) {
    switch (type !== null && type !== void 0 ? type : 0) {
        case 0:
            return 'Standard';
        case 1:
            return 'InfrequentAccess';
        case 2:
            return 'Glacier';
        default:
            throw new Error(`Unknown file type: ${type}`);
    }
}
function encodeObject(object) {
    return encodeBucketKey(object.bucket, object.key);
}
function encodeBucketKey(bucket, key) {
    let data = bucket;
    if (key) {
        data += `:${key}`;
    }
    return urlSafeBase64(data);
}
function urlSafeBase64(data) {
    return kodo_auth_1.base64ToUrlSafe(js_base64_1.encode(data));
}
function getObjectHeader(response) {
    const size = parseInt(response.headers['content-length']);
    const contentType = response.headers['content-type'];
    const lastModified = new Date(response.headers['last-modified']);
    const metadata = {};
    for (const [metaKey, metaValue] of Object.entries(response.headers)) {
        if (metaKey === null || metaKey === void 0 ? void 0 : metaKey.startsWith('x-qn-meta-')) {
            metadata[metaKey.substring('x-qn-meta-'.length)] = metaValue;
        }
    }
    return { size: size, contentType: contentType, lastModified: lastModified, metadata: metadata };
}
//# sourceMappingURL=kodo.js.map