"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.Kodo = exports.USER_AGENT = void 0;
const async_lock_1 = __importDefault(require("async-lock"));
const os_1 = __importDefault(require("os"));
const package_json_1 = __importDefault(require("./package.json"));
const form_data_1 = __importDefault(require("form-data"));
const buffer_crc32_1 = __importDefault(require("buffer-crc32"));
const js_md5_1 = __importDefault(require("js-md5"));
const semaphore_promise_1 = require("semaphore-promise");
const region_service_1 = require("./region_service");
const url_1 = require("url");
const js_base64_1 = require("js-base64");
const kodo_auth_1 = require("./kodo-auth");
const kodo_http_client_1 = require("./kodo-http-client");
const uplog_1 = require("./uplog");
exports.USER_AGENT = `Qiniu-Kodo-S3-Adapter-NodeJS-SDK/${package_json_1.default.version} (${os_1.default.type()}; ${os_1.default.platform()}; ${os_1.default.arch()}; )/kodo`;
class Kodo {
    constructor(adapterOption) {
        this.adapterOption = adapterOption;
        this.bucketDomainsCache = {};
        this.bucketDomainsCacheLock = new async_lock_1.default();
        let userAgent = exports.USER_AGENT;
        if (adapterOption.appendedUserAgent) {
            userAgent += `/${adapterOption.appendedUserAgent}`;
        }
        this.client = new kodo_http_client_1.KodoHttpClient({
            accessKey: adapterOption.accessKey,
            secretKey: adapterOption.secretKey,
            ucUrl: adapterOption.ucUrl,
            regions: adapterOption.regions,
            appendedUserAgent: adapterOption.appendedUserAgent,
            appName: adapterOption.appName,
            appVersion: adapterOption.appVersion,
            uplogBufferSize: adapterOption.uplogBufferSize,
            userAgent: userAgent,
            timeout: [30000, 300000],
            retry: 10,
            retryDelay: 500,
            requestCallback: adapterOption.requestCallback,
            responseCallback: adapterOption.responseCallback,
        });
        this.regionService = new region_service_1.RegionService(adapterOption);
    }
    enter(sdkApiName, f) {
        const scope = new KodoScope(sdkApiName, this.adapterOption);
        return new Promise((resolve, reject) => {
            f(scope, scope.getRegionRequestOptions()).then((data) => {
                scope.done(true).finally(() => { resolve(data); });
            }).catch((err) => {
                scope.done(false).finally(() => { reject(err); });
            });
        });
    }
    createBucket(s3RegionId, bucket) {
        return new Promise((resolve, reject) => {
            this.regionService.fromS3IdToKodoRegionId(s3RegionId, this.getRegionRequestOptions()).then((kodoRegionId) => {
                this.call({
                    method: 'POST',
                    serviceName: kodo_http_client_1.ServiceName.Uc,
                    path: `mkbucketv3/${bucket}/region/${kodoRegionId}`,
                }).then(() => {
                    resolve();
                }).catch(reject);
            }).catch(reject);
        });
    }
    deleteBucket(_region, bucket) {
        return new Promise((resolve, reject) => {
            this.call({
                method: 'POST',
                serviceName: kodo_http_client_1.ServiceName.Uc,
                bucketName: bucket,
                path: `drop/${bucket}`,
            }).then(() => {
                resolve();
            }).catch(reject);
        });
    }
    getBucketLocation(bucket) {
        return new Promise((resolve, reject) => {
            this.call({
                method: 'GET',
                serviceName: kodo_http_client_1.ServiceName.Uc,
                bucketName: bucket,
                path: `bucket/${bucket}`,
                dataType: 'json',
            }).then((response) => {
                const kodoRegionId = response.data.region;
                this.regionService.fromKodoRegionIdToS3Id(kodoRegionId, this.getRegionRequestOptions())
                    .then(resolve).catch(reject);
            }).catch(reject);
        });
    }
    listBuckets() {
        return new Promise((resolve, reject) => {
            const bucketsQuery = new url_1.URLSearchParams();
            bucketsQuery.set('shared', 'rd');
            this.call({
                method: 'GET',
                serviceName: kodo_http_client_1.ServiceName.Uc,
                path: 'v2/buckets',
                dataType: 'json',
                query: bucketsQuery,
            }).then((response) => {
                if (!response.data) {
                    resolve([]);
                    return;
                }
                const regionsPromises = response.data.map((info) => {
                    return new Promise((resolve) => {
                        this.regionService.fromKodoRegionIdToS3Id(info.region, this.getRegionRequestOptions())
                            .then(resolve).catch(() => { resolve(undefined); });
                    });
                });
                Promise.all(regionsPromises).then((regionsInfo) => {
                    const bucketInfos = response.data.map((info, index) => {
                        let grantedPermission = undefined;
                        switch (info.perm) {
                            case 1:
                                grantedPermission = 'readonly';
                                break;
                            case 2:
                                grantedPermission = 'readwrite';
                                break;
                        }
                        return {
                            id: info.id, name: info.tbl,
                            createDate: new Date(info.ctime * 1000),
                            regionId: regionsInfo[index], grantedPermission: grantedPermission,
                        };
                    });
                    resolve(bucketInfos);
                }).catch(reject);
            }).catch(reject);
        });
    }
    listDomains(s3RegionId, bucket) {
        return new Promise((resolve, reject) => {
            const domainsQuery = new url_1.URLSearchParams();
            domainsQuery.set('sourceTypes', 'qiniuBucket');
            domainsQuery.set('sourceQiniuBucket', bucket);
            domainsQuery.set('operatingState', 'success');
            domainsQuery.set('limit', '50');
            const getBucketInfoQuery = new url_1.URLSearchParams();
            getBucketInfoQuery.set('bucket', bucket);
            const bucketDefaultDomainQuery = new url_1.URLSearchParams();
            bucketDefaultDomainQuery.set('bucket', bucket);
            const promises = [
                this.call({
                    method: 'GET',
                    serviceName: kodo_http_client_1.ServiceName.Qcdn,
                    path: 'domain',
                    query: domainsQuery,
                    dataType: 'json',
                    s3RegionId: s3RegionId,
                }),
                this.call({
                    method: 'POST',
                    serviceName: kodo_http_client_1.ServiceName.Uc,
                    path: 'v2/bucketInfo',
                    query: getBucketInfoQuery,
                    dataType: 'json',
                    s3RegionId: s3RegionId,
                }),
                this.call({
                    method: 'GET',
                    serviceName: kodo_http_client_1.ServiceName.Portal,
                    path: 'api/kodov2/domain/default/get',
                    query: bucketDefaultDomainQuery,
                    dataType: 'json',
                    s3RegionId: s3RegionId,
                }),
            ];
            Promise.all(promises).then(([domainResponse, bucketResponse, defaultDomainQuery]) => {
                if (bucketResponse.data.perm && bucketResponse.data.perm > 0) {
                    const result = defaultDomainQuery.data;
                    const domains = [];
                    if (result.domain && result.protocol) {
                        domains.push({
                            name: result.domain, protocol: result.protocol,
                            type: 'normal', private: bucketResponse.data.private != 0,
                        });
                    }
                    resolve(domains);
                }
                else {
                    const domains = domainResponse.data.domains.filter((domain) => {
                        switch (domain.type) {
                            case 'normal':
                            case 'pan':
                            case 'test':
                                return true;
                            default:
                                return false;
                        }
                    }).map((domain) => {
                        return {
                            name: domain.name, protocol: domain.protocol, type: domain.type,
                            private: bucketResponse.data.private != 0,
                        };
                    });
                    resolve(domains);
                }
            }).catch(reject);
        });
    }
    _listDomains(s3RegionId, bucket) {
        return new Promise((resolve, reject) => {
            if (this.bucketDomainsCache[bucket]) {
                resolve(this.bucketDomainsCache[bucket]);
                return;
            }
            this.bucketDomainsCacheLock.acquire(bucket, () => {
                if (this.bucketDomainsCache[bucket]) {
                    return Promise.resolve(this.bucketDomainsCache[bucket]);
                }
                return this.listDomains(s3RegionId, bucket);
            }).then((domains) => {
                this.bucketDomainsCache[bucket] = domains;
                resolve(domains);
            }).catch(reject);
        });
    }
    listBucketIdNames() {
        return new Promise((resolve, reject) => {
            this.call({
                method: 'GET',
                serviceName: kodo_http_client_1.ServiceName.Uc,
                path: 'v2/buckets',
                dataType: 'json',
            }).then((response) => {
                const bucketInfos = response.data.map((info) => {
                    return { id: info.id, name: info.tbl };
                });
                resolve(bucketInfos);
            }).catch(reject);
        });
    }
    isExists(s3RegionId, object) {
        return new Promise((resolve, reject) => {
            this.getObjectInfo(s3RegionId, object).then(() => {
                resolve(true);
            }).catch((error) => {
                if (error.message === 'no such file or directory') {
                    resolve(false);
                }
                else {
                    reject(error);
                }
            });
        });
    }
    deleteObject(s3RegionId, object) {
        return new Promise((resolve, reject) => {
            this.call({
                method: 'POST',
                serviceName: kodo_http_client_1.ServiceName.Rs,
                path: `delete/${encodeObject(object)}`,
                dataType: 'json',
                s3RegionId: s3RegionId,
                contentType: 'application/x-www-form-urlencoded',
            }).then(() => { resolve(); }).catch(reject);
        });
    }
    putObject(s3RegionId, object, data, originalFileName, header, option) {
        return new Promise((resolve, reject) => {
            const token = kodo_auth_1.makeUploadToken(this.adapterOption.accessKey, this.adapterOption.secretKey, kodo_auth_1.newUploadPolicy(object.bucket, object.key));
            const form = new form_data_1.default();
            form.append('key', object.key);
            form.append('token', token);
            if (header === null || header === void 0 ? void 0 : header.metadata) {
                for (const [metaKey, metaValue] of Object.entries(header.metadata)) {
                    form.append(`x-qn-meta-${metaKey}`, metaValue);
                }
            }
            form.append('crc32', buffer_crc32_1.default.unsigned(data));
            const fileOption = {
                filename: originalFileName,
            };
            if (header === null || header === void 0 ? void 0 : header.contentType) {
                fileOption.contentType = header.contentType;
            }
            form.append('file', data, fileOption);
            this.call({
                method: 'POST',
                serviceName: kodo_http_client_1.ServiceName.Up,
                dataType: 'json',
                s3RegionId: s3RegionId,
                contentType: form.getHeaders()['content-type'],
                form: form,
                uploadProgress: option === null || option === void 0 ? void 0 : option.progressCallback,
                uploadThrottle: option === null || option === void 0 ? void 0 : option.throttle,
            }).then(() => { resolve(); }).catch(reject);
        });
    }
    getObject(s3RegionId, object, domain) {
        return new Promise((resolve, reject) => {
            this.getObjectURL(s3RegionId, object, domain).then((url) => {
                this.callUrl([url.toString()], {
                    fullUrl: true,
                    appendAuthorization: false,
                    method: 'GET',
                }).then((response) => {
                    resolve({ data: response.data, header: getObjectHeader(response) });
                }).catch(reject);
            }).catch(reject);
        });
    }
    getObjectStream(s3RegionId, object, domain, option) {
        var _a, _b;
        const headers = {};
        if ((option === null || option === void 0 ? void 0 : option.rangeStart) || (option === null || option === void 0 ? void 0 : option.rangeEnd)) {
            headers['Range'] = `bytes=${(_a = option === null || option === void 0 ? void 0 : option.rangeStart) !== null && _a !== void 0 ? _a : ''}-${(_b = option === null || option === void 0 ? void 0 : option.rangeEnd) !== null && _b !== void 0 ? _b : ''}`;
        }
        return new Promise((resolve, reject) => {
            this.getObjectURL(s3RegionId, object, domain).then((url) => {
                this.callUrl([url.toString()], {
                    fullUrl: true,
                    appendAuthorization: false,
                    method: 'GET',
                    headers: headers,
                    streaming: true,
                }).then((response) => {
                    resolve(response.res);
                }).catch(reject);
            }).catch(reject);
        });
    }
    getObjectURL(s3RegionId, object, domain, deadline) {
        return new Promise((resolve, reject) => {
            const domainPromise = new Promise((resolve, reject) => {
                if (domain) {
                    resolve(domain);
                    return;
                }
                this._listDomains(s3RegionId, object.bucket).then((domains) => {
                    if (domains.length === 0) {
                        reject(new Error('no domain found'));
                        return;
                    }
                    const domainTypeScope = (domain) => {
                        switch (domain.type) {
                            case 'normal': return 1;
                            case 'pan': return 2;
                            case 'test': return 3;
                        }
                    };
                    domains = domains.sort((domain1, domain2) => domainTypeScope(domain1) - domainTypeScope(domain2));
                    resolve(domains[0]);
                }).catch(reject);
            });
            domainPromise.then((domain) => {
                let url = new url_1.URL(`${domain.protocol}://${domain.name}`);
                url.pathname = object.key;
                if (domain.private) {
                    url = kodo_auth_1.signPrivateURL(this.adapterOption.accessKey, this.adapterOption.secretKey, url, deadline);
                }
                resolve(url);
            }).catch(reject);
        });
    }
    getObjectInfo(s3RegionId, object) {
        return new Promise((resolve, reject) => {
            this.call({
                method: 'GET',
                serviceName: kodo_http_client_1.ServiceName.Rs,
                path: `stat/${encodeObject(object)}`,
                dataType: 'json',
                s3RegionId: s3RegionId,
                contentType: 'application/x-www-form-urlencoded',
            }).then((response) => {
                resolve({
                    bucket: object.bucket, key: response.data.key, size: response.data.fsize,
                    lastModified: new Date(response.data.putTime / 10000), storageClass: toStorageClass(response.data.type),
                });
            }).catch(reject);
        });
    }
    getObjectHeader(s3RegionId, object, domain) {
        return new Promise((resolve, reject) => {
            this.getObjectURL(s3RegionId, object, domain).then((url) => {
                this.callUrl([url.toString()], {
                    fullUrl: true,
                    appendAuthorization: false,
                    method: 'HEAD',
                }).then((response) => {
                    resolve(getObjectHeader(response));
                }).catch(reject);
            }).catch(reject);
        });
    }
    moveObject(s3RegionId, transferObject) {
        return new Promise((resolve, reject) => {
            this.call({
                method: 'POST',
                serviceName: kodo_http_client_1.ServiceName.Rs,
                path: `move/${encodeObject(transferObject.from)}/${encodeObject(transferObject.to)}/force/true`,
                dataType: 'json',
                s3RegionId: s3RegionId,
                contentType: 'application/x-www-form-urlencoded',
            }).then(() => { resolve(); }).catch(reject);
        });
    }
    copyObject(s3RegionId, transferObject) {
        return new Promise((resolve, reject) => {
            this.call({
                method: 'POST',
                serviceName: kodo_http_client_1.ServiceName.Rs,
                path: `copy/${encodeObject(transferObject.from)}/${encodeObject(transferObject.to)}/force/true`,
                dataType: 'json',
                s3RegionId: s3RegionId,
                contentType: 'application/x-www-form-urlencoded',
            }).then(() => { resolve(); }).catch(reject);
        });
    }
    moveObjects(s3RegionId, transferObjects, callback) {
        return this.batchOps(transferObjects.map((to) => new MoveObjectOp(to)), 100, s3RegionId, callback);
    }
    copyObjects(s3RegionId, transferObjects, callback) {
        return this.batchOps(transferObjects.map((to) => new CopyObjectOp(to)), 100, s3RegionId, callback);
    }
    deleteObjects(s3RegionId, bucket, keys, callback) {
        return this.batchOps(keys.map((key) => new DeleteObjectOp({ bucket, key })), 100, s3RegionId, callback);
    }
    setObjectsStorageClass(s3RegionId, bucket, keys, storageClass, callback) {
        return this.batchOps(keys.map((key) => new SetObjectStorageClassOp({ bucket, key }, storageClass)), 100, s3RegionId, callback);
    }
    restoreObjects(s3RegionId, bucket, keys, days, callback) {
        return this.batchOps(keys.map((key) => new RestoreObjectsOp({ bucket, key }, days)), 100, s3RegionId, callback);
    }
    batchOps(ops, batchCount, s3RegionId, callback) {
        const semaphore = new semaphore_promise_1.Semaphore(20);
        const opsBatches = [];
        while (ops.length >= batchCount) {
            const batch = ops.splice(0, batchCount);
            opsBatches.push(batch);
        }
        if (ops.length > 0) {
            opsBatches.push(ops);
        }
        let counter = 0;
        const promises = opsBatches.map((batch) => {
            const firstIndexInCurrentBatch = counter;
            counter += batch.length;
            return new Promise((resolve, reject) => {
                const params = new url_1.URLSearchParams();
                for (const op of batch) {
                    params.append('op', op.getOp());
                }
                semaphore.acquire().then((release) => {
                    this.call({
                        method: 'POST',
                        serviceName: kodo_http_client_1.ServiceName.Rs,
                        path: 'batch',
                        dataType: 'json',
                        s3RegionId: s3RegionId,
                        contentType: 'application/x-www-form-urlencoded',
                        data: params.toString(),
                    }).then((response) => {
                        let aborted = false;
                        const results = response.data.map((item, index) => {
                            var _a, _b;
                            const currentIndex = firstIndexInCurrentBatch + index;
                            const result = batch[index].getObject();
                            if ((_a = item === null || item === void 0 ? void 0 : item.data) === null || _a === void 0 ? void 0 : _a.error) {
                                const error = new Error((_b = item === null || item === void 0 ? void 0 : item.data) === null || _b === void 0 ? void 0 : _b.error);
                                if (callback && callback(currentIndex, error) === false) {
                                    aborted = true;
                                }
                                result.error = error;
                            }
                            else if (callback && callback(currentIndex) === false) {
                                aborted = true;
                            }
                            return result;
                        });
                        if (aborted) {
                            reject(new Error('aborted'));
                        }
                        else {
                            resolve(results);
                        }
                    }).catch((error) => {
                        let aborted = false;
                        const results = batch.map((op, index) => {
                            const currentIndex = firstIndexInCurrentBatch + index;
                            if (callback && callback(currentIndex, error) === false) {
                                aborted = true;
                            }
                            return Object.assign({}, op.getObject());
                        });
                        if (aborted) {
                            reject(new Error('aborted'));
                        }
                        else {
                            resolve(results);
                        }
                    }).finally(() => {
                        release();
                    });
                });
            });
        });
        return new Promise((resolve, reject) => {
            Promise.all(promises).then((batches) => {
                let results = [];
                for (const batch of batches) {
                    results = results.concat(batch);
                }
                resolve(results);
            }).catch(reject);
        });
    }
    getFrozenInfo(s3RegionId, object) {
        return new Promise((resolve, reject) => {
            this.call({
                method: 'POST',
                serviceName: kodo_http_client_1.ServiceName.Rs,
                path: `stat/${encodeObject(object)}`,
                dataType: 'json',
                s3RegionId: s3RegionId,
                contentType: 'application/x-www-form-urlencoded',
            }).then((response) => {
                if (response.data.type === 2) {
                    if (response.data.restoreStatus) {
                        if (response.data.restoreStatus === 1) {
                            resolve({ status: 'Unfreezing' });
                        }
                        else {
                            resolve({ status: 'Unfrozen' });
                        }
                    }
                    else {
                        resolve({ status: 'Frozen' });
                    }
                }
                else {
                    resolve({ status: 'Normal' });
                }
            }).catch(reject);
        });
    }
    restoreObject(s3RegionId, object, days) {
        return new Promise((resolve, reject) => {
            this.call({
                method: 'POST',
                serviceName: kodo_http_client_1.ServiceName.Rs,
                path: `restoreAr/${encodeObject(object)}/freezeAfterDays/${days}`,
                dataType: 'json',
                s3RegionId: s3RegionId,
                contentType: 'application/x-www-form-urlencoded',
            }).then(() => { resolve(); }).catch(reject);
        });
    }
    setObjectStorageClass(s3RegionId, object, storageClass) {
        return new Promise((resolve, reject) => {
            this.call({
                method: 'POST',
                serviceName: kodo_http_client_1.ServiceName.Rs,
                path: `chtype/${encodeObject(object)}/type/${convertStorageClassToFileType(storageClass)}`,
                dataType: 'json',
                s3RegionId: s3RegionId,
                contentType: 'application/x-www-form-urlencoded',
            }).then(() => { resolve(); }).catch(reject);
        });
    }
    listObjects(s3RegionId, bucket, prefix, option) {
        return new Promise((resolve, reject) => {
            const results = { objects: [] };
            this._listObjects(s3RegionId, bucket, prefix, resolve, reject, results, option);
        });
    }
    _listObjects(s3RegionId, bucket, prefix, resolve, reject, results, option) {
        const query = new url_1.URLSearchParams();
        query.set('bucket', bucket);
        query.set('prefix', prefix);
        if (option === null || option === void 0 ? void 0 : option.nextContinuationToken) {
            query.set('marker', option.nextContinuationToken);
        }
        if (option === null || option === void 0 ? void 0 : option.maxKeys) {
            query.set('limit', option.maxKeys.toString());
        }
        if (option === null || option === void 0 ? void 0 : option.delimiter) {
            query.set('delimiter', option.delimiter);
        }
        const newOption = {
            delimiter: option === null || option === void 0 ? void 0 : option.delimiter,
        };
        this.call({
            method: 'POST',
            serviceName: kodo_http_client_1.ServiceName.Rsf,
            path: 'v2/list',
            s3RegionId: s3RegionId,
            query: query,
            dataType: 'multijson',
            contentType: 'application/x-www-form-urlencoded',
        }).then((response) => {
            let marker = undefined;
            delete results.nextContinuationToken;
            response.data.forEach((data) => {
                if (data.item) {
                    results.objects.push({
                        bucket: bucket, key: data.item.key, size: data.item.fsize,
                        lastModified: new Date(data.item.putTime / 10000), storageClass: toStorageClass(data.item.type),
                    });
                }
                else if (data.dir) {
                    if (results.commonPrefixes === undefined) {
                        results.commonPrefixes = [];
                    }
                    let foundDup = false;
                    for (const commonPrefix of results.commonPrefixes) {
                        if (commonPrefix.key === data.dir) {
                            foundDup = true;
                            break;
                        }
                    }
                    if (!foundDup) {
                        results.commonPrefixes.push({
                            bucket: bucket, key: data.dir,
                        });
                    }
                }
                marker = data.marker;
            });
            if (marker) {
                newOption.nextContinuationToken = marker;
                results.nextContinuationToken = marker;
                if (option === null || option === void 0 ? void 0 : option.minKeys) {
                    let resultsSize = results.objects.length;
                    if (results.commonPrefixes) {
                        resultsSize += results.commonPrefixes.length;
                    }
                    if (resultsSize < option.minKeys) {
                        newOption.minKeys = option.minKeys;
                        newOption.maxKeys = option.minKeys - resultsSize;
                        this._listObjects(s3RegionId, bucket, prefix, resolve, reject, results, newOption);
                        return;
                    }
                }
            }
            resolve(results);
        }).catch(reject);
    }
    createMultipartUpload(s3RegionId, object, _originalFileName, _header) {
        return new Promise((resolve, reject) => {
            const token = kodo_auth_1.makeUploadToken(this.adapterOption.accessKey, this.adapterOption.secretKey, kodo_auth_1.newUploadPolicy(object.bucket, object.key));
            const path = `/buckets/${object.bucket}/objects/${urlSafeBase64(object.key)}/uploads`;
            this.call({
                method: 'POST',
                serviceName: kodo_http_client_1.ServiceName.Up,
                path: path,
                dataType: 'json',
                s3RegionId: s3RegionId,
                contentType: 'application/x-www-form-urlencoded',
                headers: { 'authorization': `UpToken ${token}` },
            }).then((response) => {
                resolve({ uploadId: response.data.uploadId });
            }).catch(reject);
        });
    }
    uploadPart(s3RegionId, object, uploadId, partNumber, data, option) {
        return new Promise((resolve, reject) => {
            const token = kodo_auth_1.makeUploadToken(this.adapterOption.accessKey, this.adapterOption.secretKey, kodo_auth_1.newUploadPolicy(object.bucket, object.key));
            const path = `/buckets/${object.bucket}/objects/${urlSafeBase64(object.key)}/uploads/${uploadId}/${partNumber}`;
            this.call({
                method: 'PUT',
                serviceName: kodo_http_client_1.ServiceName.Up,
                path: path,
                data: data,
                dataType: 'json',
                s3RegionId: s3RegionId,
                contentType: 'application/octet-stream',
                headers: {
                    'authorization': `UpToken ${token}`,
                    'content-md5': js_md5_1.default.hex(data),
                },
                uploadProgress: option === null || option === void 0 ? void 0 : option.progressCallback,
                uploadThrottle: option === null || option === void 0 ? void 0 : option.throttle,
            }).then((response) => {
                resolve({ etag: response.data.etag });
            }).catch(reject);
        });
    }
    completeMultipartUpload(s3RegionId, object, uploadId, parts, originalFileName, header) {
        return new Promise((resolve, reject) => {
            const token = kodo_auth_1.makeUploadToken(this.adapterOption.accessKey, this.adapterOption.secretKey, kodo_auth_1.newUploadPolicy(object.bucket, object.key));
            const path = `/buckets/${object.bucket}/objects/${urlSafeBase64(object.key)}/uploads/${uploadId}`;
            const metadata = {};
            if (header === null || header === void 0 ? void 0 : header.metadata) {
                for (const [metaKey, metaValue] of Object.entries(header.metadata)) {
                    metadata[`x-qn-meta-${metaKey}`] = metaValue;
                }
            }
            const data = { fname: originalFileName, parts: parts, metadata: metadata };
            if (header === null || header === void 0 ? void 0 : header.contentType) {
                data.mimeType = header.contentType;
            }
            this.call({
                method: 'POST',
                serviceName: kodo_http_client_1.ServiceName.Up,
                path: path,
                data: JSON.stringify(data),
                dataType: 'json',
                s3RegionId: s3RegionId,
                headers: { 'authorization': `UpToken ${token}` },
            }).then(() => { resolve(); }).catch(reject);
        });
    }
    clearCache() {
        Object.keys(this.bucketDomainsCache).forEach((key) => { delete this.bucketDomainsCache[key]; });
        this.client.clearCache();
        this.regionService.clearCache();
    }
    call(options) {
        return this.client.call(options);
    }
    callUrl(urls, options) {
        return this.client.callUrls(urls, options);
    }
    getRegionRequestOptions() {
        return {
            timeout: [30000, 300000],
            retry: 10,
            retryDelay: 500,
        };
    }
    log(entry) {
        return this.client.log(entry);
    }
}
exports.Kodo = Kodo;
class KodoScope extends Kodo {
    constructor(sdkApiName, adapterOption) {
        super(adapterOption);
        this.beginTime = new Date();
        this.requestStats = {
            sdkApiName: sdkApiName,
            requestsCount: 0,
        };
    }
    done(successful) {
        const uplog = {
            log_type: uplog_1.LogType.SdkApi,
            api_name: this.requestStats.sdkApiName,
            requests_count: this.requestStats.requestsCount,
            total_elapsed_time: new Date().getTime() - this.beginTime.getTime(),
        };
        if (!successful) {
            if (this.requestStats.errorType) {
                uplog.error_type = this.requestStats.errorType;
            }
            if (this.requestStats.errorDescription) {
                uplog.error_description = this.requestStats.errorDescription;
            }
        }
        this.requestStats.requestsCount = 0;
        this.requestStats.errorType = undefined;
        this.requestStats.errorDescription = undefined;
        return this.log(uplog);
    }
    call(options) {
        options.stats = this.requestStats;
        return super.call(options);
    }
    getRegionRequestOptions() {
        const options = super.getRegionRequestOptions();
        options.stats = this.requestStats;
        return options;
    }
}
function toStorageClass(type) {
    switch (type !== null && type !== void 0 ? type : 0) {
        case 0:
            return 'Standard';
        case 1:
            return 'InfrequentAccess';
        case 2:
            return 'Glacier';
        default:
            throw new Error(`Unknown file type: ${type}`);
    }
}
function encodeObject(object) {
    return encodeBucketKey(object.bucket, object.key);
}
function encodeBucketKey(bucket, key) {
    let data = bucket;
    if (key) {
        data += `:${key}`;
    }
    return urlSafeBase64(data);
}
function urlSafeBase64(data) {
    return kodo_auth_1.base64ToUrlSafe(js_base64_1.encode(data));
}
function getObjectHeader(response) {
    const size = parseInt(response.headers['content-length']);
    const contentType = response.headers['content-type'];
    const lastModified = new Date(response.headers['last-modified']);
    const metadata = {};
    for (const [metaKey, metaValue] of Object.entries(response.headers)) {
        if (metaKey === null || metaKey === void 0 ? void 0 : metaKey.startsWith('x-qn-meta-')) {
            metadata[metaKey.substring('x-qn-meta-'.length)] = metaValue;
        }
    }
    return { size: size, contentType: contentType, lastModified: lastModified, metadata: metadata };
}
class ObjectOp {
}
class MoveObjectOp extends ObjectOp {
    constructor(object) {
        super();
        this.object = object;
    }
    getObject() {
        return this.object.from;
    }
    getOp() {
        return `/move/${encodeObject(this.object.from)}/${encodeObject(this.object.to)}/force/true`;
    }
}
class CopyObjectOp extends ObjectOp {
    constructor(object) {
        super();
        this.object = object;
    }
    getObject() {
        return this.object.from;
    }
    getOp() {
        return `/copy/${encodeObject(this.object.from)}/${encodeObject(this.object.to)}/force/true`;
    }
}
class DeleteObjectOp extends ObjectOp {
    constructor(object) {
        super();
        this.object = object;
    }
    getObject() {
        return this.object;
    }
    getOp() {
        return `/delete/${encodeObject(this.object)}`;
    }
}
class SetObjectStorageClassOp extends ObjectOp {
    constructor(object, storageClass) {
        super();
        this.object = object;
        this.storageClass = storageClass;
    }
    getObject() {
        return this.object;
    }
    getOp() {
        return `chtype/${encodeObject(this.object)}/type/${convertStorageClassToFileType(this.storageClass)}`;
    }
}
function convertStorageClassToFileType(storageClass) {
    let fileType = 0;
    switch (storageClass) {
        case 'Standard':
            fileType = 0;
            break;
        case 'InfrequentAccess':
            fileType = 1;
            break;
        case 'Glacier':
            fileType = 2;
            break;
    }
    return fileType;
}
class RestoreObjectsOp extends ObjectOp {
    constructor(object, days) {
        super();
        this.object = object;
        this.days = days;
    }
    getObject() {
        return this.object;
    }
    getOp() {
        return `restoreAr/${encodeObject(this.object)}/freezeAfterDays/${this.days}`;
    }
}
//# sourceMappingURL=kodo.js.map