//
// ========================================================================
// Copyright (c) 1995-2020 Mort Bay Consulting Pty Ltd and others.
//
// This program and the accompanying materials are made available under
// the terms of the Eclipse Public License 2.0 which is available at
// https://www.eclipse.org/legal/epl-2.0
//
// This Source Code may also be made available under the following
// Secondary Licenses when the conditions for such availability set
// forth in the Eclipse Public License, v. 2.0 are satisfied:
// the Apache License v2.0 which is available at
// https://www.apache.org/licenses/LICENSE-2.0
//
// SPDX-License-Identifier: EPL-2.0 OR Apache-2.0
// ========================================================================
//

package org.eclipse.jetty.websocket.core;

import java.io.IOException;
import java.net.URI;
import java.util.List;

import org.eclipse.jetty.server.NetworkConnector;
import org.eclipse.jetty.server.Server;
import org.eclipse.jetty.server.ServerConnector;
import org.eclipse.jetty.server.handler.ContextHandler;
import org.eclipse.jetty.websocket.core.server.Negotiation;
import org.eclipse.jetty.websocket.core.server.WebSocketNegotiator;
import org.eclipse.jetty.websocket.core.server.WebSocketUpgradeHandler;

public class WebSocketServer
{
    private final Server server;
    private URI serverUri;

    public void start() throws Exception
    {
        server.start();
        serverUri = new URI("ws://localhost:" + getLocalPort());
    }

    public void stop() throws Exception
    {
        server.stop();
    }

    public int getLocalPort()
    {
        return server.getBean(NetworkConnector.class).getLocalPort();
    }

    public Server getServer()
    {
        return server;
    }

    public WebSocketServer(FrameHandler frameHandler)
    {
        this(new DefaultNegotiator(frameHandler));
    }

    public WebSocketServer(WebSocketNegotiator negotiator)
    {
        server = new Server();
        ServerConnector connector = new ServerConnector(server);
        server.addConnector(connector);

        ContextHandler context = new ContextHandler("/");
        server.setHandler(context);

        WebSocketUpgradeHandler upgradeHandler = new WebSocketUpgradeHandler(negotiator);
        context.setHandler(upgradeHandler);
    }

    public URI getUri()
    {
        return serverUri;
    }

    private static class DefaultNegotiator extends WebSocketNegotiator.AbstractNegotiator
    {
        private final FrameHandler frameHandler;

        public DefaultNegotiator(FrameHandler frameHandler)
        {
            this.frameHandler = frameHandler;
        }

        @Override
        public FrameHandler negotiate(Negotiation negotiation) throws IOException
        {
            List<String> offeredSubprotocols = negotiation.getOfferedSubprotocols();
            if (!offeredSubprotocols.isEmpty())
                negotiation.setSubprotocol(offeredSubprotocols.get(0));

            return frameHandler;
        }
    }
}
